'use strict';

import { objToArr, reviverOrAsIs } from './U';
import AbstractMap from './AbstractMap';
import AsIs from './AsIs';
import Any from './Any';

const stringifyMapper = pair => ({key: pair[0], value: pair[1]});

const parseMapper = (keyMetadata, valueMetadata) => { return pairObject => {
  const reviveKey = reviverOrAsIs(keyMetadata);
  const key = reviveKey('', pairObject.key);

  const reviveVal = reviverOrAsIs(valueMetadata);
  const val = reviveVal('', pairObject.value);

  return [key, val];
}};

const reviverFactory = (keyMetadata, valueMetadata) => { return (k, v) => {
  if (k !== '') {
    return v;
  }

  const innerMap = (v === null) ?
    null :
    new Map(v.map(parseMapper(keyMetadata, valueMetadata)));

  return new ModelicoMap(innerMap);
}};

class ModelicoMap extends AbstractMap {
  constructor(innerMap) {
    super(ModelicoMap, innerMap);

    Object.freeze(this);
  }

  set(key, value) {
    return AbstractMap.set.call(this, ModelicoMap, key, value);
  }

  toJSON() {
    return [...this.inner()].map(stringifyMapper);
  }

  static fromMap(map) {
    return new ModelicoMap(map);
  }

  static fromArray(pairs) {
    return ModelicoMap.fromMap(new Map(pairs));
  }

  static of(...arr) {
    const len = arr.length;

    if (len % 2 === 1) {
      throw TypeError('Map.of requires an even number of arguments');
    }

    const pairs = [];

    for (let i = 0; i < len; i += 2) {
      pairs.push([arr[i], arr[i + 1]]);
    }

    return ModelicoMap.fromArray(pairs);
  }

  static fromObject(obj) {
    return ModelicoMap.fromArray(objToArr(obj));
  }

  static metadata(keyMetadata, valueMetadata) {
    return AbstractMap.metadata(ModelicoMap, reviverFactory(keyMetadata, valueMetadata));
  }
}

ModelicoMap.EMPTY = ModelicoMap.fromArray([]);

export default Object.freeze(ModelicoMap);
