/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
var __extends = (this && this.__extends) || (function () {
    var extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
import * as nls from '../../../nls.js';
import { Emitter } from '../../../base/common/event.js';
import { Disposable } from '../../../base/common/lifecycle.js';
import * as objects from '../../../base/common/objects.js';
import * as platform from '../../../base/common/platform.js';
import { Extensions, ConfigurationScope } from '../../../platform/configuration/common/configurationRegistry.js';
import { Registry } from '../../../platform/registry/common/platform.js';
import { BareFontInfo } from './fontInfo.js';
import { EditorZoom } from './editorZoom.js';
import * as editorOptions from './editorOptions.js';
var EDITOR_DEFAULTS = editorOptions.EDITOR_DEFAULTS;
var EDITOR_FONT_DEFAULTS = editorOptions.EDITOR_FONT_DEFAULTS;
var EDITOR_MODEL_DEFAULTS = editorOptions.EDITOR_MODEL_DEFAULTS;
export var TabFocus = new /** @class */ (function () {
    function class_1() {
        this._tabFocus = false;
        this._onDidChangeTabFocus = new Emitter();
        this.onDidChangeTabFocus = this._onDidChangeTabFocus.event;
    }
    class_1.prototype.getTabFocusMode = function () {
        return this._tabFocus;
    };
    class_1.prototype.setTabFocusMode = function (tabFocusMode) {
        if (this._tabFocus === tabFocusMode) {
            return;
        }
        this._tabFocus = tabFocusMode;
        this._onDidChangeTabFocus.fire(this._tabFocus);
    };
    return class_1;
}());
var CommonEditorConfiguration = /** @class */ (function (_super) {
    __extends(CommonEditorConfiguration, _super);
    function CommonEditorConfiguration(options) {
        var _this = _super.call(this) || this;
        _this._onDidChange = _this._register(new Emitter());
        _this.onDidChange = _this._onDidChange.event;
        // Do a "deep clone of sorts" on the incoming options
        _this._rawOptions = objects.mixin({}, options || {});
        _this._rawOptions.scrollbar = objects.mixin({}, _this._rawOptions.scrollbar || {});
        _this._rawOptions.minimap = objects.mixin({}, _this._rawOptions.minimap || {});
        _this._rawOptions.find = objects.mixin({}, _this._rawOptions.find || {});
        _this._validatedOptions = editorOptions.EditorOptionsValidator.validate(_this._rawOptions, EDITOR_DEFAULTS);
        _this.editor = null;
        _this._isDominatedByLongLines = false;
        _this._lineNumbersDigitCount = 1;
        _this._register(EditorZoom.onDidChangeZoomLevel(function (_) { return _this._recomputeOptions(); }));
        _this._register(TabFocus.onDidChangeTabFocus(function (_) { return _this._recomputeOptions(); }));
        return _this;
    }
    CommonEditorConfiguration.prototype.dispose = function () {
        _super.prototype.dispose.call(this);
    };
    CommonEditorConfiguration.prototype._recomputeOptions = function () {
        var oldOptions = this.editor;
        var newOptions = this._computeInternalOptions();
        if (oldOptions && oldOptions.equals(newOptions)) {
            return;
        }
        this.editor = newOptions;
        if (oldOptions) {
            this._onDidChange.fire(oldOptions.createChangeEvent(newOptions));
        }
    };
    CommonEditorConfiguration.prototype.getRawOptions = function () {
        return this._rawOptions;
    };
    CommonEditorConfiguration.prototype._computeInternalOptions = function () {
        var opts = this._validatedOptions;
        var partialEnv = this._getEnvConfiguration();
        var bareFontInfo = BareFontInfo.createFromRawSettings(this._rawOptions, partialEnv.zoomLevel);
        var env = {
            outerWidth: partialEnv.outerWidth,
            outerHeight: partialEnv.outerHeight,
            fontInfo: this.readConfiguration(bareFontInfo),
            extraEditorClassName: partialEnv.extraEditorClassName,
            isDominatedByLongLines: this._isDominatedByLongLines,
            lineNumbersDigitCount: this._lineNumbersDigitCount,
            emptySelectionClipboard: partialEnv.emptySelectionClipboard,
            pixelRatio: partialEnv.pixelRatio,
            tabFocusMode: TabFocus.getTabFocusMode(),
            accessibilitySupport: partialEnv.accessibilitySupport
        };
        return editorOptions.InternalEditorOptionsFactory.createInternalEditorOptions(env, opts);
    };
    CommonEditorConfiguration.prototype.updateOptions = function (newOptions) {
        this._rawOptions = objects.mixin(this._rawOptions, newOptions || {});
        this._validatedOptions = editorOptions.EditorOptionsValidator.validate(this._rawOptions, EDITOR_DEFAULTS);
        this._recomputeOptions();
    };
    CommonEditorConfiguration.prototype.setIsDominatedByLongLines = function (isDominatedByLongLines) {
        this._isDominatedByLongLines = isDominatedByLongLines;
        this._recomputeOptions();
    };
    CommonEditorConfiguration.prototype.setMaxLineNumber = function (maxLineNumber) {
        var digitCount = CommonEditorConfiguration._digitCount(maxLineNumber);
        if (this._lineNumbersDigitCount === digitCount) {
            return;
        }
        this._lineNumbersDigitCount = digitCount;
        this._recomputeOptions();
    };
    CommonEditorConfiguration._digitCount = function (n) {
        var r = 0;
        while (n) {
            n = Math.floor(n / 10);
            r++;
        }
        return r ? r : 1;
    };
    return CommonEditorConfiguration;
}(Disposable));
export { CommonEditorConfiguration };
var configurationRegistry = Registry.as(Extensions.Configuration);
var editorConfiguration = {
    'id': 'editor',
    'order': 5,
    'type': 'object',
    'title': nls.localize('editorConfigurationTitle', "Editor"),
    'overridable': true,
    'scope': ConfigurationScope.RESOURCE,
    'properties': {
        'editor.fontFamily': {
            'type': 'string',
            'default': EDITOR_FONT_DEFAULTS.fontFamily,
            'description': nls.localize('fontFamily', "Controls the font family.")
        },
        'editor.fontWeight': {
            'type': 'string',
            'enum': ['normal', 'bold', '100', '200', '300', '400', '500', '600', '700', '800', '900'],
            'default': EDITOR_FONT_DEFAULTS.fontWeight,
            'description': nls.localize('fontWeight', "Controls the font weight.")
        },
        'editor.fontSize': {
            'type': 'number',
            'default': EDITOR_FONT_DEFAULTS.fontSize,
            'description': nls.localize('fontSize', "Controls the font size in pixels.")
        },
        'editor.lineHeight': {
            'type': 'number',
            'default': EDITOR_FONT_DEFAULTS.lineHeight,
            'description': nls.localize('lineHeight', "Controls the line height. Use 0 to compute the lineHeight from the fontSize.")
        },
        'editor.letterSpacing': {
            'type': 'number',
            'default': EDITOR_FONT_DEFAULTS.letterSpacing,
            'description': nls.localize('letterSpacing', "Controls the letter spacing in pixels.")
        },
        'editor.lineNumbers': {
            'type': 'string',
            'enum': ['off', 'on', 'relative', 'interval'],
            'enumDescriptions': [
                nls.localize('lineNumbers.off', "Line numbers are not rendered."),
                nls.localize('lineNumbers.on', "Line numbers are rendered as absolute number."),
                nls.localize('lineNumbers.relative', "Line numbers are rendered as distance in lines to cursor position."),
                nls.localize('lineNumbers.interval', "Line numbers are rendered every 10 lines.")
            ],
            'default': 'on',
            'description': nls.localize('lineNumbers', "Controls the display of line numbers. Possible values are 'on', 'off', 'relative' and 'interval'.")
        },
        'editor.rulers': {
            'type': 'array',
            'items': {
                'type': 'number'
            },
            'default': EDITOR_DEFAULTS.viewInfo.rulers,
            'description': nls.localize('rulers', "Render vertical rulers after a certain number of monospace characters. Use multiple values for multiple rulers. No rulers are drawn if array is empty")
        },
        'editor.wordSeparators': {
            'type': 'string',
            'default': EDITOR_DEFAULTS.wordSeparators,
            'description': nls.localize('wordSeparators', "Characters that will be used as word separators when doing word related navigations or operations")
        },
        'editor.tabSize': {
            'type': 'number',
            'default': EDITOR_MODEL_DEFAULTS.tabSize,
            'minimum': 1,
            'description': nls.localize('tabSize', "The number of spaces a tab is equal to. This setting is overridden based on the file contents when `editor.detectIndentation` is on."),
            'errorMessage': nls.localize('tabSize.errorMessage', "Expected 'number'. Note that the value \"auto\" has been replaced by the `editor.detectIndentation` setting.")
        },
        'editor.insertSpaces': {
            'type': 'boolean',
            'default': EDITOR_MODEL_DEFAULTS.insertSpaces,
            'description': nls.localize('insertSpaces', "Insert spaces when pressing Tab. This setting is overridden based on the file contents when `editor.detectIndentation` is on."),
            'errorMessage': nls.localize('insertSpaces.errorMessage', "Expected 'boolean'. Note that the value \"auto\" has been replaced by the `editor.detectIndentation` setting.")
        },
        'editor.detectIndentation': {
            'type': 'boolean',
            'default': EDITOR_MODEL_DEFAULTS.detectIndentation,
            'description': nls.localize('detectIndentation', "When opening a file, `editor.tabSize` and `editor.insertSpaces` will be detected based on the file contents.")
        },
        'editor.roundedSelection': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.roundedSelection,
            'description': nls.localize('roundedSelection', "Controls if selections have rounded corners")
        },
        'editor.scrollBeyondLastLine': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.scrollBeyondLastLine,
            'description': nls.localize('scrollBeyondLastLine', "Controls if the editor will scroll beyond the last line")
        },
        'editor.smoothScrolling': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.smoothScrolling,
            'description': nls.localize('smoothScrolling', "Controls if the editor will scroll using an animation")
        },
        'editor.minimap.enabled': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.minimap.enabled,
            'description': nls.localize('minimap.enabled', "Controls if the minimap is shown")
        },
        'editor.minimap.side': {
            'type': 'string',
            'enum': ['left', 'right'],
            'default': EDITOR_DEFAULTS.viewInfo.minimap.side,
            'description': nls.localize('minimap.side', "Controls the side where to render the minimap. Possible values are \'right\' and \'left\'")
        },
        'editor.minimap.showSlider': {
            'type': 'string',
            'enum': ['always', 'mouseover'],
            'default': EDITOR_DEFAULTS.viewInfo.minimap.showSlider,
            'description': nls.localize('minimap.showSlider', "Controls whether the minimap slider is automatically hidden. Possible values are \'always\' and \'mouseover\'")
        },
        'editor.minimap.renderCharacters': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.minimap.renderCharacters,
            'description': nls.localize('minimap.renderCharacters', "Render the actual characters on a line (as opposed to color blocks)")
        },
        'editor.minimap.maxColumn': {
            'type': 'number',
            'default': EDITOR_DEFAULTS.viewInfo.minimap.maxColumn,
            'description': nls.localize('minimap.maxColumn', "Limit the width of the minimap to render at most a certain number of columns")
        },
        'editor.find.seedSearchStringFromSelection': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.find.seedSearchStringFromSelection,
            'description': nls.localize('find.seedSearchStringFromSelection', "Controls if we seed the search string in Find Widget from editor selection")
        },
        'editor.find.autoFindInSelection': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.find.autoFindInSelection,
            'description': nls.localize('find.autoFindInSelection', "Controls if Find in Selection flag is turned on when multiple characters or lines of text are selected in the editor")
        },
        'editor.find.globalFindClipboard': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.find.globalFindClipboard,
            'description': nls.localize('find.globalFindClipboard', "Controls if the Find Widget should read or modify the shared find clipboard on macOS"),
            'included': platform.isMacintosh
        },
        'editor.wordWrap': {
            'type': 'string',
            'enum': ['off', 'on', 'wordWrapColumn', 'bounded'],
            'enumDescriptions': [
                nls.localize('wordWrap.off', "Lines will never wrap."),
                nls.localize('wordWrap.on', "Lines will wrap at the viewport width."),
                nls.localize({
                    key: 'wordWrap.wordWrapColumn',
                    comment: [
                        '- `editor.wordWrapColumn` refers to a different setting and should not be localized.'
                    ]
                }, "Lines will wrap at `editor.wordWrapColumn`."),
                nls.localize({
                    key: 'wordWrap.bounded',
                    comment: [
                        '- viewport means the edge of the visible window size.',
                        '- `editor.wordWrapColumn` refers to a different setting and should not be localized.'
                    ]
                }, "Lines will wrap at the minimum of viewport and `editor.wordWrapColumn`."),
            ],
            'default': EDITOR_DEFAULTS.wordWrap,
            'description': nls.localize({
                key: 'wordWrap',
                comment: [
                    '- \'off\', \'on\', \'wordWrapColumn\' and \'bounded\' refer to values the setting can take and should not be localized.',
                    '- `editor.wordWrapColumn` refers to a different setting and should not be localized.'
                ]
            }, "Controls how lines should wrap. Can be:\n - 'off' (disable wrapping),\n - 'on' (viewport wrapping),\n - 'wordWrapColumn' (wrap at `editor.wordWrapColumn`) or\n - 'bounded' (wrap at minimum of viewport and `editor.wordWrapColumn`).")
        },
        'editor.wordWrapColumn': {
            'type': 'integer',
            'default': EDITOR_DEFAULTS.wordWrapColumn,
            'minimum': 1,
            'description': nls.localize({
                key: 'wordWrapColumn',
                comment: [
                    '- `editor.wordWrap` refers to a different setting and should not be localized.',
                    '- \'wordWrapColumn\' and \'bounded\' refer to values the different setting can take and should not be localized.'
                ]
            }, "Controls the wrapping column of the editor when `editor.wordWrap` is 'wordWrapColumn' or 'bounded'.")
        },
        'editor.wrappingIndent': {
            'type': 'string',
            'enum': ['none', 'same', 'indent'],
            'default': 'same',
            'description': nls.localize('wrappingIndent', "Controls the indentation of wrapped lines. Can be one of 'none', 'same' or 'indent'.")
        },
        'editor.mouseWheelScrollSensitivity': {
            'type': 'number',
            'default': EDITOR_DEFAULTS.viewInfo.scrollbar.mouseWheelScrollSensitivity,
            'description': nls.localize('mouseWheelScrollSensitivity', "A multiplier to be used on the `deltaX` and `deltaY` of mouse wheel scroll events")
        },
        'editor.multiCursorModifier': {
            'type': 'string',
            'enum': ['ctrlCmd', 'alt'],
            'enumDescriptions': [
                nls.localize('multiCursorModifier.ctrlCmd', "Maps to `Control` on Windows and Linux and to `Command` on macOS."),
                nls.localize('multiCursorModifier.alt', "Maps to `Alt` on Windows and Linux and to `Option` on macOS.")
            ],
            'default': 'alt',
            'description': nls.localize({
                key: 'multiCursorModifier',
                comment: [
                    '- `ctrlCmd` refers to a value the setting can take and should not be localized.',
                    '- `Control` and `Command` refer to the modifier keys Ctrl or Cmd on the keyboard and can be localized.'
                ]
            }, "The modifier to be used to add multiple cursors with the mouse. `ctrlCmd` maps to `Control` on Windows and Linux and to `Command` on macOS. The Go To Definition and Open Link mouse gestures will adapt such that they do not conflict with the multicursor modifier.")
        },
        'editor.quickSuggestions': {
            'anyOf': [
                {
                    type: 'boolean',
                },
                {
                    type: 'object',
                    properties: {
                        strings: {
                            type: 'boolean',
                            default: false,
                            description: nls.localize('quickSuggestions.strings', "Enable quick suggestions inside strings.")
                        },
                        comments: {
                            type: 'boolean',
                            default: false,
                            description: nls.localize('quickSuggestions.comments', "Enable quick suggestions inside comments.")
                        },
                        other: {
                            type: 'boolean',
                            default: true,
                            description: nls.localize('quickSuggestions.other', "Enable quick suggestions outside of strings and comments.")
                        },
                    }
                }
            ],
            'default': EDITOR_DEFAULTS.contribInfo.quickSuggestions,
            'description': nls.localize('quickSuggestions', "Controls if suggestions should automatically show up while typing")
        },
        'editor.quickSuggestionsDelay': {
            'type': 'integer',
            'default': EDITOR_DEFAULTS.contribInfo.quickSuggestionsDelay,
            'minimum': 0,
            'description': nls.localize('quickSuggestionsDelay', "Controls the delay in ms after which quick suggestions will show up")
        },
        'editor.parameterHints': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.parameterHints,
            'description': nls.localize('parameterHints', "Enables pop-up that shows parameter documentation and type information as you type")
        },
        'editor.autoClosingBrackets': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.autoClosingBrackets,
            'description': nls.localize('autoClosingBrackets', "Controls if the editor should automatically close brackets after opening them")
        },
        'editor.formatOnType': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.formatOnType,
            'description': nls.localize('formatOnType', "Controls if the editor should automatically format the line after typing")
        },
        'editor.formatOnPaste': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.formatOnPaste,
            'description': nls.localize('formatOnPaste', "Controls if the editor should automatically format the pasted content. A formatter must be available and the formatter should be able to format a range in a document.")
        },
        'editor.autoIndent': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.autoIndent,
            'description': nls.localize('autoIndent', "Controls if the editor should automatically adjust the indentation when users type, paste or move lines. Indentation rules of the language must be available.")
        },
        'editor.suggestOnTriggerCharacters': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.suggestOnTriggerCharacters,
            'description': nls.localize('suggestOnTriggerCharacters', "Controls if suggestions should automatically show up when typing trigger characters")
        },
        'editor.acceptSuggestionOnEnter': {
            'type': 'string',
            'enum': ['on', 'smart', 'off'],
            'default': EDITOR_DEFAULTS.contribInfo.acceptSuggestionOnEnter,
            'description': nls.localize('acceptSuggestionOnEnter', "Controls if suggestions should be accepted on 'Enter' - in addition to 'Tab'. Helps to avoid ambiguity between inserting new lines or accepting suggestions. The value 'smart' means only accept a suggestion with Enter when it makes a textual change")
        },
        'editor.acceptSuggestionOnCommitCharacter': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.acceptSuggestionOnCommitCharacter,
            'description': nls.localize('acceptSuggestionOnCommitCharacter', "Controls if suggestions should be accepted on commit characters. For instance in JavaScript the semi-colon (';') can be a commit character that accepts a suggestion and types that character.")
        },
        'editor.snippetSuggestions': {
            'type': 'string',
            'enum': ['top', 'bottom', 'inline', 'none'],
            'enumDescriptions': [
                nls.localize('snippetSuggestions.top', "Show snippet suggestions on top of other suggestions."),
                nls.localize('snippetSuggestions.bottom', "Show snippet suggestions below other suggestions."),
                nls.localize('snippetSuggestions.inline', "Show snippets suggestions with other suggestions."),
                nls.localize('snippetSuggestions.none', "Do not show snippet suggestions."),
            ],
            'default': EDITOR_DEFAULTS.contribInfo.snippetSuggestions,
            'description': nls.localize('snippetSuggestions', "Controls whether snippets are shown with other suggestions and how they are sorted.")
        },
        'editor.emptySelectionClipboard': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.emptySelectionClipboard,
            'description': nls.localize('emptySelectionClipboard', "Controls whether copying without a selection copies the current line.")
        },
        'editor.wordBasedSuggestions': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.wordBasedSuggestions,
            'description': nls.localize('wordBasedSuggestions', "Controls whether completions should be computed based on words in the document.")
        },
        'editor.suggestSelection': {
            'type': 'string',
            'enum': ['first', 'recentlyUsed', 'recentlyUsedByPrefix'],
            'enumDescriptions': [
                nls.localize('suggestSelection.first', "Always select the first suggestion."),
                nls.localize('suggestSelection.recentlyUsed', "Select recent suggestions unless further typing selects one, e.g. `console.| -> console.log` because `log` has been completed recently."),
                nls.localize('suggestSelection.recentlyUsedByPrefix', "Select suggestions based on previous prefixes that have completed those suggestions, e.g. `co -> console` and `con -> const`."),
            ],
            'default': 'recentlyUsed',
            'description': nls.localize('suggestSelection', "Controls how suggestions are pre-selected when showing the suggest list.")
        },
        'editor.suggestFontSize': {
            'type': 'integer',
            'default': 0,
            'minimum': 0,
            'description': nls.localize('suggestFontSize', "Font size for the suggest widget")
        },
        'editor.suggestLineHeight': {
            'type': 'integer',
            'default': 0,
            'minimum': 0,
            'description': nls.localize('suggestLineHeight', "Line height for the suggest widget")
        },
        'editor.selectionHighlight': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.selectionHighlight,
            'description': nls.localize('selectionHighlight', "Controls whether the editor should highlight similar matches to the selection")
        },
        'editor.occurrencesHighlight': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.occurrencesHighlight,
            'description': nls.localize('occurrencesHighlight', "Controls whether the editor should highlight semantic symbol occurrences")
        },
        'editor.overviewRulerLanes': {
            'type': 'integer',
            'default': 3,
            'description': nls.localize('overviewRulerLanes', "Controls the number of decorations that can show up at the same position in the overview ruler")
        },
        'editor.overviewRulerBorder': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.overviewRulerBorder,
            'description': nls.localize('overviewRulerBorder', "Controls if a border should be drawn around the overview ruler.")
        },
        'editor.cursorBlinking': {
            'type': 'string',
            'enum': ['blink', 'smooth', 'phase', 'expand', 'solid'],
            'default': editorOptions.blinkingStyleToString(EDITOR_DEFAULTS.viewInfo.cursorBlinking),
            'description': nls.localize('cursorBlinking', "Control the cursor animation style, possible values are 'blink', 'smooth', 'phase', 'expand' and 'solid'")
        },
        'editor.mouseWheelZoom': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.mouseWheelZoom,
            'description': nls.localize('mouseWheelZoom', "Zoom the font of the editor when using mouse wheel and holding Ctrl")
        },
        'editor.cursorStyle': {
            'type': 'string',
            'enum': ['block', 'block-outline', 'line', 'line-thin', 'underline', 'underline-thin'],
            'default': editorOptions.cursorStyleToString(EDITOR_DEFAULTS.viewInfo.cursorStyle),
            'description': nls.localize('cursorStyle', "Controls the cursor style, accepted values are 'block', 'block-outline', 'line', 'line-thin', 'underline' and 'underline-thin'")
        },
        'editor.cursorWidth': {
            'type': 'integer',
            'default': EDITOR_DEFAULTS.viewInfo.cursorWidth,
            'description': nls.localize('cursorWidth', "Controls the width of the cursor when editor.cursorStyle is set to 'line'")
        },
        'editor.fontLigatures': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.fontLigatures,
            'description': nls.localize('fontLigatures', "Enables font ligatures")
        },
        'editor.hideCursorInOverviewRuler': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.hideCursorInOverviewRuler,
            'description': nls.localize('hideCursorInOverviewRuler', "Controls if the cursor should be hidden in the overview ruler.")
        },
        'editor.renderWhitespace': {
            'type': 'string',
            'enum': ['none', 'boundary', 'all'],
            default: EDITOR_DEFAULTS.viewInfo.renderWhitespace,
            description: nls.localize('renderWhitespace', "Controls how the editor should render whitespace characters, possibilities are 'none', 'boundary', and 'all'. The 'boundary' option does not render single spaces between words.")
        },
        'editor.renderControlCharacters': {
            'type': 'boolean',
            default: EDITOR_DEFAULTS.viewInfo.renderControlCharacters,
            description: nls.localize('renderControlCharacters', "Controls whether the editor should render control characters")
        },
        'editor.renderIndentGuides': {
            'type': 'boolean',
            default: EDITOR_DEFAULTS.viewInfo.renderIndentGuides,
            description: nls.localize('renderIndentGuides', "Controls whether the editor should render indent guides")
        },
        'editor.renderLineHighlight': {
            'type': 'string',
            'enum': ['none', 'gutter', 'line', 'all'],
            default: EDITOR_DEFAULTS.viewInfo.renderLineHighlight,
            description: nls.localize('renderLineHighlight', "Controls how the editor should render the current line highlight, possibilities are 'none', 'gutter', 'line', and 'all'.")
        },
        'editor.codeLens': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.codeLens,
            'description': nls.localize('codeLens', "Controls if the editor shows code lenses")
        },
        'editor.folding': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.folding,
            'description': nls.localize('folding', "Controls whether the editor has code folding enabled")
        },
        'editor.foldingStrategy': {
            'type': 'string',
            'enum': ['auto', 'indentation'],
            'enumDescriptions': [
                nls.localize('foldingStrategyAuto', 'If available, use a langauge specific folding strategy, otherwise falls back to the indentation based strategy.'),
                nls.localize('foldingStrategyIndentation', 'Always use the indentation based folding strategy')
            ],
            'default': EDITOR_DEFAULTS.contribInfo.foldingStrategy,
            'description': nls.localize('foldingStrategy', "Controls the way folding ranges are computed. 'auto' picks uses a language specific folding strategy, if available. 'indentation' forces that the indentation based folding strategy is used.")
        },
        'editor.showFoldingControls': {
            'type': 'string',
            'enum': ['always', 'mouseover'],
            'default': EDITOR_DEFAULTS.contribInfo.showFoldingControls,
            'description': nls.localize('showFoldingControls', "Controls whether the fold controls on the gutter are automatically hidden.")
        },
        'editor.matchBrackets': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.matchBrackets,
            'description': nls.localize('matchBrackets', "Highlight matching brackets when one of them is selected.")
        },
        'editor.glyphMargin': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.viewInfo.glyphMargin,
            'description': nls.localize('glyphMargin', "Controls whether the editor should render the vertical glyph margin. Glyph margin is mostly used for debugging.")
        },
        'editor.useTabStops': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.useTabStops,
            'description': nls.localize('useTabStops', "Inserting and deleting whitespace follows tab stops")
        },
        'editor.trimAutoWhitespace': {
            'type': 'boolean',
            'default': EDITOR_MODEL_DEFAULTS.trimAutoWhitespace,
            'description': nls.localize('trimAutoWhitespace', "Remove trailing auto inserted whitespace")
        },
        'editor.stablePeek': {
            'type': 'boolean',
            'default': false,
            'description': nls.localize('stablePeek', "Keep peek editors open even when double clicking their content or when hitting Escape.")
        },
        'editor.dragAndDrop': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.dragAndDrop,
            'description': nls.localize('dragAndDrop', "Controls if the editor should allow to move selections via drag and drop.")
        },
        'editor.accessibilitySupport': {
            'type': 'string',
            'enum': ['auto', 'on', 'off'],
            'enumDescriptions': [
                nls.localize('accessibilitySupport.auto', "The editor will use platform APIs to detect when a Screen Reader is attached."),
                nls.localize('accessibilitySupport.on', "The editor will be permanently optimized for usage with a Screen Reader."),
                nls.localize('accessibilitySupport.off', "The editor will never be optimized for usage with a Screen Reader."),
            ],
            'default': EDITOR_DEFAULTS.accessibilitySupport,
            'description': nls.localize('accessibilitySupport', "Controls whether the editor should run in a mode where it is optimized for screen readers.")
        },
        'editor.links': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.links,
            'description': nls.localize('links', "Controls whether the editor should detect links and make them clickable")
        },
        'editor.colorDecorators': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.colorDecorators,
            'description': nls.localize('colorDecorators', "Controls whether the editor should render the inline color decorators and color picker.")
        },
        'editor.lightbulb.enabled': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.lightbulbEnabled,
            'description': nls.localize('codeActions', "Enables the code action lightbulb")
        },
        'editor.selectionClipboard': {
            'type': 'boolean',
            'default': EDITOR_DEFAULTS.contribInfo.selectionClipboard,
            'description': nls.localize('selectionClipboard', "Controls if the Linux primary clipboard should be supported."),
            'included': platform.isLinux
        },
        'diffEditor.renderSideBySide': {
            'type': 'boolean',
            'default': true,
            'description': nls.localize('sideBySide', "Controls if the diff editor shows the diff side by side or inline")
        },
        'diffEditor.ignoreTrimWhitespace': {
            'type': 'boolean',
            'default': true,
            'description': nls.localize('ignoreTrimWhitespace', "Controls if the diff editor shows changes in leading or trailing whitespace as diffs")
        },
        'diffEditor.renderIndicators': {
            'type': 'boolean',
            'default': true,
            'description': nls.localize('renderIndicators', "Controls if the diff editor shows +/- indicators for added/removed changes")
        }
    }
};
configurationRegistry.registerConfiguration(editorConfiguration);
