/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
import { onUnexpectedError } from '../../../base/common/errors.js';
var EditStack = /** @class */ (function () {
    function EditStack(model) {
        this.model = model;
        this.currentOpenStackElement = null;
        this.past = [];
        this.future = [];
    }
    EditStack.prototype.pushStackElement = function () {
        if (this.currentOpenStackElement !== null) {
            this.past.push(this.currentOpenStackElement);
            this.currentOpenStackElement = null;
        }
    };
    EditStack.prototype.clear = function () {
        this.currentOpenStackElement = null;
        this.past = [];
        this.future = [];
    };
    EditStack.prototype.pushEditOperation = function (beforeCursorState, editOperations, cursorStateComputer) {
        // No support for parallel universes :(
        this.future = [];
        if (!this.currentOpenStackElement) {
            this.currentOpenStackElement = {
                beforeVersionId: this.model.getAlternativeVersionId(),
                beforeCursorState: beforeCursorState,
                editOperations: [],
                afterCursorState: null,
                afterVersionId: -1
            };
        }
        var inverseEditOperation = {
            operations: this.model.applyEdits(editOperations)
        };
        this.currentOpenStackElement.editOperations.push(inverseEditOperation);
        try {
            this.currentOpenStackElement.afterCursorState = cursorStateComputer ? cursorStateComputer(inverseEditOperation.operations) : null;
        }
        catch (e) {
            onUnexpectedError(e);
            this.currentOpenStackElement.afterCursorState = null;
        }
        this.currentOpenStackElement.afterVersionId = this.model.getVersionId();
        return this.currentOpenStackElement.afterCursorState;
    };
    EditStack.prototype.undo = function () {
        this.pushStackElement();
        if (this.past.length > 0) {
            var pastStackElement = this.past.pop();
            try {
                // Apply all operations in reverse order
                for (var i = pastStackElement.editOperations.length - 1; i >= 0; i--) {
                    pastStackElement.editOperations[i] = {
                        operations: this.model.applyEdits(pastStackElement.editOperations[i].operations)
                    };
                }
            }
            catch (e) {
                this.clear();
                return null;
            }
            this.future.push(pastStackElement);
            return {
                selections: pastStackElement.beforeCursorState,
                recordedVersionId: pastStackElement.beforeVersionId
            };
        }
        return null;
    };
    EditStack.prototype.redo = function () {
        if (this.future.length > 0) {
            if (this.currentOpenStackElement) {
                throw new Error('How is this possible?');
            }
            var futureStackElement = this.future.pop();
            try {
                // Apply all operations
                for (var i = 0; i < futureStackElement.editOperations.length; i++) {
                    futureStackElement.editOperations[i] = {
                        operations: this.model.applyEdits(futureStackElement.editOperations[i].operations)
                    };
                }
            }
            catch (e) {
                this.clear();
                return null;
            }
            this.past.push(futureStackElement);
            return {
                selections: futureStackElement.afterCursorState,
                recordedVersionId: futureStackElement.afterVersionId
            };
        }
        return null;
    };
    return EditStack;
}());
export { EditStack };
