import { getRange } from 'graphql-language-service';
import { LanguageService } from './LanguageService';
import { toGraphQLPosition, toMonacoRange, toMarkerData, toCompletion, } from './utils';
export class GraphQLWorker {
    _ctx;
    _languageService;
    _formattingOptions;
    constructor(ctx, createData) {
        this._ctx = ctx;
        this._languageService = new LanguageService(createData.languageConfig);
        this._formattingOptions = createData.formattingOptions;
    }
    async doValidation(uri) {
        try {
            const documentModel = this._getTextModel(uri);
            const document = documentModel?.getValue();
            if (!document) {
                return [];
            }
            const graphqlDiagnostics = this._languageService.getDiagnostics(uri, document);
            return graphqlDiagnostics.map(toMarkerData);
        }
        catch (err) {
            console.error(err);
            return [];
        }
    }
    async doComplete(uri, position) {
        try {
            const documentModel = this._getTextModel(uri);
            const document = documentModel?.getValue();
            if (!document) {
                return [];
            }
            const graphQLPosition = toGraphQLPosition(position);
            const suggestions = this._languageService.getCompletion(uri, document, graphQLPosition);
            return suggestions.map(suggestion => toCompletion(suggestion));
        }
        catch (err) {
            console.error(err);
            return [];
        }
    }
    async doHover(uri, position) {
        try {
            const documentModel = this._getTextModel(uri);
            const document = documentModel?.getValue();
            if (!document) {
                return null;
            }
            const graphQLPosition = toGraphQLPosition(position);
            const hover = this._languageService.getHover(uri, document, graphQLPosition);
            const location = {
                column: graphQLPosition.character,
                line: graphQLPosition.line,
            };
            return {
                content: hover,
                range: toMonacoRange(getRange(location, document)),
            };
        }
        catch (err) {
            console.error(err);
            return null;
        }
    }
    async doGetVariablesJSONSchema(uri) {
        const documentModel = this._getTextModel(uri);
        const document = documentModel?.getValue();
        if (!documentModel || !document) {
            return null;
        }
        const jsonSchema = this._languageService.getVariablesJSONSchema(uri, document, { useMarkdownDescription: true });
        if (jsonSchema) {
            return {
                ...jsonSchema,
                $id: 'monaco://variables-schema.json',
                title: 'GraphQL Variables',
            };
        }
        return null;
    }
    async doFormat(uri) {
        const documentModel = this._getTextModel(uri);
        const document = documentModel?.getValue();
        if (!documentModel || !document) {
            return null;
        }
        const prettierStandalone = await import('prettier/standalone');
        const prettierGraphqlParser = await import('prettier/parser-graphql');
        return prettierStandalone.format(document, {
            parser: 'graphql',
            plugins: [prettierGraphqlParser],
            ...this._formattingOptions?.prettierConfig,
        });
    }
    _getTextModel(uri) {
        const models = this._ctx.getMirrorModels();
        for (const model of models) {
            if (model.uri.toString() === uri) {
                return model;
            }
        }
        return null;
    }
    doUpdateSchema(schema) {
        return this._languageService.updateSchema(schema);
    }
    doUpdateSchemas(schemas) {
        return this._languageService.updateSchemas(schemas);
    }
}
export default {
    GraphQLWorker,
};
export function create(ctx, createData) {
    return new GraphQLWorker(ctx, createData);
}
//# sourceMappingURL=GraphQLWorker.js.map