import { parse, visit, } from 'graphql';
import picomatch from 'picomatch-browser';
import { getAutocompleteSuggestions, getDiagnostics, getHoverInformation, getVariablesJSONSchema, getOperationASTFacts, } from 'graphql-language-service';
import { defaultSchemaLoader } from './schemaLoader';
const schemaCache = new Map();
export class LanguageService {
    _parser = parse;
    _schemas = [];
    _schemaCache = schemaCache;
    _schemaLoader = defaultSchemaLoader;
    _parseOptions;
    _customValidationRules;
    _externalFragmentDefinitionNodes = null;
    _externalFragmentDefinitionsString = null;
    _completionSettings;
    constructor({ parser, schemas, parseOptions, externalFragmentDefinitions, customValidationRules, fillLeafsOnComplete, completionSettings, }) {
        this._schemaLoader = defaultSchemaLoader;
        if (schemas) {
            this._schemas = schemas;
            this._cacheSchemas();
        }
        if (parser) {
            this._parser = parser;
        }
        this._completionSettings = {
            ...completionSettings,
            fillLeafsOnComplete: completionSettings?.fillLeafsOnComplete ?? fillLeafsOnComplete,
        };
        if (parseOptions) {
            this._parseOptions = parseOptions;
        }
        if (customValidationRules) {
            this._customValidationRules = customValidationRules;
        }
        if (externalFragmentDefinitions) {
            if (Array.isArray(externalFragmentDefinitions)) {
                this._externalFragmentDefinitionNodes = externalFragmentDefinitions;
            }
            else {
                this._externalFragmentDefinitionsString = externalFragmentDefinitions;
            }
        }
    }
    _cacheSchemas() {
        for (const schema of this._schemas) {
            this._cacheSchema(schema);
        }
    }
    _cacheSchema(schemaConfig) {
        const schema = this._schemaLoader(schemaConfig, this.parse.bind(this));
        return this._schemaCache.set(schemaConfig.uri, {
            ...schemaConfig,
            schema,
        });
    }
    getSchemaForFile(uri) {
        if (!this._schemas.length) {
            return;
        }
        if (this._schemas.length === 1) {
            return this._schemaCache.get(this._schemas[0].uri);
        }
        const schema = this._schemas.find(schemaConfig => {
            if (!schemaConfig.fileMatch) {
                return false;
            }
            return schemaConfig.fileMatch.some(glob => {
                const isMatch = picomatch(glob);
                return isMatch(uri);
            });
        });
        if (schema) {
            const cacheEntry = this._schemaCache.get(schema.uri);
            if (cacheEntry) {
                return cacheEntry;
            }
            const cache = this._cacheSchema(schema);
            return cache.get(schema.uri);
        }
    }
    getExternalFragmentDefinitions() {
        if (!this._externalFragmentDefinitionNodes &&
            this._externalFragmentDefinitionsString) {
            const definitionNodes = [];
            try {
                visit(this._parser(this._externalFragmentDefinitionsString), {
                    FragmentDefinition(node) {
                        definitionNodes.push(node);
                    },
                });
            }
            catch {
                throw new Error(`Failed parsing externalFragmentDefinitions string:\n${this._externalFragmentDefinitionsString}`);
            }
            this._externalFragmentDefinitionNodes = definitionNodes;
        }
        return this._externalFragmentDefinitionNodes;
    }
    async updateSchemas(schemas) {
        this._schemas = schemas;
        this._cacheSchemas();
    }
    updateSchema(schema) {
        const schemaIndex = this._schemas.findIndex(c => c.uri === schema.uri);
        if (schemaIndex < 0) {
            console.warn('updateSchema could not find a schema in your config by that URI', schema.uri);
            return;
        }
        this._schemas[schemaIndex] = schema;
        this._cacheSchema(schema);
    }
    addSchema(schema) {
        this._schemas.push(schema);
        this._cacheSchema(schema);
    }
    parse(text, options) {
        return this._parser(text, options || this._parseOptions);
    }
    getCompletion = (uri, documentText, position) => {
        const schema = this.getSchemaForFile(uri);
        if (!documentText || !schema?.schema) {
            return [];
        }
        return getAutocompleteSuggestions(schema.schema, documentText, position, undefined, this.getExternalFragmentDefinitions(), { uri, ...this._completionSettings });
    };
    getDiagnostics = (uri, documentText, customRules) => {
        const schema = this.getSchemaForFile(uri);
        if (!documentText || documentText.trim().length < 2 || !schema?.schema) {
            return [];
        }
        return getDiagnostics(documentText, schema.schema, customRules ?? this._customValidationRules, false, this.getExternalFragmentDefinitions());
    };
    getHover = (uri, documentText, position, options) => {
        const schema = this.getSchemaForFile(uri);
        if (schema && documentText.length > 3) {
            return getHoverInformation(schema.schema, documentText, position, undefined, {
                useMarkdown: true,
                ...options,
            });
        }
    };
    getVariablesJSONSchema = (uri, documentText, options) => {
        const schema = this.getSchemaForFile(uri);
        if (schema && documentText.length > 3) {
            try {
                const documentAST = this.parse(documentText);
                const { variableToType } = getOperationASTFacts(documentAST, schema.schema);
                if (variableToType) {
                    return getVariablesJSONSchema(variableToType, {
                        ...options,
                        scalarSchemas: schema.customScalarSchemas,
                    });
                }
            }
            catch { }
        }
        return null;
    };
}
//# sourceMappingURL=LanguageService.js.map