import { languages } from './monaco-editor';
import { WorkerManager } from './workerManager';
import * as languageFeatures from './languageFeatures';
export function setupMode(defaults) {
    const disposables = [];
    const providers = [];
    const client = new WorkerManager(defaults);
    disposables.push(client);
    const worker = (...uris) => {
        try {
            return client.getLanguageServiceWorker(...uris);
        }
        catch {
            throw new Error('Error fetching graphql language service worker');
        }
    };
    function registerSchemaLessProviders() {
        const { modeConfiguration, languageId } = defaults;
        if (modeConfiguration.documentFormattingEdits) {
            providers.push(languages.registerDocumentFormattingEditProvider(languageId, new languageFeatures.DocumentFormattingAdapter(worker)));
        }
    }
    function registerAllProviders(api) {
        const { modeConfiguration, languageId } = defaults;
        disposeAll(providers);
        if (modeConfiguration.completionItems) {
            providers.push(languages.registerCompletionItemProvider(languageId, new languageFeatures.CompletionAdapter(worker)));
        }
        if (modeConfiguration.diagnostics) {
            providers.push(new languageFeatures.DiagnosticsAdapter(api, worker));
        }
        if (modeConfiguration.hovers) {
            providers.push(languages.registerHoverProvider(languageId, new languageFeatures.HoverAdapter(worker)));
        }
        registerSchemaLessProviders();
    }
    let { modeConfiguration, formattingOptions, diagnosticSettings, externalFragmentDefinitions, schemas, } = defaults;
    registerAllProviders(defaults);
    defaults.onDidChange(newDefaults => {
        if (newDefaults.modeConfiguration !== modeConfiguration) {
            modeConfiguration = newDefaults.modeConfiguration;
            registerAllProviders(newDefaults);
        }
        if (newDefaults.formattingOptions !== formattingOptions) {
            formattingOptions = newDefaults.formattingOptions;
            registerSchemaLessProviders();
        }
        if (newDefaults.externalFragmentDefinitions !== externalFragmentDefinitions) {
            externalFragmentDefinitions = newDefaults.externalFragmentDefinitions;
            registerAllProviders(newDefaults);
        }
        if (newDefaults.diagnosticSettings !== diagnosticSettings) {
            diagnosticSettings = newDefaults.diagnosticSettings;
            registerAllProviders(newDefaults);
        }
        if (newDefaults.schemas !== schemas) {
            schemas = newDefaults.schemas;
            registerAllProviders(newDefaults);
        }
    });
    disposables.push(asDisposable(providers));
    return asDisposable(disposables);
}
function asDisposable(disposables) {
    return { dispose: () => disposeAll(disposables) };
}
function disposeAll(disposables) {
    while (disposables.length) {
        disposables.pop().dispose();
    }
}
//# sourceMappingURL=graphqlMode.js.map