import { Uri, languages } from './monaco-editor';
import { editor } from 'monaco-editor/esm/vs/editor/editor.api';
import { CompletionItemKind as lsCompletionItemKind } from 'graphql-language-service';
import { getModelLanguageId } from './utils';
export class DiagnosticsAdapter {
    defaults;
    _worker;
    _disposables = [];
    _listener = Object.create(null);
    constructor(defaults, _worker) {
        this.defaults = defaults;
        this._worker = _worker;
        this._worker = _worker;
        let onChangeTimeout;
        const onModelAdd = (model) => {
            const modeId = getModelLanguageId(model);
            if (modeId !== this.defaults.languageId) {
                return;
            }
            const modelUri = model.uri.toString();
            const jsonValidationForModel = defaults.diagnosticSettings.validateVariablesJSON?.[modelUri];
            onChangeTimeout = setTimeout(() => {
                void this._doValidate(model.uri, modeId, jsonValidationForModel);
            }, 400);
            this._listener[modelUri] = model.onDidChangeContent(() => {
                clearTimeout(onChangeTimeout);
                onChangeTimeout = setTimeout(() => {
                    void this._doValidate(model.uri, modeId, jsonValidationForModel);
                }, 400);
            });
        };
        const onModelRemoved = (model) => {
            editor.setModelMarkers(model, this.defaults.languageId, []);
            const uriStr = model.uri.toString();
            const listener = this._listener[uriStr];
            if (listener) {
                listener.dispose();
                delete this._listener[uriStr];
            }
        };
        this._disposables.push(editor.onDidCreateModel(onModelAdd), {
            dispose() {
                clearTimeout(onChangeTimeout);
            },
        }, editor.onWillDisposeModel(model => {
            onModelRemoved(model);
        }), editor.onDidChangeModelLanguage(event => {
            onModelRemoved(event.model);
            onModelAdd(event.model);
        }), {
            dispose: () => {
                for (const listener of Object.values(this._listener)) {
                    listener.dispose();
                }
            },
        }, defaults.onDidChange(() => {
            for (const model of editor.getModels()) {
                if (getModelLanguageId(model) === this.defaults.languageId) {
                    onModelRemoved(model);
                    onModelAdd(model);
                }
            }
        }));
        for (const model of editor.getModels()) {
            if (getModelLanguageId(model) === this.defaults.languageId) {
                onModelAdd(model);
            }
        }
    }
    dispose() {
        for (const disposable of this._disposables) {
            disposable.dispose();
        }
        this._disposables = [];
    }
    async _doValidate(resource, languageId, variablesUris) {
        const worker = await this._worker(resource);
        if (!worker) {
            return;
        }
        const diagnostics = await worker.doValidation(resource.toString());
        editor.setModelMarkers(editor.getModel(resource), languageId, diagnostics);
        if (variablesUris) {
            await import('monaco-editor/esm/vs/language/json/monaco.contribution.js');
            if (!variablesUris.length) {
                throw new Error('No variables URI strings provided to validate');
            }
            const jsonSchema = await worker.doGetVariablesJSONSchema(resource.toString());
            if (!jsonSchema) {
                return;
            }
            const schemaUri = Uri.file(variablesUris[0].replace('.json', '-schema.json')).toString();
            const configResult = {
                uri: schemaUri,
                schema: jsonSchema,
                fileMatch: variablesUris,
            };
            const currentSchemas = languages.json.jsonDefaults.diagnosticsOptions.schemas?.filter(s => s.uri !== schemaUri) || [];
            languages.json.jsonDefaults.setDiagnosticsOptions({
                schemaValidation: 'error',
                validate: true,
                ...this.defaults.diagnosticSettings.jsonDiagnosticSettings,
                schemas: [...currentSchemas, configResult],
                enableSchemaRequest: false,
            });
        }
    }
}
const mKind = languages.CompletionItemKind;
const kindMap = {
    [lsCompletionItemKind.Text]: mKind.Text,
    [lsCompletionItemKind.Method]: mKind.Method,
    [lsCompletionItemKind.Function]: mKind.Function,
    [lsCompletionItemKind.Constructor]: mKind.Constructor,
    [lsCompletionItemKind.Field]: mKind.Field,
    [lsCompletionItemKind.Variable]: mKind.Variable,
    [lsCompletionItemKind.Class]: mKind.Class,
    [lsCompletionItemKind.Interface]: mKind.Interface,
    [lsCompletionItemKind.Module]: mKind.Module,
    [lsCompletionItemKind.Property]: mKind.Property,
    [lsCompletionItemKind.Unit]: mKind.Unit,
    [lsCompletionItemKind.Value]: mKind.Value,
    [lsCompletionItemKind.Enum]: mKind.Enum,
    [lsCompletionItemKind.Keyword]: mKind.Keyword,
    [lsCompletionItemKind.Snippet]: mKind.Snippet,
    [lsCompletionItemKind.Color]: mKind.Color,
    [lsCompletionItemKind.File]: mKind.File,
    [lsCompletionItemKind.Reference]: mKind.Reference,
    [lsCompletionItemKind.Folder]: mKind.Folder,
    [lsCompletionItemKind.EnumMember]: mKind.EnumMember,
    [lsCompletionItemKind.Constant]: mKind.Constant,
    [lsCompletionItemKind.Struct]: mKind.Struct,
    [lsCompletionItemKind.Event]: mKind.Event,
    [lsCompletionItemKind.Operator]: mKind.Operator,
    [lsCompletionItemKind.TypeParameter]: mKind.TypeParameter,
};
export function toCompletionItemKind(kind) {
    return kind in kindMap ? kindMap[kind] : mKind.Text;
}
export function toCompletion(entry) {
    const suggestions = {
        range: entry.range,
        kind: toCompletionItemKind(entry.kind),
        label: entry.label,
        insertText: entry.insertText ?? entry.label,
        insertTextRules: entry.insertText
            ? languages.CompletionItemInsertTextRule.InsertAsSnippet
            : undefined,
        sortText: entry.sortText,
        filterText: entry.filterText,
        documentation: entry.documentation,
        detail: entry.detail,
        command: entry.command,
    };
    return suggestions;
}
export class CompletionAdapter {
    _worker;
    constructor(_worker) {
        this._worker = _worker;
        this._worker = _worker;
    }
    get triggerCharacters() {
        return [':', '$', ' ', '(', '@'];
    }
    async provideCompletionItems(model, position, _context, _token) {
        try {
            const worker = await this._worker(model.uri);
            const completionItems = await worker.doComplete(model.uri.toString(), position);
            return {
                incomplete: true,
                suggestions: completionItems.map(toCompletion),
            };
        }
        catch (err) {
            console.error('Error fetching completion items', err);
            return { suggestions: [] };
        }
    }
}
export class DocumentFormattingAdapter {
    _worker;
    constructor(_worker) {
        this._worker = _worker;
        this._worker = _worker;
    }
    async provideDocumentFormattingEdits(document, _options, _token) {
        const worker = await this._worker(document.uri);
        const formatted = await worker.doFormat(document.uri.toString());
        if (!formatted) {
            return [];
        }
        return [
            {
                range: document.getFullModelRange(),
                text: formatted,
            },
        ];
    }
}
export class HoverAdapter {
    _worker;
    constructor(_worker) {
        this._worker = _worker;
    }
    async provideHover(model, position, _token) {
        const resource = model.uri;
        const worker = await this._worker(model.uri);
        const hoverItem = await worker.doHover(resource.toString(), position);
        if (hoverItem) {
            return {
                range: hoverItem.range,
                contents: [{ value: hoverItem.content }],
            };
        }
        return {
            contents: [],
        };
    }
    dispose() { }
}
//# sourceMappingURL=languageFeatures.js.map