import { buildASTSchema, printSchema } from 'graphql';
import { Position } from 'graphql-language-service';
export const getModelLanguageId = (model) => {
    if ('getModeId' in model) {
        return model.getModeId();
    }
    return model.getLanguageId();
};
export function toMonacoRange(range) {
    return {
        startLineNumber: range.start.line + 1,
        startColumn: range.start.character + 1,
        endLineNumber: range.end.line + 1,
        endColumn: range.end.character + 1,
    };
}
export function toGraphQLPosition(position) {
    return new Position(position.lineNumber - 1, position.column - 1);
}
export function toCompletion(entry, range) {
    return {
        label: entry.label,
        insertText: entry.insertText,
        sortText: entry.sortText,
        filterText: entry.filterText,
        ...(entry.documentation && {
            documentation: {
                value: entry.documentation,
            },
        }),
        detail: entry.detail,
        ...(range && { range: toMonacoRange(range) }),
        kind: entry.kind,
        ...(entry.insertTextFormat && { insertTextFormat: entry.insertTextFormat }),
        ...(entry.insertTextMode && { insertTextMode: entry.insertTextMode }),
        ...(entry.command && {
            command: { ...entry.command, id: entry.command.command },
        }),
        ...(entry.labelDetails && { labelDetails: entry.labelDetails }),
    };
}
export function toMonacoSeverity(severity) {
    const severityMap = {
        1: 8,
        2: 4,
        3: 2,
        4: 1,
    };
    return severity ? severityMap[severity] : severityMap[2];
}
export function toMarkerData(diagnostic) {
    return {
        startLineNumber: diagnostic.range.start.line + 1,
        endLineNumber: diagnostic.range.end.line + 1,
        startColumn: diagnostic.range.start.character + 1,
        endColumn: diagnostic.range.end.character,
        message: diagnostic.message,
        severity: toMonacoSeverity(diagnostic.severity),
        code: diagnostic.code || undefined,
    };
}
export const getStringSchema = (schemaConfig) => {
    const { schema: graphQLSchema, documentAST, introspectionJSON, introspectionJSONString, documentString, ...rest } = schemaConfig;
    if (graphQLSchema) {
        return {
            ...rest,
            documentString: printSchema(graphQLSchema),
        };
    }
    if (introspectionJSONString) {
        return {
            ...rest,
            introspectionJSONString,
        };
    }
    if (documentString) {
        return {
            ...rest,
            documentString,
        };
    }
    if (introspectionJSON) {
        return {
            ...rest,
            introspectionJSONString: JSON.stringify(introspectionJSON),
        };
    }
    if (documentAST) {
        const schema = buildASTSchema(documentAST, rest.buildSchemaOptions);
        return {
            ...rest,
            documentString: printSchema(schema),
        };
    }
    throw new Error('No schema supplied');
};
//# sourceMappingURL=utils.js.map