import { editor } from './monaco-editor';
import { getStringSchema } from './utils';
const STOP_WHEN_IDLE_FOR = 2 * 60 * 1000;
export class WorkerManager {
    _defaults;
    _idleCheckInterval;
    _lastUsedTime = 0;
    _configChangeListener;
    _worker = null;
    _client = null;
    constructor(defaults) {
        this._defaults = defaults;
        this._idleCheckInterval = window.setInterval(() => this._checkIfIdle(), 30 * 1000);
        this._configChangeListener = this._defaults.onDidChange(() => {
            this._stopWorker();
        });
    }
    _stopWorker() {
        if (this._worker) {
            this._worker.dispose();
            this._worker = null;
        }
        this._client = null;
    }
    dispose() {
        clearInterval(this._idleCheckInterval);
        this._configChangeListener.dispose();
        this._stopWorker();
    }
    _checkIfIdle() {
        if (!this._worker) {
            return;
        }
        const timePassedSinceLastUsed = Date.now() - this._lastUsedTime;
        if (timePassedSinceLastUsed > STOP_WHEN_IDLE_FOR) {
            this._stopWorker();
        }
    }
    async _getClient() {
        this._lastUsedTime = Date.now();
        if (!this._client && !this._worker) {
            try {
                const { languageId, formattingOptions, schemas, externalFragmentDefinitions, completionSettings, } = this._defaults;
                this._worker = editor.createWebWorker({
                    moduleId: 'monaco-graphql/esm/GraphQLWorker.js',
                    label: languageId,
                    createData: {
                        languageId,
                        formattingOptions,
                        languageConfig: {
                            schemas: schemas?.map(getStringSchema),
                            externalFragmentDefinitions,
                            fillLeafsOnComplete: completionSettings.__experimental__fillLeafsOnComplete,
                        },
                    },
                });
                this._client = this._worker.getProxy();
            }
            catch (error) {
                console.error('error loading worker', error);
            }
        }
        return this._client;
    }
    async getLanguageServiceWorker(...resources) {
        const client = await this._getClient();
        await this._worker.withSyncedResources(resources);
        return client;
    }
}
//# sourceMappingURL=workerManager.js.map