// src/index.ts
import { registerMarkerDataProvider } from "monaco-marker-data-provider";
import { createWorkerManager } from "monaco-worker-manager";

// src/languageFeatures.ts
import {
  fromCodeActionContext,
  fromPosition,
  fromRange,
  toCodeAction,
  toCompletionList,
  toDocumentSymbol,
  toFoldingRange,
  toHover,
  toLink,
  toLocationLink,
  toMarkerData,
  toTextEdit
} from "monaco-languageserver-types";
function createMarkerDataProvider(getWorker) {
  return {
    owner: "yaml",
    async provideMarkerData(model) {
      const worker = await getWorker(model.uri);
      const diagnostics = await worker.doValidation(String(model.uri));
      return diagnostics == null ? void 0 : diagnostics.map((diagnostic) => toMarkerData(diagnostic));
    },
    async doReset(model) {
      const worker = await getWorker(model.uri);
      await worker.resetSchema(String(model.uri));
    }
  };
}
function createCompletionItemProvider(getWorker) {
  return {
    triggerCharacters: [" ", ":"],
    async provideCompletionItems(model, position) {
      const resource = model.uri;
      const wordInfo = model.getWordUntilPosition(position);
      const worker = await getWorker(resource);
      const info = await worker.doComplete(String(resource), fromPosition(position));
      if (info) {
        return toCompletionList(info, {
          range: {
            startLineNumber: position.lineNumber,
            startColumn: wordInfo.startColumn,
            endLineNumber: position.lineNumber,
            endColumn: wordInfo.endColumn
          }
        });
      }
    }
  };
}
function createDefinitionProvider(getWorker) {
  return {
    async provideDefinition(model, position) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const locationLinks = await worker.doDefinition(String(resource), fromPosition(position));
      return locationLinks == null ? void 0 : locationLinks.map(toLocationLink);
    }
  };
}
function createHoverProvider(getWorker) {
  return {
    async provideHover(model, position) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const info = await worker.doHover(String(resource), fromPosition(position));
      if (info) {
        return toHover(info);
      }
    }
  };
}
function createDocumentSymbolProvider(getWorker) {
  return {
    displayName: "yaml",
    async provideDocumentSymbols(model) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const items = await worker.findDocumentSymbols(String(resource));
      return items == null ? void 0 : items.map(toDocumentSymbol);
    }
  };
}
function createDocumentFormattingEditProvider(getWorker) {
  return {
    displayName: "yaml",
    async provideDocumentFormattingEdits(model) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const edits = await worker.format(String(resource));
      return edits == null ? void 0 : edits.map(toTextEdit);
    }
  };
}
function createLinkProvider(getWorker) {
  return {
    async provideLinks(model) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const links = await worker.findLinks(String(resource));
      if (links) {
        return {
          links: links.map(toLink)
        };
      }
    }
  };
}
function createCodeActionProvider(getWorker) {
  return {
    async provideCodeActions(model, range, context) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const codeActions = await worker.getCodeAction(
        String(resource),
        fromRange(range),
        fromCodeActionContext(context)
      );
      if (codeActions) {
        return {
          actions: codeActions.map((codeAction) => toCodeAction(codeAction)),
          dispose() {
          }
        };
      }
    }
  };
}
function createFoldingRangeProvider(getWorker) {
  return {
    async provideFoldingRanges(model) {
      const resource = model.uri;
      const worker = await getWorker(resource);
      const foldingRanges = await worker.getFoldingRanges(String(resource));
      return foldingRanges == null ? void 0 : foldingRanges.map(toFoldingRange);
    }
  };
}

// src/index.ts
function configureMonacoYaml(monaco, options) {
  const createData = {
    completion: true,
    customTags: [],
    enableSchemaRequest: false,
    format: true,
    isKubernetes: false,
    hover: true,
    schemas: [],
    validate: true,
    yamlVersion: "1.2",
    ...options
  };
  monaco.languages.register({
    id: "yaml",
    extensions: [".yaml", ".yml"],
    aliases: ["YAML", "yaml", "YML", "yml"],
    mimetypes: ["application/x-yaml"]
  });
  const worker = createWorkerManager(monaco, {
    label: "yaml",
    moduleId: "monaco-yaml/yaml.worker",
    createData
  });
  let markerDataProvider = registerMarkerDataProvider(
    monaco,
    "yaml",
    createMarkerDataProvider(worker.getWorker)
  );
  const disposables = [
    worker,
    monaco.languages.registerCompletionItemProvider(
      "yaml",
      createCompletionItemProvider(worker.getWorker)
    ),
    monaco.languages.registerHoverProvider("yaml", createHoverProvider(worker.getWorker)),
    monaco.languages.registerDefinitionProvider("yaml", createDefinitionProvider(worker.getWorker)),
    monaco.languages.registerDocumentSymbolProvider(
      "yaml",
      createDocumentSymbolProvider(worker.getWorker)
    ),
    monaco.languages.registerDocumentFormattingEditProvider(
      "yaml",
      createDocumentFormattingEditProvider(worker.getWorker)
    ),
    monaco.languages.registerLinkProvider("yaml", createLinkProvider(worker.getWorker)),
    monaco.languages.registerCodeActionProvider("yaml", createCodeActionProvider(worker.getWorker)),
    monaco.languages.registerFoldingRangeProvider(
      "yaml",
      createFoldingRangeProvider(worker.getWorker)
    ),
    monaco.languages.setLanguageConfiguration("yaml", {
      comments: {
        lineComment: "#"
      },
      brackets: [
        ["{", "}"],
        ["[", "]"],
        ["(", ")"]
      ],
      autoClosingPairs: [
        { open: "{", close: "}" },
        { open: "[", close: "]" },
        { open: "(", close: ")" },
        { open: '"', close: '"' },
        { open: "'", close: "'" }
      ],
      surroundingPairs: [
        { open: "{", close: "}" },
        { open: "[", close: "]" },
        { open: "(", close: ")" },
        { open: '"', close: '"' },
        { open: "'", close: "'" }
      ],
      onEnterRules: [
        {
          beforeText: /:\s*$/,
          action: { indentAction: monaco.languages.IndentAction.Indent }
        }
      ]
    })
  ];
  return {
    dispose() {
      for (const disposable of disposables) {
        disposable.dispose();
      }
      markerDataProvider.dispose();
    },
    update(newOptions) {
      worker.updateCreateData(Object.assign(createData, newOptions));
      markerDataProvider.dispose();
      markerDataProvider = registerMarkerDataProvider(
        monaco,
        "yaml",
        createMarkerDataProvider(worker.getWorker)
      );
    }
  };
}
export {
  configureMonacoYaml
};
//# sourceMappingURL=index.js.map
