"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BackgroundColorCalculator_1 = require("./BackgroundColorCalculator");
const htmlHelpers_1 = require("./htmlHelpers");
exports.ROOT_NAME = 'All files';
const SEPARATOR = '/';
function flatMap(source, fn) {
    const result = [];
    source.map(fn).forEach(items => result.push(...items));
    return result;
}
exports.flatMap = flatMap;
function pathJoin(...parts) {
    return parts.reduce((prev, current) => prev.length ? current ? `${prev}/${current}` : prev : current, '');
}
exports.pathJoin = pathJoin;
function normalizeFileNames(input) {
    const fileNames = Object.keys(input);
    const commonBasePath = determineCommonBasePath(fileNames);
    const output = {};
    fileNames.forEach(fileName => {
        output[normalize(fileName.substr(commonBasePath.length))] = input[fileName];
    });
    return output;
}
exports.normalizeFileNames = normalizeFileNames;
function normalize(fileName) {
    return fileName.split(/\/|\\/)
        .filter(pathPart => pathPart)
        .join('/');
}
function determineCommonBasePath(fileNames) {
    const directories = fileNames.map(fileName => fileName.split(/\/|\\/).slice(0, -1));
    if (fileNames.length) {
        return directories.reduce(filterDirectories).join(SEPARATOR);
    }
    else {
        return '';
    }
    function filterDirectories(previousDirectories, currentDirectories) {
        for (let i = 0; i < previousDirectories.length; i++) {
            if (previousDirectories[i] !== currentDirectories[i]) {
                return previousDirectories.splice(0, i);
            }
        }
        return previousDirectories;
    }
}
exports.determineCommonBasePath = determineCommonBasePath;
/**
 * Walks over the code in this.model.source and adds the
 * `<mutation-test-report-mutant>` elements.
 * It also adds the background color using
 * `<span class="bg-danger-light">` and friends.
 */
function renderCode(model) {
    const backgroundState = new BackgroundColorCalculator_1.BackgroundColorCalculator();
    const startedMutants = [];
    const walker = (char, pos) => {
        const currentMutants = model.mutants.filter(m => eq(m.location.start, pos));
        const mutantsEnding = startedMutants.filter(m => gte(pos, m.location.end));
        mutantsEnding.forEach(mutant => startedMutants.splice(startedMutants.indexOf(mutant), 1));
        startedMutants.push(...currentMutants);
        const builder = [];
        if (currentMutants.length || mutantsEnding.length) {
            currentMutants.forEach(backgroundState.markMutantStart);
            mutantsEnding.forEach(backgroundState.markMutantEnd);
            // End previous color span
            builder.push('</span>');
            // End mutants
            mutantsEnding.forEach(() => builder.push('</mutation-test-report-mutant>'));
            // Start mutants
            currentMutants.forEach(mutant => builder.push(`<mutation-test-report-mutant mutant-id="${mutant.id}">`));
            // Start new color span
            builder.push(`<span class="bg-${backgroundState.determineBackground()}">`);
        }
        // Append the code character
        builder.push(htmlHelpers_1.escapeHtml(char));
        return builder.join('');
    };
    return `<span>${walkString(model.source, walker)}</span>`;
}
exports.renderCode = renderCode;
exports.COLUMN_START_INDEX = 1;
exports.LINE_START_INDEX = 1;
exports.NEW_LINE = '\n';
exports.CARRIAGE_RETURN = '\r';
function lines(content) {
    return content.split(exports.NEW_LINE)
        .map(line => line.endsWith(exports.CARRIAGE_RETURN) ? line.substr(0, line.length - 1) : line);
}
exports.lines = lines;
/**
 * Walks a string. Executes a function on each character of the string (except for new lines and carriage returns)
 * @param source the string to walk
 * @param fn The function to execute on each character of the string
 */
function walkString(source, fn) {
    let column = exports.COLUMN_START_INDEX;
    let line = exports.LINE_START_INDEX;
    const builder = [];
    for (const currentChar of source) {
        if (column === exports.COLUMN_START_INDEX && currentChar === exports.CARRIAGE_RETURN) {
            continue;
        }
        if (currentChar === exports.NEW_LINE) {
            line++;
            column = exports.COLUMN_START_INDEX;
            builder.push(exports.NEW_LINE);
            continue;
        }
        builder.push(fn(currentChar, { line, column: column++ }));
    }
    return builder.join('');
}
function gte(a, b) {
    return a.line > b.line || (a.line === b.line && a.column >= b.column);
}
function eq(a, b) {
    return a.line === b.line && a.column === b.column;
}
//# sourceMappingURL=helpers.js.map