"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.credsAuthenticator = exports.jwtAuthenticator = exports.nkeyAuthenticator = exports.noAuthFn = exports.buildAuthenticator = void 0;
/*
 * Copyright 2020 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const nkeys_1 = require("./nkeys");
const mod_1 = require("./mod");
const encoders_1 = require("./encoders");
function buildAuthenticator(opts) {
    // jwtAuthenticator is created by the user, since it
    // will require possibly reading files which
    // some of the clients are simply unable to do
    if (opts.authenticator) {
        return opts.authenticator;
    }
    if (opts.token) {
        return tokenFn(opts.token);
    }
    if (opts.user) {
        return passFn(opts.user, opts.pass);
    }
    return noAuthFn();
}
exports.buildAuthenticator = buildAuthenticator;
function noAuthFn() {
    return () => {
        return;
    };
}
exports.noAuthFn = noAuthFn;
/**
 * Returns a user/pass authenticator
 * @param { string }user
 * @param {string } pass
 * @return {UserPass}
 */
function passFn(user, pass) {
    return () => {
        return { user, pass };
    };
}
/**
 * Returns a token authenticator
 * @param {string } token
 * @return {TokenAuth}
 */
function tokenFn(token) {
    return () => {
        return { auth_token: token };
    };
}
/**
 * Returns an nkey authenticator that returns a public key
 * @param {Uint8Array | (() => Uint8Array)} seed
 * @return {NKeyAuth}
 */
function nkeyAuthenticator(seed) {
    return (nonce) => {
        seed = typeof seed === "function" ? seed() : seed;
        const kp = seed ? nkeys_1.nkeys.fromSeed(seed) : undefined;
        const nkey = kp ? kp.getPublicKey() : "";
        const challenge = encoders_1.TE.encode(nonce || "");
        const sigBytes = kp !== undefined && nonce ? kp.sign(challenge) : undefined;
        const sig = sigBytes ? nkeys_1.nkeys.encode(sigBytes) : "";
        return { nkey, sig };
    };
}
exports.nkeyAuthenticator = nkeyAuthenticator;
/**
 * Returns a jwt authenticator. If a seed is provided, the public
 * key, and signature are calculated. Note if a signature is provided
 * the returned value should be a base64 encoded string.
 *
 * @return {JwtAuth}
 * @param ajwt
 * @param seed
 */
function jwtAuthenticator(ajwt, seed) {
    return (nonce) => {
        const jwt = typeof ajwt === "function" ? ajwt() : ajwt;
        const fn = nkeyAuthenticator(seed);
        const { nkey, sig } = fn(nonce);
        return { jwt, nkey, sig };
    };
}
exports.jwtAuthenticator = jwtAuthenticator;
/**
 * Returns a jwt authenticator configured from the specified creds file contents.
 * @param creds
 * @returns {JwtAuth}
 */
function credsAuthenticator(creds) {
    const CREDS = /\s*(?:(?:[-]{3,}[^\n]*[-]{3,}\n)(.+)(?:\n\s*[-]{3,}[^\n]*[-]{3,}\n))/ig;
    const s = encoders_1.TD.decode(creds);
    // get the JWT
    let m = CREDS.exec(s);
    if (!m) {
        throw mod_1.NatsError.errorForCode(mod_1.ErrorCode.BadCreds);
    }
    const jwt = m[1].trim();
    // get the nkey
    m = CREDS.exec(s);
    if (!m) {
        throw mod_1.NatsError.errorForCode(mod_1.ErrorCode.BadCreds);
    }
    const seed = encoders_1.TE.encode(m[1].trim());
    return jwtAuthenticator(jwt, seed);
}
exports.credsAuthenticator = credsAuthenticator;
//# sourceMappingURL=authenticator.js.map