"use strict";
/*
 * Copyright 2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JetStreamClientImpl = void 0;
const types_1 = require("./types");
const jsbaseclient_api_1 = require("./jsbaseclient_api");
const jsutil_1 = require("./jsutil");
const jsconsumer_api_1 = require("./jsconsumer_api");
const jsmsg_1 = require("./jsmsg");
const typedsub_1 = require("./typedsub");
const error_1 = require("./error");
const queued_iterator_1 = require("./queued_iterator");
const util_1 = require("./util");
const protocol_1 = require("./protocol");
const headers_1 = require("./headers");
const jsconsumeropts_1 = require("./jsconsumeropts");
var PubHeaders;
(function (PubHeaders) {
    PubHeaders["MsgIdHdr"] = "Nats-Msg-Id";
    PubHeaders["ExpectedStreamHdr"] = "Nats-Expected-Stream";
    PubHeaders["ExpectedLastSeqHdr"] = "Nats-Expected-Last-Sequence";
    PubHeaders["ExpectedLastMsgIdHdr"] = "Nats-Expected-Last-Msg-Id";
})(PubHeaders || (PubHeaders = {}));
class JetStreamClientImpl extends jsbaseclient_api_1.BaseApiClient {
    constructor(nc, opts) {
        super(nc, opts);
        this.api = new jsconsumer_api_1.ConsumerAPIImpl(nc, opts);
    }
    publish(subj, data = types_1.Empty, opts) {
        return __awaiter(this, void 0, void 0, function* () {
            opts = opts || {};
            opts.expect = opts.expect || {};
            const mh = (opts === null || opts === void 0 ? void 0 : opts.headers) || headers_1.headers();
            if (opts) {
                if (opts.msgID) {
                    mh.set(PubHeaders.MsgIdHdr, opts.msgID);
                }
                if (opts.expect.lastMsgID) {
                    mh.set(PubHeaders.ExpectedLastMsgIdHdr, opts.expect.lastMsgID);
                }
                if (opts.expect.streamName) {
                    mh.set(PubHeaders.ExpectedStreamHdr, opts.expect.streamName);
                }
                if (opts.expect.lastSequence) {
                    mh.set(PubHeaders.ExpectedLastSeqHdr, `${opts.expect.lastSequence}`);
                }
            }
            const to = opts.timeout || this.timeout;
            const ro = {};
            if (to) {
                ro.timeout = to;
            }
            if (opts) {
                ro.headers = mh;
            }
            const r = yield this.nc.request(subj, data, ro);
            const pa = this.parseJsResponse(r);
            if (pa.stream === "") {
                throw error_1.NatsError.errorForCode(error_1.ErrorCode.JetStreamInvalidAck);
            }
            pa.duplicate = pa.duplicate ? pa.duplicate : false;
            return pa;
        });
    }
    pull(stream, durable) {
        return __awaiter(this, void 0, void 0, function* () {
            jsutil_1.validateStreamName(stream);
            jsutil_1.validateDurableName(durable);
            const msg = yield this.nc.request(
            // FIXME: specify expires
            `${this.prefix}.CONSUMER.MSG.NEXT.${stream}.${durable}`, this.jc.encode({ no_wait: true, batch: 1, expires: jsutil_1.nanos(this.timeout) }), { noMux: true, timeout: this.timeout });
            const err = jsutil_1.checkJsError(msg);
            if (err) {
                throw (err);
            }
            return jsmsg_1.toJsMsg(msg);
        });
    }
    /*
    * Returns available messages upto specified batch count.
    * If expires is set the iterator will wait for the specified
    * amount of millis before closing the subscription.
    * If no_wait is specified, the iterator will return no messages.
    * @param stream
    * @param durable
    * @param opts
    */
    fetch(stream, durable, opts = {}) {
        jsutil_1.validateStreamName(stream);
        jsutil_1.validateDurableName(durable);
        let timer = null;
        const args = {};
        args.batch = opts.batch || 1;
        args.no_wait = opts.no_wait || false;
        const expires = opts.expires || 0;
        if (expires) {
            args.expires = jsutil_1.nanos(expires);
        }
        if (expires === 0 && args.no_wait === false) {
            throw new Error("expires or no_wait is required");
        }
        const qi = new queued_iterator_1.QueuedIteratorImpl();
        const wants = args.batch;
        let received = 0;
        qi.dispatchedFn = (m) => {
            if (m) {
                received++;
                if (timer && m.info.pending === 0) {
                    // the expiration will close it
                    return;
                }
                // if we have one pending and we got the expected
                // or there are no more stop the iterator
                if (qi.getPending() === 1 && m.info.pending === 0 || wants === received) {
                    qi.stop();
                }
            }
        };
        const inbox = protocol_1.createInbox(this.nc.options.inboxPrefix);
        const sub = this.nc.subscribe(inbox, {
            max: opts.batch,
            callback: (err, msg) => {
                if (err === null) {
                    err = jsutil_1.checkJsError(msg);
                }
                if (err !== null) {
                    if (timer) {
                        timer.cancel();
                        timer = null;
                    }
                    if (error_1.isNatsError(err) && err.code === error_1.ErrorCode.JetStream404NoMessages) {
                        qi.stop();
                    }
                    else {
                        qi.stop(err);
                    }
                }
                else {
                    qi.received++;
                    qi.push(jsmsg_1.toJsMsg(msg));
                }
            },
        });
        // timer on the client  the issue is that the request
        // is started on the client, which means that it will expire
        // on the client first
        if (expires) {
            timer = util_1.timeout(expires);
            timer.catch(() => {
                if (!sub.isClosed()) {
                    sub.drain();
                    timer = null;
                }
            });
        }
        (() => __awaiter(this, void 0, void 0, function* () {
            // close the iterator if the connection or subscription closes unexpectedly
            yield sub.closed;
            if (timer !== null) {
                timer.cancel();
                timer = null;
            }
            qi.stop();
        }))().catch();
        this.nc.publish(`${this.prefix}.CONSUMER.MSG.NEXT.${stream}.${durable}`, this.jc.encode(args), { reply: inbox });
        return qi;
    }
    pullSubscribe(subject, opts = jsconsumeropts_1.consumerOpts()) {
        return __awaiter(this, void 0, void 0, function* () {
            const cso = yield this._processOptions(subject, opts);
            if (!cso.attached) {
                cso.config.filter_subject = subject;
            }
            if (cso.config.deliver_subject) {
                throw new Error("consumer info specifies deliver_subject - pull consumers cannot have deliver_subject set");
            }
            const ackPolicy = cso.config.ack_policy;
            if (ackPolicy === types_1.AckPolicy.None || ackPolicy === types_1.AckPolicy.All) {
                throw new Error("ack policy for pull consumers must be explicit");
            }
            const so = this._buildTypedSubscriptionOpts(cso);
            const sub = new JetStreamPullSubscriptionImpl(this, cso.deliver, so);
            try {
                yield this._maybeCreateConsumer(cso);
            }
            catch (err) {
                sub.unsubscribe();
                throw err;
            }
            sub.info = cso;
            return sub;
        });
    }
    subscribe(subject, opts = jsconsumeropts_1.consumerOpts()) {
        return __awaiter(this, void 0, void 0, function* () {
            const cso = yield this._processOptions(subject, opts);
            // this effectively requires deliver subject to be specified
            // as an option otherwise we have a pull consumer
            if (!cso.config.deliver_subject) {
                throw new Error("consumer info specifies a pull consumer - deliver_subject is required");
            }
            const so = this._buildTypedSubscriptionOpts(cso);
            const sub = new JetStreamSubscriptionImpl(this, cso.deliver, so);
            try {
                yield this._maybeCreateConsumer(cso);
            }
            catch (err) {
                sub.unsubscribe();
                throw err;
            }
            sub.info = cso;
            return sub;
        });
    }
    _processOptions(subject, opts = jsconsumeropts_1.consumerOpts()) {
        return __awaiter(this, void 0, void 0, function* () {
            const jsi = (jsconsumeropts_1.isConsumerOptsBuilder(opts)
                ? opts.getOpts()
                : opts);
            jsi.api = this;
            jsi.config = jsi.config || {};
            jsi.stream = jsi.stream ? jsi.stream : yield this.findStream(subject);
            jsi.attached = false;
            if (jsi.config.durable_name) {
                try {
                    const info = yield this.api.info(jsi.stream, jsi.config.durable_name);
                    if (info) {
                        if (info.config.filter_subject && info.config.filter_subject !== subject) {
                            throw new Error("subject does not match consumer");
                        }
                        jsi.config = info.config;
                        jsi.attached = true;
                    }
                }
                catch (err) {
                    //consumer doesn't exist
                    if (err.code !== "404") {
                        throw err;
                    }
                }
            }
            if (!jsi.attached) {
                jsi.config.filter_subject = subject;
                // jsi.config.deliver_subject = jsi.config.deliver_subject ??
                //   createInbox(this.nc.options.inboxPrefix);
            }
            jsi.deliver = jsi.config.deliver_subject ||
                protocol_1.createInbox(this.nc.options.inboxPrefix);
            return jsi;
        });
    }
    _buildTypedSubscriptionOpts(jsi) {
        const so = {};
        so.adapter = msgAdapter(jsi.callbackFn === undefined);
        if (jsi.callbackFn) {
            so.callback = jsi.callbackFn;
        }
        if (!jsi.mack) {
            so.dispatchedFn = autoAckJsMsg;
        }
        so.max = jsi.max || 0;
        return so;
    }
    _maybeCreateConsumer(jsi) {
        return __awaiter(this, void 0, void 0, function* () {
            if (jsi.attached) {
                return;
            }
            jsi.config = Object.assign({
                deliver_policy: types_1.DeliverPolicy.All,
                ack_policy: types_1.AckPolicy.Explicit,
                ack_wait: jsutil_1.nanos(30 * 1000),
                replay_policy: types_1.ReplayPolicy.Instant,
            }, jsi.config);
            const ci = yield this.api.add(jsi.stream, jsi.config);
            jsi.name = ci.name;
            jsi.config = ci.config;
        });
    }
}
exports.JetStreamClientImpl = JetStreamClientImpl;
class JetStreamSubscriptionImpl extends typedsub_1.TypedSubscription {
    constructor(js, subject, opts) {
        super(js.nc, subject, opts);
    }
    set info(info) {
        this.sub.info = info;
    }
    get info() {
        return this.sub.info;
    }
    destroy() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.isClosed()) {
                yield this.drain();
            }
            const jinfo = this.sub.info;
            const name = jinfo.config.durable_name || jinfo.name;
            const subj = `${jinfo.api.prefix}.CONSUMER.DELETE.${jinfo.stream}.${name}`;
            yield jinfo.api._request(subj);
        });
    }
    consumerInfo() {
        return __awaiter(this, void 0, void 0, function* () {
            const jinfo = this.sub.info;
            const name = jinfo.config.durable_name || jinfo.name;
            const subj = `${jinfo.api.prefix}.CONSUMER.INFO.${jinfo.stream}.${name}`;
            return yield jinfo.api._request(subj);
        });
    }
}
class JetStreamPullSubscriptionImpl extends JetStreamSubscriptionImpl {
    constructor(js, subject, opts) {
        super(js, subject, opts);
    }
    pull(opts = { batch: 1 }) {
        const { stream, config } = this.sub.info;
        const consumer = config.durable_name;
        const args = {};
        args.batch = opts.batch || 1;
        args.no_wait = opts.no_wait || false;
        // FIXME: this is nanos
        if (opts.expires && opts.expires > 0) {
            args.expires = opts.expires;
        }
        if (this.info) {
            const api = this.info.api;
            const subj = `${api.prefix}.CONSUMER.MSG.NEXT.${stream}.${consumer}`;
            const reply = this.sub.subject;
            api.nc.publish(subj, api.jc.encode(args), { reply: reply });
        }
    }
}
function msgAdapter(iterator) {
    if (iterator) {
        return iterMsgAdapter;
    }
    else {
        return cbMsgAdapter;
    }
}
function cbMsgAdapter(err, msg) {
    if (err) {
        return [err, null];
    }
    err = jsutil_1.checkJsError(msg);
    if (err) {
        return [err, null];
    }
    if (jsutil_1.isFlowControlMsg(msg)) {
        msg.respond();
        return [null, null];
    }
    const jm = jsmsg_1.toJsMsg(msg);
    try {
        // this will throw if not a JsMsg
        jm.info;
        return [null, jm];
    }
    catch (err) {
        return [err, null];
    }
}
function iterMsgAdapter(err, msg) {
    if (err) {
        return [err, null];
    }
    // iterator will close if we have an error
    // check for errors that shouldn't close it
    const ne = jsutil_1.checkJsError(msg);
    if (ne !== null) {
        switch (ne.code) {
            case error_1.ErrorCode.JetStream404NoMessages:
            case error_1.ErrorCode.JetStream408RequestTimeout:
            case error_1.ErrorCode.JetStream409MaxAckPendingExceeded:
                return [null, null];
            default:
                return [ne, null];
        }
    }
    if (jsutil_1.isFlowControlMsg(msg)) {
        msg.respond();
        return [null, null];
    }
    const jm = jsmsg_1.toJsMsg(msg);
    try {
        // this will throw if not a JsMsg
        jm.info;
        return [null, jm];
    }
    catch (err) {
        return [err, null];
    }
}
function autoAckJsMsg(data) {
    if (data) {
        data.ack();
    }
}
//# sourceMappingURL=jsclient.js.map