"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkJsErrorCode = exports.checkJsError = exports.isFlowControlMsg = exports.millis = exports.nanos = exports.defaultConsumer = exports.validateName = exports.validateStreamName = exports.validateDurableName = void 0;
/*
 * Copyright 2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const types_1 = require("./types");
const error_1 = require("./error");
function validateDurableName(name) {
    return validateName("durable", name);
}
exports.validateDurableName = validateDurableName;
function validateStreamName(name) {
    return validateName("stream", name);
}
exports.validateStreamName = validateStreamName;
function validateName(context, name = "") {
    if (name === "") {
        throw Error(`${context} name required`);
    }
    const bad = [".", "*", ">"];
    bad.forEach((v) => {
        if (name.indexOf(v) !== -1) {
            throw Error(`invalid ${context} name - ${context} name cannot contain '${v}'`);
        }
    });
}
exports.validateName = validateName;
function defaultConsumer(name, opts = {}) {
    return Object.assign({
        name: name,
        deliver_policy: types_1.DeliverPolicy.All,
        ack_policy: types_1.AckPolicy.Explicit,
        ack_wait: nanos(30 * 1000),
        replay_policy: types_1.ReplayPolicy.Instant,
    }, opts);
}
exports.defaultConsumer = defaultConsumer;
function nanos(millis) {
    return millis * 1000000;
}
exports.nanos = nanos;
function millis(ns) {
    return ns / 1000000;
}
exports.millis = millis;
function isFlowControlMsg(msg) {
    const h = msg.headers;
    if (!h) {
        return false;
    }
    return h.code >= 100 && h.code < 200;
}
exports.isFlowControlMsg = isFlowControlMsg;
function checkJsError(msg) {
    const h = msg.headers;
    if (!h) {
        return null;
    }
    return checkJsErrorCode(h.code, h.status);
}
exports.checkJsError = checkJsError;
function checkJsErrorCode(code, description = "") {
    if (code < 300) {
        return null;
    }
    description = description.toLowerCase();
    switch (code) {
        case 503:
            return error_1.NatsError.errorForCode(error_1.ErrorCode.JetStreamNotEnabled, new Error(description));
        default:
            if (description === "") {
                description = error_1.ErrorCode.Unknown;
            }
            return new error_1.NatsError(description, `${code}`);
    }
}
exports.checkJsErrorCode = checkJsErrorCode;
//# sourceMappingURL=jsutil.js.map