"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MsgImpl = exports.isRequestError = void 0;
/*
 * Copyright 2020 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const types_1 = require("./types");
const headers_1 = require("./headers");
const encoders_1 = require("./encoders");
const error_1 = require("./error");
function isRequestError(msg) {
    if (msg && msg.headers) {
        const headers = msg.headers;
        if (headers.hasError) {
            if (headers.code === 503) {
                return error_1.NatsError.errorForCode(error_1.ErrorCode.NoResponders);
            }
            else {
                let desc = headers.description;
                if (desc === "") {
                    desc = error_1.ErrorCode.RequestError;
                }
                desc = desc.toLowerCase();
                return new error_1.NatsError(desc, headers.status);
            }
        }
    }
    return null;
}
exports.isRequestError = isRequestError;
class MsgImpl {
    constructor(msg, data, publisher) {
        this._msg = msg;
        this._rdata = data;
        this.publisher = publisher;
    }
    get subject() {
        if (this._subject) {
            return this._subject;
        }
        this._subject = encoders_1.TD.decode(this._msg.subject);
        return this._subject;
    }
    get reply() {
        if (this._reply) {
            return this._reply;
        }
        this._reply = encoders_1.TD.decode(this._msg.reply);
        return this._reply;
    }
    get sid() {
        return this._msg.sid;
    }
    get headers() {
        if (this._msg.hdr > -1 && !this._headers) {
            const buf = this._rdata.subarray(0, this._msg.hdr);
            this._headers = headers_1.MsgHdrsImpl.decode(buf);
        }
        return this._headers;
    }
    get data() {
        if (!this._rdata) {
            return new Uint8Array(0);
        }
        return this._msg.hdr > -1
            ? this._rdata.subarray(this._msg.hdr)
            : this._rdata;
    }
    // eslint-ignore-next-line @typescript-eslint/no-explicit-any
    respond(data = types_1.Empty, opts) {
        if (this.reply) {
            this.publisher.publish(this.reply, data, opts);
            return true;
        }
        return false;
    }
}
exports.MsgImpl = MsgImpl;
//# sourceMappingURL=msg.js.map