"use strict";
/*
 * Copyright 2018-2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatsConnectionImpl = void 0;
const util_1 = require("./util");
const protocol_1 = require("./protocol");
const subscription_1 = require("./subscription");
const error_1 = require("./error");
const types_1 = require("./types");
const options_1 = require("./options");
const queued_iterator_1 = require("./queued_iterator");
const request_1 = require("./request");
const msg_1 = require("./msg");
const jsm_1 = require("./jsm");
const jsclient_1 = require("./jsclient");
class NatsConnectionImpl {
    constructor(opts) {
        this.draining = false;
        this.options = options_1.parseOptions(opts);
        this.listeners = [];
    }
    static connect(opts = {}) {
        return new Promise((resolve, reject) => {
            const nc = new NatsConnectionImpl(opts);
            protocol_1.ProtocolHandler.connect(nc.options, nc)
                .then((ph) => {
                nc.protocol = ph;
                (function () {
                    var e_1, _a;
                    return __awaiter(this, void 0, void 0, function* () {
                        try {
                            for (var _b = __asyncValues(ph.status()), _c; _c = yield _b.next(), !_c.done;) {
                                const s = _c.value;
                                nc.listeners.forEach((l) => {
                                    l.push(s);
                                });
                            }
                        }
                        catch (e_1_1) { e_1 = { error: e_1_1 }; }
                        finally {
                            try {
                                if (_c && !_c.done && (_a = _b.return)) yield _a.call(_b);
                            }
                            finally { if (e_1) throw e_1.error; }
                        }
                    });
                })();
                resolve(nc);
            })
                .catch((err) => {
                reject(err);
            });
        });
    }
    closed() {
        return this.protocol.closed;
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.protocol.close();
        });
    }
    publish(subject, data = types_1.Empty, options) {
        subject = subject || "";
        if (subject.length === 0) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.BadSubject);
        }
        // if argument is not undefined/null and not a Uint8Array, toss
        if (data && !util_1.isUint8Array(data)) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.BadPayload);
        }
        this.protocol.publish(subject, data, options);
    }
    subscribe(subject, opts = {}) {
        if (this.isClosed()) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionClosed);
        }
        if (this.isDraining()) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionDraining);
        }
        subject = subject || "";
        if (subject.length === 0) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.BadSubject);
        }
        const sub = new subscription_1.SubscriptionImpl(this.protocol, subject, opts);
        this.protocol.subscribe(sub);
        return sub;
    }
    request(subject, data = types_1.Empty, opts = { timeout: 1000, noMux: false }) {
        if (this.isClosed()) {
            return Promise.reject(error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionClosed));
        }
        if (this.isDraining()) {
            return Promise.reject(error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionDraining));
        }
        subject = subject || "";
        if (subject.length === 0) {
            return Promise.reject(error_1.NatsError.errorForCode(error_1.ErrorCode.BadSubject));
        }
        opts.timeout = opts.timeout || 1000;
        if (opts.timeout < 1) {
            return Promise.reject(new error_1.NatsError("timeout", error_1.ErrorCode.InvalidOption));
        }
        if (!opts.noMux && opts.reply) {
            return Promise.reject(new error_1.NatsError("reply can only be used with noMux", error_1.ErrorCode.InvalidOption));
        }
        if (opts.noMux) {
            const inbox = opts.reply
                ? opts.reply
                : protocol_1.createInbox(this.options.inboxPrefix);
            const d = util_1.deferred();
            this.subscribe(inbox, {
                max: 1,
                timeout: opts.timeout,
                callback: (err, msg) => {
                    if (err) {
                        d.reject(err);
                    }
                    else {
                        err = msg_1.isRequestError(msg);
                        if (err) {
                            d.reject(err);
                        }
                        else {
                            d.resolve(msg);
                        }
                    }
                },
            });
            this.publish(subject, data, { reply: inbox });
            return d;
        }
        else {
            const r = new request_1.Request(this.protocol.muxSubscriptions, opts);
            this.protocol.request(r);
            try {
                this.publish(subject, data, {
                    reply: `${this.protocol.muxSubscriptions.baseInbox}${r.token}`,
                    headers: opts.headers,
                });
            }
            catch (err) {
                r.cancel(err);
            }
            const p = Promise.race([r.timer, r.deferred]);
            p.catch(() => {
                r.cancel();
            });
            return p;
        }
    }
    /***
       * Flushes to the server. Promise resolves when round-trip completes.
       * @returns {Promise<void>}
       */
    flush() {
        return this.protocol.flush();
    }
    drain() {
        if (this.isClosed()) {
            return Promise.reject(error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionClosed));
        }
        if (this.isDraining()) {
            return Promise.reject(error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionDraining));
        }
        this.draining = true;
        return this.protocol.drain();
    }
    isClosed() {
        return this.protocol.isClosed();
    }
    isDraining() {
        return this.draining;
    }
    getServer() {
        const srv = this.protocol.getServer();
        return srv ? srv.listen : "";
    }
    status() {
        const iter = new queued_iterator_1.QueuedIteratorImpl();
        this.listeners.push(iter);
        return iter;
    }
    get info() {
        return this.protocol.isClosed() ? undefined : this.protocol.info;
    }
    stats() {
        return {
            inBytes: this.protocol.inBytes,
            outBytes: this.protocol.outBytes,
            inMsgs: this.protocol.inMsgs,
            outMsgs: this.protocol.outMsgs,
        };
    }
    jetstreamManager(opts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            jetstreamPreview(this);
            const adm = new jsm_1.JetStreamManagerImpl(this, opts);
            try {
                yield adm.getAccountInfo();
            }
            catch (err) {
                const ne = err;
                if (ne.code === error_1.ErrorCode.NoResponders) {
                    throw error_1.NatsError.errorForCode(error_1.ErrorCode.JetStreamNotEnabled);
                }
                throw ne;
            }
            return adm;
        });
    }
    jetstream(opts = {}) {
        jetstreamPreview(this);
        return new jsclient_1.JetStreamClientImpl(this, opts);
    }
}
exports.NatsConnectionImpl = NatsConnectionImpl;
const jetstreamPreview = (() => {
    let once = false;
    return (nci) => {
        var _a;
        if (!once) {
            once = true;
            const { lang } = (_a = nci === null || nci === void 0 ? void 0 : nci.protocol) === null || _a === void 0 ? void 0 : _a.transport;
            if (lang) {
                console.log(`\u001B[33m >> jetstream functionality in ${lang} is preview functionality \u001B[0m`);
            }
            else {
                console.log(`\u001B[33m >> jetstream functionality is preview functionality \u001B[0m`);
            }
        }
    };
})();
//# sourceMappingURL=nats.js.map