"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servers = exports.ServerImpl = void 0;
/*
 * Copyright 2018-2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
const types_1 = require("./types");
const transport_1 = require("./transport");
const util_1 = require("./util");
const ipparser_1 = require("./ipparser");
/**
 * @hidden
 */
class ServerImpl {
    constructor(u, gossiped = false) {
        this.src = u;
        this.tlsName = "";
        // remove any protocol that may have been provided
        if (u.match(/^(.*:\/\/)(.*)/m)) {
            u = u.replace(/^(.*:\/\/)(.*)/gm, "$2");
        }
        // in web environments, URL may not be a living standard
        // that means that protocols other than HTTP/S are not
        // parsable correctly.
        const url = new URL(`http://${u}`);
        if (!url.port) {
            url.port = `${types_1.DEFAULT_PORT}`;
        }
        this.listen = url.host;
        this.hostname = url.hostname;
        this.port = parseInt(url.port, 10);
        this.didConnect = false;
        this.reconnects = 0;
        this.lastConnect = 0;
        this.gossiped = gossiped;
    }
    toString() {
        return this.listen;
    }
}
exports.ServerImpl = ServerImpl;
/**
 * @hidden
 */
class Servers {
    constructor(randomize, listens = []) {
        this.firstSelect = true;
        this.servers = [];
        this.tlsName = "";
        const urlParseFn = transport_1.getUrlParseFn();
        if (listens) {
            listens.forEach((hp) => {
                hp = urlParseFn ? urlParseFn(hp) : hp;
                this.servers.push(new ServerImpl(hp));
            });
            if (randomize) {
                this.servers = util_1.shuffle(this.servers);
            }
        }
        if (this.servers.length === 0) {
            this.addServer(`${types_1.DEFAULT_HOST}:${transport_1.defaultPort()}`, false);
        }
        this.currentServer = this.servers[0];
    }
    updateTLSName() {
        const cs = this.getCurrentServer();
        if (!ipparser_1.isIP(cs.hostname)) {
            this.tlsName = cs.hostname;
            this.servers.forEach((s) => {
                if (s.gossiped) {
                    s.tlsName = this.tlsName;
                }
            });
        }
    }
    getCurrentServer() {
        return this.currentServer;
    }
    addServer(u, implicit = false) {
        const urlParseFn = transport_1.getUrlParseFn();
        u = urlParseFn ? urlParseFn(u) : u;
        const s = new ServerImpl(u, implicit);
        if (ipparser_1.isIP(s.hostname)) {
            s.tlsName = this.tlsName;
        }
        this.servers.push(s);
    }
    selectServer() {
        // allow using select without breaking the order of the servers
        if (this.firstSelect) {
            this.firstSelect = false;
            return this.currentServer;
        }
        const t = this.servers.shift();
        if (t) {
            this.servers.push(t);
            this.currentServer = t;
        }
        return t;
    }
    removeCurrentServer() {
        this.removeServer(this.currentServer);
    }
    removeServer(server) {
        if (server) {
            const index = this.servers.indexOf(server);
            this.servers.splice(index, 1);
        }
    }
    length() {
        return this.servers.length;
    }
    next() {
        return this.servers.length ? this.servers[0] : undefined;
    }
    getServers() {
        return this.servers;
    }
    update(info) {
        const added = [];
        let deleted = [];
        const urlParseFn = transport_1.getUrlParseFn();
        const discovered = new Map();
        if (info.connect_urls && info.connect_urls.length > 0) {
            info.connect_urls.forEach((hp) => {
                hp = urlParseFn ? urlParseFn(hp) : hp;
                const s = new ServerImpl(hp, true);
                discovered.set(hp, s);
            });
        }
        // remove gossiped servers that are no longer reported
        const toDelete = [];
        this.servers.forEach((s, index) => {
            const u = s.listen;
            if (s.gossiped && this.currentServer.listen !== u &&
                discovered.get(u) === undefined) {
                // server was removed
                toDelete.push(index);
            }
            // remove this entry from reported
            discovered.delete(u);
        });
        // perform the deletion
        toDelete.reverse();
        toDelete.forEach((index) => {
            const removed = this.servers.splice(index, 1);
            deleted = deleted.concat(removed[0].listen);
        });
        // remaining servers are new
        discovered.forEach((v, k) => {
            this.servers.push(v);
            added.push(k);
        });
        return { added, deleted };
    }
}
exports.Servers = Servers;
//# sourceMappingURL=servers.js.map