"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscriptionImpl = void 0;
/*
 * Copyright 2020-2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const queued_iterator_1 = require("./queued_iterator");
const util_1 = require("./util");
const error_1 = require("./error");
class SubscriptionImpl extends queued_iterator_1.QueuedIteratorImpl {
    constructor(protocol, subject, opts = {}) {
        super();
        util_1.extend(this, opts);
        this.protocol = protocol;
        this.subject = subject;
        this.draining = false;
        this.noIterator = typeof opts.callback === "function";
        this.closed = util_1.deferred();
        if (opts.timeout) {
            this.timer = util_1.timeout(opts.timeout);
            this.timer
                .then(() => {
                // timer was cancelled
                this.timer = undefined;
            })
                .catch((err) => {
                // timer fired
                this.stop(err);
                if (this.noIterator) {
                    this.callback(err, {});
                }
            });
        }
    }
    setDispatchedFn(cb) {
        // user specified callback
        if (this.noIterator) {
            const uc = this.callback;
            this.callback = (err, msg) => {
                uc(err, msg);
                cb(msg);
            };
        }
        else {
            this.dispatchedFn = cb;
        }
    }
    callback(err, msg) {
        this.cancelTimeout();
        err ? this.stop(err) : this.push(msg);
    }
    close() {
        if (!this.isClosed()) {
            this.cancelTimeout();
            this.stop();
            if (this.cleanupFn) {
                try {
                    this.cleanupFn(this, this.info);
                }
                catch (_err) {
                    // ignoring
                }
            }
            this.closed.resolve();
        }
    }
    unsubscribe(max) {
        this.protocol.unsubscribe(this, max);
    }
    cancelTimeout() {
        if (this.timer) {
            this.timer.cancel();
            this.timer = undefined;
        }
    }
    drain() {
        if (this.protocol.isClosed()) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.ConnectionClosed);
        }
        if (this.isClosed()) {
            throw error_1.NatsError.errorForCode(error_1.ErrorCode.SubClosed);
        }
        if (!this.drained) {
            this.protocol.unsub(this);
            this.drained = this.protocol.flush(util_1.deferred());
            this.drained.then(() => {
                this.protocol.subscriptions.cancel(this);
            });
        }
        return this.drained;
    }
    isDraining() {
        return this.draining;
    }
    isClosed() {
        return this.done;
    }
    getSubject() {
        return this.subject;
    }
    getMax() {
        return this.max;
    }
    getID() {
        return this.sid;
    }
}
exports.SubscriptionImpl = SubscriptionImpl;
//# sourceMappingURL=subscription.js.map