/**
 * Copyright (c) 2002-2016 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _createClass = (function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; })();

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var _log = require("./log");

var _buf = require("./buf");

var _utf8 = require("./utf8");

var _utf82 = _interopRequireDefault(_utf8);

var _integer = require("../integer");

var _error = require('./../error');

var MAX_CHUNK_SIZE = 16383,
    TINY_STRING = 0x80,
    TINY_LIST = 0x90,
    TINY_MAP = 0xA0,
    TINY_STRUCT = 0xB0,
    NULL = 0xC0,
    FLOAT_64 = 0xC1,
    FALSE = 0xC2,
    TRUE = 0xC3,
    INT_8 = 0xC8,
    INT_16 = 0xC9,
    INT_32 = 0xCA,
    INT_64 = 0xCB,
    STRING_8 = 0xD0,
    STRING_16 = 0xD1,
    STRING_32 = 0xD2,
    LIST_8 = 0xD4,
    LIST_16 = 0xD5,
    LIST_32 = 0xD6,
    MAP_8 = 0xD8,
    MAP_16 = 0xD9,
    MAP_32 = 0xDA,
    STRUCT_8 = 0xDC,
    STRUCT_16 = 0xDD;

/**
  * A Structure have a signature and fields.
  * @access private
  */

var Structure = (function () {
  /**
   * Create new instance
   */

  function Structure(signature, fields) {
    _classCallCheck(this, Structure);

    this.signature = signature;
    this.fields = fields;
  }

  /**
    * Class to pack
    * @access private
    */

  _createClass(Structure, [{
    key: "toString",
    value: function toString() {
      var fieldStr = "";
      for (var i = 0; i < this.fields.length; i++) {
        if (i > 0) {
          fieldStr += ", ";
        }
        fieldStr += this.fields[i];
      }
      return "Structure(" + this.signature + ", [" + this.fields + "])";
    }
  }]);

  return Structure;
})();

var Packer = (function () {
  function Packer(channel) {
    _classCallCheck(this, Packer);

    this._ch = channel;
  }

  /**
    * Class to unpack
    * @access private
    */

  _createClass(Packer, [{
    key: "pack",
    value: function pack(x) {
      if (x === null) {
        this._ch.writeUInt8(NULL);
      } else if (x === true) {
        this._ch.writeUInt8(TRUE);
      } else if (x === false) {
        this._ch.writeUInt8(FALSE);
      } else if (typeof x == "number") {
        this.packFloat(x);
      } else if (typeof x == "string") {
        this.packString(x);
      } else if (x instanceof _integer.Integer) {
        this.packInteger(x);
      } else if (x instanceof Array) {
        this.packListHeader(x.length);
        for (var i = 0; i < x.length; i++) {
          this.pack(x[i]);
        }
      } else if (x instanceof Structure) {
        this.packStruct(x.signature, x.fields);
      } else if (typeof x == "object") {
        var keys = Object.keys(x);
        this.packMapHeader(keys.length);
        for (var i = 0; i < keys.length; i++) {
          var key = keys[i];
          this.packString(key);
          this.pack(x[key]);
        }
      } else {
        throw (0, _error.newError)("Cannot pack this value: " + x);
      }
    }
  }, {
    key: "packStruct",
    value: function packStruct(signature, fields) {
      fields = fields || [];
      this.packStructHeader(fields.length, signature);
      for (var i = 0; i < fields.length; i++) {
        this.pack(fields[i]);
      }
    }
  }, {
    key: "packInteger",
    value: function packInteger(x) {
      var high = x.high,
          low = x.low;

      if (x.greaterThanOrEqual(-0x10) && x.lessThan(0x80)) {
        this._ch.writeInt8(low);
      } else if (x.greaterThanOrEqual(-0x80) && x.lessThan(-0x10)) {
        this._ch.writeUInt8(INT_8);
        this._ch.writeInt8(low);
      } else if (x.greaterThanOrEqual(-0x8000) && x.lessThan(0x8000)) {
        this._ch.writeUInt8(INT_16);
        this._ch.writeInt16(low);
      } else if (x.greaterThanOrEqual(-0x80000000) && x.lessThan(0x80000000)) {
        this._ch.writeUInt8(INT_32);
        this._ch.writeInt32(low);
      } else {
        this._ch.writeUInt8(INT_64);
        this._ch.writeInt32(high);
        this._ch.writeInt32(low);
      }
    }
  }, {
    key: "packFloat",
    value: function packFloat(x) {
      this._ch.writeUInt8(FLOAT_64);
      this._ch.writeFloat64(x);
    }
  }, {
    key: "packString",
    value: function packString(x) {
      var bytes = _utf82["default"].encode(x);
      var size = bytes.length;
      if (size < 0x10) {
        this._ch.writeUInt8(TINY_STRING | size);
        this._ch.writeBytes(bytes);
      } else if (size < 0x100) {
        this._ch.writeUInt8(STRING_8);
        this._ch.writeUInt8(size);
        this._ch.writeBytes(bytes);
      } else if (size < 0x10000) {
        this._ch.writeUInt8(STRING_16);
        this._ch.writeUInt8(size / 256 >> 0);
        this._ch.writeUInt8(size % 256);
        this._ch.writeBytes(bytes);
      } else if (size < 0x100000000) {
        this._ch.writeUInt8(STRING_32);
        this._ch.writeUInt8((size / 16777216 >> 0) % 256);
        this._ch.writeUInt8((size / 65536 >> 0) % 256);
        this._ch.writeUInt8((size / 256 >> 0) % 256);
        this._ch.writeUInt8(size % 256);
        this._ch.writeBytes(bytes);
      } else {
        throw (0, _error.newError)("UTF-8 strings of size " + size + " are not supported");
      }
    }
  }, {
    key: "packListHeader",
    value: function packListHeader(size) {
      if (size < 0x10) {
        this._ch.writeUInt8(TINY_LIST | size);
      } else if (size < 0x100) {
        this._ch.writeUInt8(LIST_8);
        this._ch.writeUInt8(size);
      } else if (size < 0x10000) {
        this._ch.writeUInt8(LIST_16);
        this._ch.writeUInt8((size / 256 >> 0) % 256);
        this._ch.writeUInt8(size % 256);
      } else if (size < 0x100000000) {
        this._ch.writeUInt8(LIST_32);
        this._ch.writeUInt8((size / 16777216 >> 0) % 256);
        this._ch.writeUInt8((size / 65536 >> 0) % 256);
        this._ch.writeUInt8((size / 256 >> 0) % 256);
        this._ch.writeUInt8(size % 256);
      } else {
        throw (0, _error.newError)("Lists of size " + size + " are not supported");
      }
    }
  }, {
    key: "packMapHeader",
    value: function packMapHeader(size) {
      if (size < 0x10) {
        this._ch.writeUInt8(TINY_MAP | size);
      } else if (size < 0x100) {
        this._ch.writeUInt8(MAP_8);
        this._ch.writeUInt8(size);
      } else if (size < 0x10000) {
        this._ch.writeUInt8(MAP_16);
        this._ch.writeUInt8(size / 256 >> 0);
        this._ch.writeUInt8(size % 256);
      } else if (size < 0x100000000) {
        this._ch.writeUInt8(MAP_32);
        this._ch.writeUInt8((size / 16777216 >> 0) % 256);
        this._ch.writeUInt8((size / 65536 >> 0) % 256);
        this._ch.writeUInt8((size / 256 >> 0) % 256);
        this._ch.writeUInt8(size % 256);
      } else {
        throw (0, _error.newError)("Maps of size " + size + " are not supported");
      }
    }
  }, {
    key: "packStructHeader",
    value: function packStructHeader(size, signature) {
      if (size < 0x10) {
        this._ch.writeUInt8(TINY_STRUCT | size);
        this._ch.writeUInt8(signature);
      } else if (size < 0x100) {
        this._ch.writeUInt8(STRUCT_8);
        this._ch.writeUInt8(size);
        this._ch.writeUInt8(signature);
      } else if (size < 0x10000) {
        this._ch.writeUInt8(STRUCT_16);
        this._ch.writeUInt8(size / 256 >> 0);
        this._ch.writeUInt8(size % 256);
      } else {
        throw (0, _error.newError)("Structures of size " + size + " are not supported");
      }
    }
  }]);

  return Packer;
})();

var Unpacker = (function () {
  function Unpacker() {
    _classCallCheck(this, Unpacker);

    // Higher level layers can specify how to map structs to higher-level objects.
    // If we recieve a struct that has a signature that does not have a mapper,
    // we simply return a Structure object.
    this.structMappers = {};
  }

  _createClass(Unpacker, [{
    key: "unpackList",
    value: function unpackList(size, buffer) {
      var value = [];
      for (var i = 0; i < size; i++) {
        value.push(this.unpack(buffer));
      }
      return value;
    }
  }, {
    key: "unpackMap",
    value: function unpackMap(size, buffer) {
      var value = {};
      for (var i = 0; i < size; i++) {
        var key = this.unpack(buffer);
        value[key] = this.unpack(buffer);
      }
      return value;
    }
  }, {
    key: "unpackStruct",
    value: function unpackStruct(size, buffer) {
      var signature = buffer.readUInt8();
      var mapper = this.structMappers[signature];
      if (mapper) {
        return mapper(this, buffer);
      } else {
        var value = new Structure(signature, []);
        for (var i = 0; i < size; i++) {
          value.fields.push(this.unpack(buffer));
        }
        return value;
      }
    }
  }, {
    key: "unpack",
    value: function unpack(buffer) {
      var marker = buffer.readUInt8();
      if (marker == NULL) {
        return null;
      } else if (marker == TRUE) {
        return true;
      } else if (marker == FALSE) {
        return false;
      } else if (marker == FLOAT_64) {
        return buffer.readFloat64();
      } else if (marker >= 0 && marker < 128) {
        return (0, _integer.int)(marker);
      } else if (marker >= 240 && marker < 256) {
        return (0, _integer.int)(marker - 256);
      } else if (marker == INT_8) {
        return (0, _integer.int)(buffer.readInt8());
      } else if (marker == INT_16) {
        return (0, _integer.int)(buffer.readInt16());
      } else if (marker == INT_32) {
        var b = buffer.readInt32();
        return (0, _integer.int)(b);
      } else if (marker == INT_64) {
        var high = buffer.readInt32();
        var low = buffer.readInt32();
        return new _integer.Integer(low, high);
      } else if (marker == STRING_8) {
        return _utf82["default"].decode(buffer, buffer.readUInt8());
      } else if (marker == STRING_16) {
        return _utf82["default"].decode(buffer, buffer.readUInt16());
      } else if (marker == STRING_32) {
        return _utf82["default"].decode(buffer, buffer.readUInt32());
      } else if (marker == LIST_8) {
        return this.unpackList(buffer.readUInt8(), buffer);
      } else if (marker == LIST_16) {
        return this.unpackList(buffer.readUInt16(), buffer);
      } else if (marker == LIST_32) {
        return this.unpackList(buffer.readUInt32(), buffer);
      } else if (marker == MAP_8) {
        return this.unpackMap(buffer.readUInt8(), buffer);
      } else if (marker == MAP_16) {
        return this.unpackMap(buffer.readUInt16(), buffer);
      } else if (marker == MAP_32) {
        return this.unpackMap(buffer.readUInt32(), buffer);
      } else if (marker == STRUCT_8) {
        return this.unpackStruct(buffer.readUInt8(), buffer);
      } else if (marker == STRUCT_16) {
        return this.unpackStruct(buffer.readUInt16(), buffer);
      }
      var markerHigh = marker & 0xF0;
      var markerLow = marker & 0x0F;
      if (markerHigh == 0x80) {
        return _utf82["default"].decode(buffer, markerLow);
      } else if (markerHigh == 0x90) {
        return this.unpackList(markerLow, buffer);
      } else if (markerHigh == 0xA0) {
        return this.unpackMap(markerLow, buffer);
      } else if (markerHigh == 0xB0) {
        return this.unpackStruct(markerLow, buffer);
      } else {
        throw (0, _error.newError)("Unknown packed value with marker " + marker.toString(16));
      }
    }
  }]);

  return Unpacker;
})();

exports["default"] = {
  Packer: Packer,
  Unpacker: Unpacker,
  Structure: Structure
};
module.exports = exports["default"];