"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getNetlifyGraphToken = void 0;
const process_1 = require("process");
const TOKEN_HEADER = 'X-Nf-Graph-Token';
const hasRequestStyleHeaders = function (headers) {
    return headers.get !== undefined && typeof headers.get === 'function';
};
const graphTokenFromIncomingHttpStyleHeaders = function (headers) {
    if (TOKEN_HEADER in headers) {
        const header = headers[TOKEN_HEADER];
        if (header == null || typeof header === 'string') {
            return header;
        }
        return header[0];
    }
};
// Backwards compatibility with older version of cli that doesn't inject header
const authlifyTokenFallback = function (event) {
    const token = event === null || event === void 0 ? void 0 : event.authlifyToken;
    return { token };
};
const graphTokenFromEvent = function (event) {
    const { headers } = event;
    // Check if object first in case there is a header with key `get`
    const token = graphTokenFromIncomingHttpStyleHeaders(headers);
    if (token) {
        return { token };
    }
    if (hasRequestStyleHeaders(headers)) {
        return { token: headers.get(TOKEN_HEADER) };
    }
    return authlifyTokenFallback(event);
};
const graphTokenFromEnv = function () {
    // _NETLIFY_GRAPH_TOKEN injected by next plugin
    // eslint-disable-next-line no-underscore-dangle
    const token = process_1.env._NETLIFY_GRAPH_TOKEN || process_1.env.NETLIFY_GRAPH_TOKEN;
    return { token };
};
const isEventRequired = function () {
    const localDev = process_1.env.NETLIFY_DEV === 'true';
    const localBuild = !localDev && process_1.env.NETLIFY_LOCAL === 'true';
    const remoteBuild = process_1.env.NETLIFY === 'true';
    // neither `localBuild` nor `remoteBuild` will be true in the on-demand builder case
    const inBuildPhase = localBuild || remoteBuild;
    const inGetStaticProps = 
    // Set by the nextjs plugin
    // eslint-disable-next-line no-underscore-dangle
    typeof process_1.env._NETLIFY_GRAPH_TOKEN !== 'undefined';
    return !inBuildPhase && !inGetStaticProps;
};
const incorrectArgumentsErrors = function (event) {
    const requiresEvent = isEventRequired();
    if (requiresEvent && event == null) {
        const errorMessage = 'You must provide an event or request to `getNetlifyGraphToken` when used in functions and on-demand builders.';
        return [{ type: 'missing-event-in-function', message: errorMessage }];
    }
    if (!requiresEvent && event != null) {
        const errorMessage = 'You must not pass arguments to `getNetlifyGraphToken` when used in builds.';
        return [{ type: 'provided-event-in-build', message: errorMessage }];
    }
};
const logErrors = function (errors) {
    for (const error of errors) {
        // Log errors to help guide developer
        console.error(error.message);
    }
};
const getNetlifyGraphToken = function (event, 
// caller can prevent error log. Allows getSecrets to provide better errors
supressLog) {
    const errors = incorrectArgumentsErrors(event);
    if (errors) {
        if (!supressLog) {
            logErrors(errors);
        }
        return { errors };
    }
    return event ? graphTokenFromEvent(event) : graphTokenFromEnv();
};
exports.getNetlifyGraphToken = getNetlifyGraphToken;
