/**
 * Utilities for dealing with `Date` objects.
 *
 * This module is browser compatible.
 * @module
 */
export declare const SECOND = 1000;
export declare const MINUTE: number;
export declare const HOUR: number;
export declare const DAY: number;
export declare const WEEK: number;
/**
 * Parse date from string using format string
 * @param dateString Date string
 * @param format Format string
 * @return Parsed date
 */
export declare function parse(dateString: string, formatString: string): Date;
/**
 * Format date using format string
 * @param date Date
 * @param format Format string
 * @return formatted date string
 */
export declare function format(date: Date, formatString: string): string;
/**
 * Get number of the day in the year
 * @return Number of the day in year
 */
export declare function dayOfYear(date: Date): number;
/**
 * Get number of the week in the year (ISO-8601)
 * @return Number of the week in year
 */
export declare function weekOfYear(date: Date): number;
/**
 * Parse a date to return a IMF formatted string date
 * RFC: https://tools.ietf.org/html/rfc7231#section-7.1.1.1
 * IMF is the time format to use when generating times in HTTP
 * headers. The time being formatted must be in UTC for Format to
 * generate the correct format.
 * @param date Date to parse
 * @return IMF date formatted string
 */
export declare function toIMF(date: Date): string;
/**
 * Check given year is a leap year or not.
 * based on : https://docs.microsoft.com/en-us/office/troubleshoot/excel/determine-a-leap-year
 * @param year year in number or Date format
 */
export declare function isLeap(year: Date | number): boolean;
export declare type Unit = "milliseconds" | "seconds" | "minutes" | "hours" | "days" | "weeks" | "months" | "quarters" | "years";
export declare type DifferenceFormat = Partial<Record<Unit, number>>;
export declare type DifferenceOptions = {
    units?: Unit[];
};
/**
 * Calculate difference between two dates.
 * @param from Year to calculate difference
 * @param to Year to calculate difference with
 * @param options Options for determining how to respond
 *
 * example :
 *
 * ```typescript
 * import * as datetime from "./mod.ts";
 *
 * datetime.difference(new Date("2020/1/1"),new Date("2020/2/2"),{ units : ["days","months"] })
 * ```
 */
export declare function difference(from: Date, to: Date, options?: DifferenceOptions): DifferenceFormat;
