"use strict";
// Copyright 2018-2022 the Deno authors. All rights reserved. MIT license.
/**
 * Utilities for dealing with `Date` objects.
 *
 * This module is browser compatible.
 * @module
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.difference = exports.isLeap = exports.toIMF = exports.weekOfYear = exports.dayOfYear = exports.format = exports.parse = exports.WEEK = exports.DAY = exports.HOUR = exports.MINUTE = exports.SECOND = void 0;
const formatter_js_1 = require("./formatter.js");
exports.SECOND = 1e3;
exports.MINUTE = exports.SECOND * 60;
exports.HOUR = exports.MINUTE * 60;
exports.DAY = exports.HOUR * 24;
exports.WEEK = exports.DAY * 7;
const DAYS_PER_WEEK = 7;
var Day;
(function (Day) {
    Day[Day["Sun"] = 0] = "Sun";
    Day[Day["Mon"] = 1] = "Mon";
    Day[Day["Tue"] = 2] = "Tue";
    Day[Day["Wed"] = 3] = "Wed";
    Day[Day["Thu"] = 4] = "Thu";
    Day[Day["Fri"] = 5] = "Fri";
    Day[Day["Sat"] = 6] = "Sat";
})(Day || (Day = {}));
/**
 * Parse date from string using format string
 * @param dateString Date string
 * @param format Format string
 * @return Parsed date
 */
function parse(dateString, formatString) {
    const formatter = new formatter_js_1.DateTimeFormatter(formatString);
    const parts = formatter.parseToParts(dateString);
    const sortParts = formatter.sortDateTimeFormatPart(parts);
    return formatter.partsToDate(sortParts);
}
exports.parse = parse;
/**
 * Format date using format string
 * @param date Date
 * @param format Format string
 * @return formatted date string
 */
function format(date, formatString) {
    const formatter = new formatter_js_1.DateTimeFormatter(formatString);
    return formatter.format(date);
}
exports.format = format;
/**
 * Get number of the day in the year
 * @return Number of the day in year
 */
function dayOfYear(date) {
    // Values from 0 to 99 map to the years 1900 to 1999. All other values are the actual year. (https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date/Date)
    // Using setFullYear as a workaround
    const yearStart = new Date(date);
    yearStart.setUTCFullYear(date.getUTCFullYear(), 0, 0);
    const diff = date.getTime() -
        yearStart.getTime();
    return Math.floor(diff / exports.DAY);
}
exports.dayOfYear = dayOfYear;
/**
 * Get number of the week in the year (ISO-8601)
 * @return Number of the week in year
 */
function weekOfYear(date) {
    const workingDate = new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
    const day = workingDate.getUTCDay();
    const nearestThursday = workingDate.getUTCDate() +
        Day.Thu -
        (day === Day.Sun ? DAYS_PER_WEEK : day);
    workingDate.setUTCDate(nearestThursday);
    // Get first day of year
    const yearStart = new Date(Date.UTC(workingDate.getUTCFullYear(), 0, 1));
    // return the calculated full weeks to nearest Thursday
    return Math.ceil((workingDate.getTime() - yearStart.getTime() + exports.DAY) / exports.WEEK);
}
exports.weekOfYear = weekOfYear;
/**
 * Parse a date to return a IMF formatted string date
 * RFC: https://tools.ietf.org/html/rfc7231#section-7.1.1.1
 * IMF is the time format to use when generating times in HTTP
 * headers. The time being formatted must be in UTC for Format to
 * generate the correct format.
 * @param date Date to parse
 * @return IMF date formatted string
 */
function toIMF(date) {
    function dtPad(v, lPad = 2) {
        return v.padStart(lPad, "0");
    }
    const d = dtPad(date.getUTCDate().toString());
    const h = dtPad(date.getUTCHours().toString());
    const min = dtPad(date.getUTCMinutes().toString());
    const s = dtPad(date.getUTCSeconds().toString());
    const y = date.getUTCFullYear();
    const days = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];
    const months = [
        "Jan",
        "Feb",
        "Mar",
        "Apr",
        "May",
        "Jun",
        "Jul",
        "Aug",
        "Sep",
        "Oct",
        "Nov",
        "Dec",
    ];
    return `${days[date.getUTCDay()]}, ${d} ${months[date.getUTCMonth()]} ${y} ${h}:${min}:${s} GMT`;
}
exports.toIMF = toIMF;
/**
 * Check given year is a leap year or not.
 * based on : https://docs.microsoft.com/en-us/office/troubleshoot/excel/determine-a-leap-year
 * @param year year in number or Date format
 */
function isLeap(year) {
    const yearNumber = year instanceof Date ? year.getFullYear() : year;
    return ((yearNumber % 4 === 0 && yearNumber % 100 !== 0) || yearNumber % 400 === 0);
}
exports.isLeap = isLeap;
/**
 * Calculate difference between two dates.
 * @param from Year to calculate difference
 * @param to Year to calculate difference with
 * @param options Options for determining how to respond
 *
 * example :
 *
 * ```typescript
 * import * as datetime from "./mod.ts";
 *
 * datetime.difference(new Date("2020/1/1"),new Date("2020/2/2"),{ units : ["days","months"] })
 * ```
 */
function difference(from, to, options) {
    const uniqueUnits = (options === null || options === void 0 ? void 0 : options.units) ? [...new Set(options === null || options === void 0 ? void 0 : options.units)] : [
        "milliseconds",
        "seconds",
        "minutes",
        "hours",
        "days",
        "weeks",
        "months",
        "quarters",
        "years",
    ];
    const bigger = Math.max(from.getTime(), to.getTime());
    const smaller = Math.min(from.getTime(), to.getTime());
    const differenceInMs = bigger - smaller;
    const differences = {};
    for (const uniqueUnit of uniqueUnits) {
        switch (uniqueUnit) {
            case "milliseconds":
                differences.milliseconds = differenceInMs;
                break;
            case "seconds":
                differences.seconds = Math.floor(differenceInMs / exports.SECOND);
                break;
            case "minutes":
                differences.minutes = Math.floor(differenceInMs / exports.MINUTE);
                break;
            case "hours":
                differences.hours = Math.floor(differenceInMs / exports.HOUR);
                break;
            case "days":
                differences.days = Math.floor(differenceInMs / exports.DAY);
                break;
            case "weeks":
                differences.weeks = Math.floor(differenceInMs / exports.WEEK);
                break;
            case "months":
                differences.months = calculateMonthsDifference(bigger, smaller);
                break;
            case "quarters":
                differences.quarters = Math.floor((typeof differences.months !== "undefined" &&
                    differences.months / 4) ||
                    calculateMonthsDifference(bigger, smaller) / 4);
                break;
            case "years":
                differences.years = Math.floor((typeof differences.months !== "undefined" &&
                    differences.months / 12) ||
                    calculateMonthsDifference(bigger, smaller) / 12);
                break;
        }
    }
    return differences;
}
exports.difference = difference;
function calculateMonthsDifference(bigger, smaller) {
    const biggerDate = new Date(bigger);
    const smallerDate = new Date(smaller);
    const yearsDiff = biggerDate.getFullYear() - smallerDate.getFullYear();
    const monthsDiff = biggerDate.getMonth() - smallerDate.getMonth();
    const calendarDifferences = Math.abs(yearsDiff * 12 + monthsDiff);
    const compareResult = biggerDate > smallerDate ? 1 : -1;
    biggerDate.setMonth(biggerDate.getMonth() - compareResult * calendarDifferences);
    const isLastMonthNotFull = biggerDate > smallerDate
        ? 1
        : -1 === -compareResult
            ? 1
            : 0;
    const months = compareResult * (calendarDifferences - isLastMonthNotFull);
    return months === 0 ? 0 : months;
}
