/*
 * @nevware21/ts-utils
 * https://github.com/nevware21/ts-utils
 *
 * Copyright (c) 2024 Nevware21
 * Licensed under the MIT license.
 */
import { objDefineProp } from "../object/define";
/**
 * Create and return a readonly {@link ICachedValue} instance that is populated with the provided value.
 * This is useful when you want to cache a previously fetched value and return it without having to re-fetch
 * it again.
 * This is a lightweight version of {@link getLazy} which does not support any expiration or invalidation,
 * it also will not honor the {@link setBypassLazyCache} setting and will always return the provided value.
 * @since 0.10.5
 * @group Helpers
 * @group Cache
 * @typeparam T - The type of the value to be cached
 * @param value
 * @returns A new {@link ICachedValue} instance which wraps the provided value
 * @example
 * ```ts
 * let cachedValue = createCachedValue("some value");
 * // cachedValue.v === "some value"
 *
 * JSON.stringify(cachedValue) === '{"v":"some value"}';
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function createCachedValue(value) {
    return objDefineProp({
        toJSON: function () { return value; }
    }, "v", { value: value });
}
/**
 * Create and return a readonly {@link ICachedValue} instance which will cache and return the value returned
 * by the callback function. The callback function will only be called once, multiple access of the value
 * will not cause re-execution of the callback as the result from the first call is cached internally.
 * This is a lightweight version of {@link getLazy} which does not support any expiration or invalidation,
 * it also will not honor the {@link setBypassLazyCache} setting and will always return the provided value.
 * @since 0.10.5
 * @group Helpers
 * @group Cache
 * @typeparam T - The type of the value to be cached
 * @param cb - The callback function to fetch the value to be lazily evaluated and cached
 * @returns
 */
/*#__NO_SIDE_EFFECTS__*/
export function createDeferredCachedValue(cb) {
    return objDefineProp({}, "v", {
        get: function () {
            var result = cb();
            objDefineProp(this, "v", {
                value: result,
                enumerable: true
            });
            return result;
        },
        enumerable: true,
        configurable: true
    });
}
//# sourceMappingURL=cache.js.map