/*
 * @nevware21/ts-utils
 * https://github.com/nevware21/ts-utils
 *
 * Copyright (c) 2022 Nevware21
 * Licensed under the MIT license.
 */
import { DONE, VALUE } from "../internal/constants";
import { objDefine } from "../object/define";
import { getKnownSymbol } from "../symbol/symbol";
/**
 * Create an iterable which conforms to the `Iterable` protocol, it uses the provided `ctx` to
 * create an `Iterator` via {@link createIterator}.
 *
 * @see [Iterable protocol](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols#the_iterable_protocol)
 * @since 0.4.2
 * @group Iterator
 * @typeParam T - Identifies the type that will be returned by the iterator
 * @param ctx - The context used to manage the iteration over the items.
 * @returns A new Iterable instance
 * @example
 * ```ts
 * let current = 0;
 * let next = 1;
 * let done = false;
 * let fibCtx: CreateIteratorContext<number> = {
 *     n: function() {
 *         fibCtx.v = current;
 *         current = next;
 *         next = fibCtx.v + next;
 *
 *         // Return not done
 *         return false;
 *     },
 *     r: function(value) {
 *         done = true;
 *         return value;
 *     }
 * };
 *
 * let values: number[] = [];
 * iterForOf(createIterable(fibCtx), (value) => {
 *     values.push(value);
 *     if (values.length === 10) {
 *         return -1;
 *     }
 * });
 *
 * // Done is true
 * // done === true
 * // Values: [0, 1, 1, 2, 3, 5, 8, 13, 21, 34]
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function createIterable(ctx) {
    return makeIterable({}, ctx);
}
/**
 * Adds or replaces an iterable implementation that conforms to the `Iterable` protocol to the target instance, it
 * uses the provided `ctx` to create an `Iterator` via {@link createIterator}.
 *
 * @see [Iterable protocol](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols#the_iterable_protocol)
 * @since 0.4.2
 * @group Iterator
 * @typeParam T - Identifies the target type
 * @typeParam I - Identifies the type that will be returned by the iterator
 * @param ctx - The context used to manage the iteration over the items.
 * @returns A new Iterable instance
 * @example
 * ```ts
 * let current = 0;
 * let next = 1;
 * let done = false;
 * let fibCtx: CreateIteratorContext<number> = {
 *     n: function() {
 *         fibCtx.v = current;
 *         current = next;
 *         next = fibCtx.v + next;
 *
 *         // Return not done, so it will just continue
 *         return false;
 *     }
 * };
 *
 * let values: number[] = [];
 * let theIterable: Iterable<T> = makeIterable({}, fibCtx);
 *
 * iterForOf(theIterable, (value) => {
 *     values.push(value);
 *     if (values.length === 10) {
 *         return -1;
 *     }
 * });
 *
 * // Values: [0, 1, 1, 2, 3, 5, 8, 13, 21, 34]
 * ```
 */
export function makeIterable(target, ctx) {
    var itSymbol = getKnownSymbol(3 /* WellKnownSymbols.iterator */);
    function _createIterator() {
        return createIterator(ctx);
    }
    target[itSymbol] = _createIterator;
    return target;
}
/**
 * Create an iterator which conforms to the `Iterator` protocol, it uses the provided `ctx` to
 * managed moving to the `next`.
 *
 * @see [Iterator protocol](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols#the_iterator_protocol)
 * @since 0.4.2
 * @group Iterator
 * @typeParam T - Identifies the type that will be returned by the iterator
 * @param ctx - The context used to manage the iteration over the items.
 * @returns A new Iterator instance
 * @example
 * ```ts
 * let idx = -1;
 * let theValues = [ 5, 10, 15, 20, 25, 30 ];
 *
 * function getNextFn() {
 *     idx++;
 *     let isDone = idx >= theValues.length;
 *     if (!isDone) {
 *         // this is passed as the current iterator
 *         // so you can directly assign the next "value" that will be returned
 *         this.v = theValues[idx];
 *     }
 *
 *     return isDone;
 * }
 *
 * let theIterator = createIterator<number>({ n: getNextFn });
 *
 * let values: number[] = [];
 * iterForOf(theIterator, (value) => {
 *     values.push(value);
 * });
 *
 * // Values: [5, 10, 15, 20, 25, 30 ]
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function createIterator(ctx) {
    var isDone = false;
    function _value() {
        return ctx.v;
    }
    function _next() {
        var _a;
        isDone = isDone || (ctx.n ? ctx.n(arguments) : true);
        var result = (_a = {},
            _a[DONE] = isDone,
            _a);
        if (!isDone) {
            objDefine(result, VALUE, { g: _value });
        }
        return result;
    }
    function _return(value) {
        var _a;
        isDone = true;
        return _a = {},
            _a[DONE] = true,
            _a[VALUE] = ctx.r && ctx.r(value),
            _a;
    }
    function _throw(e) {
        var _a;
        isDone = true;
        return _a = {},
            _a[DONE] = true,
            _a[VALUE] = ctx.t && ctx.t(e),
            _a;
    }
    var theIterator = {
        next: _next
    };
    if (ctx.r) {
        theIterator.return = _return;
    }
    if (ctx.t) {
        theIterator.throw = _throw;
    }
    return theIterator;
}
//# sourceMappingURL=create.js.map