/*
 * @nevware21/ts-utils
 * https://github.com/nevware21/ts-utils
 *
 * Copyright (c) 2022 Nevware21
 * Licensed under the MIT license.
 */
import { getKnownSymbol } from "../symbol/symbol";
import { isFunction, isStrictNullOrUndefined } from "../helpers/base";
/**
 * Checks if the type of value looks like an iterator instance (contains a next function).
 *
 * @since 0.4.0
 * @group Type Identity
 * @group Iterator
 * @typeParam T - Identifies the return type of the iterator defaults to any
 * @param value - The value to be checked
 * @returns {boolean} True if the value is an Iterator, otherwise false
 * @example
 * ```ts
 * isIterator(null);        // false
 * isIterator(undefined);   // false
 * isIterator("null");      // false (Strings are iterable but not iterators)
 * isIterator([]);          // false (Arrays are iterable but not iterators)
 * isIterator({
 *     next: function() { return true }
 * });                      // true, iterators must contain a "next" function
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function isIterator(value) {
    return !!value && isFunction(value.next);
}
/**
 * Checks if the value looks like it is iterable, contains a [symbol.iterator].
 *
 * @since 0.4.0
 * @group Type Identity
 * @group Iterator
 * @typeParam T - Identifies the return type of the iterator
 * @param value - The value to be checked
 * @returns {boolean} True if the value is an Iterable, otherwise false
 * @example
 * ```ts
 * isIterable(null);        // false
 * isIterable(undefined);   // false
 * isIterable("null");      // true (Strings are iterable)
 * isIterable([]);          // true (Arrays are iterable)
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function isIterable(value) {
    return !isStrictNullOrUndefined(value) && isFunction(value[getKnownSymbol(3 /* WellKnownSymbols.iterator */)]);
}
//# sourceMappingURL=iterator.js.map