/*
 * @nevware21/ts-utils
 * https://github.com/nevware21/ts-utils
 *
 * Copyright (c) 2022 Nevware21
 * Licensed under the MIT license.
 */
import { arrForEach } from "../array/forEach";
import { isArray, isDate, isNullOrUndefined, isPrimitiveType } from "../helpers/base";
import { CALL, FUNCTION, NULL_VALUE, OBJECT } from "../internal/constants";
import { objDefine } from "./define";
import { isPlainObject } from "./is_plain_object";
/**
 * @internal
 * @ignore
 * Generic Object deep copy handler which creates a new plain object and copies enumerable properties from
 * the source to the new target plain object. The source object does not have to be a plain object.
 * @param details - The details object for the current property being copied
 * @returns true if the handler processed the field.
 */
const _defaultDeepCopyHandler = (details) => {
    // Make sure we at least copy plain objects
    details.value && plainObjDeepCopyHandler(details);
    // Always return true so that the iteration completes
    return true;
};
/**
 * @internal
 * @ignore
 * The ordered default deep copy handlers
 */
const defaultDeepCopyHandlers = [
    arrayDeepCopyHandler,
    plainObjDeepCopyHandler,
    functionDeepCopyHandler,
    dateDeepCopyHandler
];
/**
 * @internal
 * @ignore
 * Helper function used for detecting and handling recursive properties
 * @param visitMap - The current map of objects that have been visited
 * @param source - The value (object) to be copied
 * @param newPath - The new access path from the origin to the current property
 * @param cb - The callback function to call if the current object has not already been processed.
 * @returns The new deep copied property, may be incomplete as the object is recursive and is still in the process of being copied
 */
function _getSetVisited(visitMap, source, newPath, cb) {
    let theEntry;
    arrForEach(visitMap, (entry) => {
        if (entry.k === source) {
            theEntry = entry;
            return -1;
        }
    });
    if (!theEntry) {
        // Add the target to the visit map so that deep nested objects refer to the single instance
        // Even if we have not finished processing it yet.
        theEntry = { k: source, v: source };
        visitMap.push(theEntry);
        // Now call the copy callback so that it populates the target
        cb(theEntry);
    }
    return theEntry.v;
}
/**
 * @internal
 * @ignore
 * Internal helper which performs the recursive deep copy
 * @param visitMap - The current map of objects that have been visited
 * @param value - The value being copied
 * @param ctx - The current copy context
 * @param key - [Optional] the current `key` for the value from the source object
 * @returns The new deep copied instance of the value.
 */
function _deepCopy(visitMap, value, ctx, key) {
    let userHandler = ctx.handler;
    let newPath = ctx.path ? (key ? ctx.path.concat(key) : ctx.path) : [];
    let newCtx = {
        handler: ctx.handler,
        src: ctx.src,
        path: newPath
    };
    const theType = typeof value;
    let isPlain = false;
    let isPrim = false;
    if (value && theType === OBJECT) {
        isPlain = isPlainObject(value);
    }
    else {
        isPrim = value === NULL_VALUE || isPrimitiveType(theType);
    }
    let details = {
        type: theType,
        isPrim: isPrim,
        isPlain: isPlain,
        value: value,
        result: value,
        path: newPath,
        origin: ctx.src,
        copy: (source, newKey) => {
            return _deepCopy(visitMap, source, newKey ? newCtx : ctx, newKey);
        },
        copyTo: (target, source) => {
            return _copyProps(visitMap, target, source, newCtx);
        }
    };
    if (!details.isPrim) {
        return _getSetVisited(visitMap, value, newPath, (newEntry) => {
            // Use an accessor to set the new value onto the new entry
            objDefine(details, "result", {
                g: function () {
                    return newEntry.v;
                },
                s: function (newValue) {
                    newEntry.v = newValue;
                }
            });
            let idx = 0;
            let handler = userHandler;
            while (!(handler || (idx < defaultDeepCopyHandlers.length ? defaultDeepCopyHandlers[idx++] : _defaultDeepCopyHandler))[CALL](ctx, details)) {
                handler = NULL_VALUE;
            }
        });
    }
    // Allow the user handler to override the provided value
    if (userHandler && userHandler[CALL](ctx, details)) {
        return details.result;
    }
    return value;
}
/**
 * @internal
 * @ignore
 * Internal helper to copy all of the enumerable properties from the source object to the new target object
 * @param visitMap - The current map of objects that have been visited
 * @param target - The target object to copy the properties to.
 * @param source - The source object to copy the properties from.
 * @param ctx - The current deep copy context
 * @returns The populated target object
 */
function _copyProps(visitMap, target, source, ctx) {
    if (!isNullOrUndefined(source)) {
        // Copy all properties (not just own properties)
        for (const key in source) {
            // Perform a deep copy of the object
            target[key] = _deepCopy(visitMap, source[key], ctx, key);
        }
    }
    return target;
}
/**
 * Object helper to copy all of the enumerable properties from the source object to the target, the
 * properties are copied via {@link objDeepCopy}. Automatic handling of recursive properties was added in v0.4.4
 * @group Object
 * @param target - The target object to populated
 * @param source - The source object to copy the properties from
 * @param handler - An optional callback that lets you provide / overide the deep cloning (Since 0.4.4)
 * @returns The target object
 * @example
 * ```ts
 * let a: any = { a: 1 };
 * let b: any = { b: 2, d: new Date(), e: new TestClass("Hello Darkness") };
 * a.b = b;        // { a: 1, b: { b: 2} }
 * b.a = a;        // { a: 1, b: { b: 2, a: { a: 1, { b: 2, a: ... }}}}
 *
 * function copyHandler(details: IObjDeepCopyHandlerDetails) {
 *     // details.origin === a
 *     // details.path[] is the path to the current value
 *     if (details.value && isDate(details.value)) {
 *         // So for the date path === [ "b", "d" ] which represents
 *         // details.origin["b"]["d"] === The Date
 *
 *         // Create a clone the Date object and set as the "newValue"
 *         details.value = new Date(details.value.getTime());
 *
 *         // Return true to indicate that we have "handled" the conversion
 *         // See objDeepCopy example for just reusing the original value (just don't replace details.value)
 *         return true;
 *     }
 *
 *     return false;
 * }
 *
 * let c: any = objCopyProps({}, a, copyHandler);
 *
 * assert.notEqual(a, c, "check a and c are not the same");
 * assert.ok(c !== c.b.a, "The root object won't be the same for the target reference as are are copying properties to our target");
 * assert.ok(c.b === c.b.a.b, "Check that the 2 'b' references are the same object");
 * assert.ok(c.b.a === c.b.a.b.a, "Check that the 2 'a' references are the same object");
 * assert.ok(c.b.d === c.b.a.b.d, "Check that the 2 'd' references are the same object");
 * assert.ok(isDate(c.b.d), "The copied date is still real 'Date' instance");
 * assert.notEqual(c.b.d, a.b.d, "And the copied date is not the same as the original");
 * assert.equal(c.b.d.getTime(), a.b.d.getTime(), "But the dates are the same");
 *
 * assert.ok(isObject(c.b.d), "The copied date is now an object");
 * ```
 */
export function objCopyProps(target, source, handler) {
    let ctx = {
        handler: handler,
        src: source,
        path: []
    };
    return _copyProps([], target, source, ctx);
}
/**
 * Performs a deep copy of the source object, this is designed to work with base (plain) objects, arrays and primitives
 * if the source object contains class objects they will either be not cloned or may be considered non-operational after
 * performing a deep copy. ie. This is performing a deep copy of the objects properties so that altering the copy will
 * not mutate the source object hierarchy.
 * Automatic handling of recursive properties was added in v0.4.4.
 * @group Object
 * @group Object - Deep Copy
 * @param source - The source object to be copied
 * @param handler - An optional callback that lets you provide / overide the deep cloning (Since 0.4.4)
 * @return A new object which contains a deep copy of the source properties
 * @example
 * ```ts
 * let a: any = { a: 1 };
 * let b: any = { b: 2, d: new Date(), e: new TestClass("Hello Darkness") };
 * a.b = b;        // { a: 1, b: { b: 2} }
 * b.a = a;        // { a: 1, b: { b: 2, a: { a: 1, { b: 2, a: ... }}}}
 *
 * function copyHandler(details: IObjDeepCopyHandlerDetails) {
 *     // details.origin === a
 *     // details.path[] is the path to the current value
 *     if (details.value && isDate(details.value)) {
 *         // So for the date path === [ "b", "d" ] which represents
 *         // details.origin["b"]["d"] === The Date
 *
 *         // Return true to indicate that we have "handled" the conversion
 *         // Which in this case will reuse the existing instance (as we didn't replace details.value)
 *         // See objCopyProps example for replacing the Date instance
 *         return true;
 *     }
 *
 *     return false;
 * }
 *
 * let c: any = objDeepCopy(a, copyHandler);
 *
 * assert.notEqual(a, c, "check a and c are not the same");
 * assert.ok(c === c.b.a, "The root object won't be the same for the target reference");
 * assert.ok(c.b === c.b.a.b, "Check that the 2 'b' references are the same object");
 * assert.ok(c.b.a === c.b.a.b.a, "Check that the 2 'a' references are the same object");
 * assert.ok(c.b.d === c.b.a.b.d, "Check that the 2 'd' references are the same object");
 * assert.ok(isDate(c.b.d), "The copied date is still real 'Date' instance");
 * assert.equal(c.b.d, a.b.d, "And the copied date is the original date");
 * assert.equal(c.b.d.getTime(), a.b.d.getTime(), "But the dates are the same");
 * assert.ok(isObject(c.b.d), "The copied date is now an object");
 * assert.ok(!isError(c.b.e), "The copied error is no longer a real 'Error' instance");
 * assert.ok(isObject(c.b.e), "The copied error is now an object");
 * assert.equal(42, c.b.e.value, "Expect that the local property was copied");
 * ```
 */
/*#__NO_SIDE_EFFECTS__*/
export function objDeepCopy(source, handler) {
    let ctx = {
        handler: handler,
        src: source
    };
    return _deepCopy([], source, ctx);
}
/**
 * Deep copy handler to identify and copy arrays.
 * @since 0.4.4
 * @group Object - Deep Copy
 * @param details - The details object for the current property being copied
 * @returns `true` if the current value is a function otherwise `false`
 */
export function arrayDeepCopyHandler(details) {
    let value = details.value;
    if (isArray(value)) {
        // Assign the "result" value before performing any additional deep Copying, so any recursive object get a reference to this instance
        let target = details.result = [];
        target.length = value.length;
        // Copying all properties as arrays can contain non-indexed based properties
        details.copyTo(target, value);
        return true;
    }
    return false;
}
/**
 * Deep copy handler to identify and copy Date instances.
 * @since 0.4.4
 * @group Object - Deep Copy
 * @param details - The details object for the current property being copied
 * @returns `true` if the current value is a function otherwise `false`
 */
export function dateDeepCopyHandler(details) {
    let value = details.value;
    if (isDate(value)) {
        details.result = new Date(value.getTime());
        return true;
    }
    return false;
}
/**
 * Deep copy handler to identify and copy functions. This handler just returns the original
 * function so the original function will be assigned to any new deep copied instance.
 * @since 0.4.4
 * @group Object - Deep Copy
 * @param details - The details object for the current property being copied
 * @returns `true` if the current value is a function otherwise `false`
 */
export function functionDeepCopyHandler(details) {
    if (details.type === FUNCTION) {
        return true;
    }
    return false;
}
/**
 * Deep copy handler to identify and copy plain objects.
 * @since 0.4.4
 * @group Object - Deep Copy
 * @param details - The details object for the current property being copied
 * @returns `true` if the current value is a function otherwise `false`
 */
export function plainObjDeepCopyHandler(details) {
    let value = details.value;
    if (value && details.isPlain) {
        // Assign the "result" value before performing any additional deep Copying, so any recursive object get a reference to this instance
        let target = details.result = {};
        details.copyTo(target, value);
        return true;
    }
    return false;
}
//# sourceMappingURL=copy.js.map