/*! https://github.com/nevware21/ts-utils v0.10.5 */
/*
 * Copyright (c) NevWare21 and contributors. All rights reserved.
 * Licensed under the MIT license.
 */
const UNDEF_VALUE = undefined;
const NULL_VALUE = null;
const EMPTY = "";
const BOOLEAN = "boolean";
const FUNCTION = "function";
const NUMBER = "number";
const OBJECT = "object";
const PROTOTYPE = "prototype";
const __PROTO__ = "__proto__";
const STRING = "string";
const UNDEFINED = "undefined";
const CONSTRUCTOR = "constructor";
const SYMBOL = "Symbol";
const POLYFILL_TAG = "_polyfill";
const INDEX_OF = "indexOf";
const LAST_INDEX_OF = "lastIndexOf";
const LENGTH = "length";
const DONE = "done";
const VALUE = "value";
const NAME = "name";
const SLICE = "slice";
const CALL = "call";
const TO_STRING = "toString";
const ObjClass = Object;
const ObjProto = ObjClass[PROTOTYPE];
const StrCls = String;
const StrProto = StrCls[PROTOTYPE];
const MathCls = Math;
const ArrCls = Array;
const ArrProto = ArrCls[PROTOTYPE];

function safe(func, argArray) {
    try {
        return {
            v: func.apply(this, argArray)
        };
    }
    catch (e) {
        return { e };
    }
}

function safeGet(cb, defValue) {
    let result = safe(cb);
    return result.e ? defValue : result.v;
}

const PRIMITIVE_TYPES = [STRING, NUMBER, BOOLEAN, UNDEFINED, "symbol", "bigint"];
/*#__NO_SIDE_EFFECTS__*/
function _createIs(theType) {
    return function (value) {
        return typeof value === theType;
    };
}
/*#__NO_SIDE_EFFECTS__*/
function _createObjIs(theName) {
    const theType = "[object " + theName + "]";
    return function (value) {
        return !!(value && objToString(value) === theType);
    };
}
/*#__NO_SIDE_EFFECTS__*/
function objToString(value) {
    return ObjProto.toString.call(value);
}
/*#__NO_SIDE_EFFECTS__*/
function isTypeof(value, theType) {
    return typeof value === theType;
}
/*#__NO_SIDE_EFFECTS__*/
function isUndefined(value) {
    return typeof value === UNDEFINED || value === UNDEFINED;
}
/*#__NO_SIDE_EFFECTS__*/
function isStrictUndefined(arg) {
    return !isDefined(arg);
}
/*#__NO_SIDE_EFFECTS__*/
function isNullOrUndefined(value) {
    return value === NULL_VALUE || isUndefined(value);
}
/*#__NO_SIDE_EFFECTS__*/
function isStrictNullOrUndefined(value) {
    return value === NULL_VALUE || !isDefined(value);
}
/*#__NO_SIDE_EFFECTS__*/
function isDefined(arg) {
    return !!arg || arg !== UNDEF_VALUE;
}
/*#__NO_SIDE_EFFECTS__*/
function isPrimitive(value) {
    return value === NULL_VALUE || isPrimitiveType(typeof value);
}
/*#__NO_SIDE_EFFECTS__*/
function isPrimitiveType(theType) {
    return theType !== OBJECT && PRIMITIVE_TYPES.indexOf(theType) !== -1;
}
const isString = ( /*#__PURE__*/_createIs(STRING));
const isFunction = ( /*#__PURE__*/_createIs(FUNCTION));
/*#__NO_SIDE_EFFECTS__*/
function isObject(value) {
    if (!value && isNullOrUndefined(value)) {
        return false;
    }
    return !!value && typeof value === OBJECT;
}
const isArray = ArrCls.isArray;
const isDate = ( /*#__PURE__*/_createObjIs("Date"));
const isNumber = ( /*#__PURE__*/_createIs(NUMBER));
const isBoolean = ( /*#__PURE__*/_createIs(BOOLEAN));
const isRegExp = ( /*#__PURE__*/_createObjIs("RegExp"));
const isFile = ( /*#__PURE__*/_createObjIs("File"));
const isFormData = ( /*#__PURE__*/_createObjIs("FormData"));
const isBlob = ( /*#__PURE__*/_createObjIs("Blob"));
const isArrayBuffer = ( /*#__PURE__*/_createObjIs("ArrayBuffer"));
const isError = ( /*#__PURE__*/_createObjIs("Error"));
/*#__NO_SIDE_EFFECTS__*/
function isPromiseLike(value) {
    return !!(value && value.then && isFunction(value.then));
}
const isThenable = isPromiseLike;
/*#__NO_SIDE_EFFECTS__*/
function isPromise(value) {
    return !!(value && value.then && value.catch && isFunction(value.then) && isFunction(value.catch));
}
/*#__NO_SIDE_EFFECTS__*/
function isNotTruthy(value) {
    return !value || !isTruthy(value);
}
/*#__NO_SIDE_EFFECTS__*/
function isTruthy(value) {
    return !(!value || safeGet(() => !(value && (0 + value)), !value));
}

const objGetOwnPropertyDescriptor = ObjClass.getOwnPropertyDescriptor;

/*#__NO_SIDE_EFFECTS__*/
function objHasOwnProperty(obj, prop) {
    return !!obj && ObjProto.hasOwnProperty[CALL](obj, prop);
}

const objHasOwn = ObjClass["hasOwn"] || polyObjHasOwn;
/*#__NO_SIDE_EFFECTS__*/
function polyObjHasOwn(obj, prop) {
    return objHasOwnProperty(obj, prop) || !!objGetOwnPropertyDescriptor(obj, prop);
}

function objForEachKey(theObject, callbackfn, thisArg) {
    if (theObject && isObject(theObject)) {
        for (const prop in theObject) {
            if (objHasOwn(theObject, prop)) {
                if (callbackfn[CALL](thisArg || theObject, prop, theObject[prop]) === -1) {
                    break;
                }
            }
        }
    }
}

/*#__NO_SIDE_EFFECTS__*/
function _createKeyValueMap(values, keyType, valueType, completeFn) {
    let theMap = {};
    objForEachKey(values, (key, value) => {
        theMap[key] = keyType ? value : key;
        theMap[value] = valueType ? value : key;
    });
    return completeFn(theMap);
}

function throwError(message) {
    throw new Error(message);
}
function throwTypeError(message) {
    throw new TypeError(message);
}
function throwRangeError(message) {
    throw new RangeError(message);
}

/*#__NO_SIDE_EFFECTS__*/
function polyObjKeys(obj) {
    if (!isObject(obj) || obj === NULL_VALUE) {
        throwTypeError("polyObjKeys called on non-object");
    }
    const result = [];
    for (const prop in obj) {
        if (objHasOwn(obj, prop)) {
            result.push(prop);
        }
    }
    return result;
}
/*#__NO_SIDE_EFFECTS__*/
function polyObjEntries(value) {
    let result = [];
    objForEachKey(value, (key, value) => {
        result.push([key, value]);
    });
    return result;
}

const _objFreeze = ObjClass["freeze"];
function _doNothing(value) {
    return value;
}
/*#__NO_SIDE_EFFECTS__*/
function _getProto(value) {
    return value[__PROTO__] || NULL_VALUE;
}
const objAssign = ObjClass["assign"];
const objKeys = ObjClass.keys;
function objDeepFreeze(value) {
    if (_objFreeze) {
        objForEachKey(value, (key, value) => {
            if (isArray(value) || isObject(value)) {
                _objFreeze(value);
            }
        });
    }
    return objFreeze(value);
}
const objFreeze = _objFreeze || _doNothing;
const objSeal = ObjClass["seal"] || _doNothing;
const objGetPrototypeOf = ObjClass["getPrototypeOf"] || _getProto;
const objEntries = ObjClass.entries || polyObjEntries;

/*#__NO_SIDE_EFFECTS__*/
function createEnum(values) {
    return _createKeyValueMap(values, 1 , 0 , objDeepFreeze);
}
/*#__NO_SIDE_EFFECTS__*/
function createEnumKeyMap(values) {
    return _createKeyValueMap(values, 0 , 0 , objDeepFreeze);
}
/*#__NO_SIDE_EFFECTS__*/
function createEnumValueMap(values) {
    return _createKeyValueMap(values, 1 , 1 , objDeepFreeze);
}
/*#__NO_SIDE_EFFECTS__*/
function createSimpleMap(values) {
    let mapClass = {};
    objForEachKey(values, (key, value) => {
        mapClass[key] = value[1];
        mapClass[value[0]] = value[1];
    });
    return objDeepFreeze(mapClass);
}
/*#__NO_SIDE_EFFECTS__*/
function createTypeMap(values) {
    return createSimpleMap(values);
}

const _wellKnownSymbolMap = /*#__PURE__*/ createEnumKeyMap({
    asyncIterator: 0 ,
    hasInstance: 1 ,
    isConcatSpreadable: 2 ,
    iterator: 3 ,
    match: 4 ,
    matchAll: 5 ,
    replace: 6 ,
    search: 7 ,
    species: 8 ,
    split: 9 ,
    toPrimitive: 10 ,
    toStringTag: 11 ,
    unscopables: 12
});

const asString = StrCls;

const GLOBAL_CONFIG_KEY = "__tsUtils$gblCfg";
let _globalCfg;
function _getGlobalValue() {
    var result;
    if (typeof globalThis !== UNDEFINED) {
        result = globalThis;
    }
    if (!result && typeof self !== UNDEFINED) {
        result = self;
    }
    if (!result && typeof window !== UNDEFINED) {
        result = window;
    }
    if (!result && typeof global !== UNDEFINED) {
        result = global;
    }
    return result;
}
/*#__NO_SIDE_EFFECTS__*/
function _getGlobalConfig() {
    if (!_globalCfg) {
        let gbl = safe(_getGlobalValue).v || {};
        _globalCfg = gbl[GLOBAL_CONFIG_KEY] = gbl[GLOBAL_CONFIG_KEY] || {};
    }
    return _globalCfg;
}

const ERROR_TYPE = "[object Error]";
/*#__NO_SIDE_EFFECTS__*/
function dumpObj(object, format) {
    let propertyValueDump = EMPTY;
    const objType = ObjProto[TO_STRING][CALL](object);
    if (objType === ERROR_TYPE) {
        object = { stack: asString(object.stack), message: asString(object.message), name: asString(object.name) };
    }
    try {
        propertyValueDump = JSON.stringify(object, NULL_VALUE, format ? ((typeof format === NUMBER) ? format : 4) : UNDEF_VALUE);
        propertyValueDump = (propertyValueDump && propertyValueDump.replace(/"(\w+)"\s*:\s{0,1}/g, "$1: ")) || asString(object);
    }
    catch (e) {
        propertyValueDump = " - " + dumpObj(e, format);
    }
    return objType + ": " + propertyValueDump;
}

let _slice;
const _unwrapInstFunction =  _unwrapFunctionWithPoly;
const _unwrapFunction =  _unwrapFunctionWithPoly;
/*#__NO_SIDE_EFFECTS__*/
function _unwrapFunctionWithPoly(funcName, clsProto, polyFunc) {
    _slice = _slice || ArrProto[SLICE];
    let clsFn = clsProto && clsProto[funcName];
    return function (thisArg) {
        let theFunc = (thisArg && thisArg[funcName]) || clsFn;
        if (theFunc || polyFunc) {
            let theArgs = arguments;
            return (theFunc || polyFunc).apply(thisArg, theFunc ? _slice[CALL](theArgs, 1) : theArgs);
        }
        throwTypeError("\"" + asString(funcName) + "\" not defined for " + dumpObj(thisArg));
    };
}
/*#__NO_SIDE_EFFECTS__*/
function _unwrapProp(propName) {
    return function (thisArg) {
        return thisArg[propName];
    };
}

const mathMin = MathCls.min;
const mathMax = MathCls.max;

const strSlice = ( /*#__PURE__*/_unwrapFunction(SLICE, StrProto));

const strSubstring = ( /*#__PURE__*/_unwrapFunction("substring", StrProto));
const strSubstr = ( /*#__PURE__*/_unwrapFunctionWithPoly("substr", StrProto, polyStrSubstr));
/*#__NO_SIDE_EFFECTS__*/
function polyStrSubstr(value, start, length) {
    if (isNullOrUndefined(value)) {
        throwTypeError("'polyStrSubstr called with invalid " + dumpObj(value));
    }
    if (length < 0) {
        return EMPTY;
    }
    start = start || 0;
    if (start < 0) {
        start = mathMax(start + value[LENGTH], 0);
    }
    if (isUndefined(length)) {
        return strSlice(value, start);
    }
    return strSlice(value, start, start + length);
}
/*#__NO_SIDE_EFFECTS__*/
function strLeft(value, count) {
    return strSubstring(value, 0, count);
}
/*#__NO_SIDE_EFFECTS__*/
function strRight(value, count) {
    return count <= 0 ? EMPTY : (value[LENGTH] > count ? strSlice(value, -count) : value);
}

const UNIQUE_REGISTRY_ID = "_urid";
let _polySymbols;
/*#__NO_SIDE_EFFECTS__*/
function _globalSymbolRegistry() {
    if (!_polySymbols) {
        let gblCfg = _getGlobalConfig();
        _polySymbols = gblCfg.gblSym = gblCfg.gblSym || { k: {}, s: {} };
    }
    return _polySymbols;
}
let _wellKnownSymbolCache;
/*#__NO_SIDE_EFFECTS__*/
function polyNewSymbol(description) {
    let theSymbol = {
        description: asString(description),
        toString: () => SYMBOL + "(" + description + ")"
    };
    theSymbol[POLYFILL_TAG] = true;
    return theSymbol;
}
/*#__NO_SIDE_EFFECTS__*/
function polySymbolFor(key) {
    let registry = _globalSymbolRegistry();
    if (!objHasOwn(registry.k, key)) {
        let newSymbol = polyNewSymbol(key);
        let regId = objKeys(registry.s).length;
        newSymbol[UNIQUE_REGISTRY_ID] = () => regId + "_" + newSymbol[TO_STRING]();
        registry.k[key] = newSymbol;
        registry.s[newSymbol[UNIQUE_REGISTRY_ID]()] = asString(key);
    }
    return registry.k[key];
}
/*#__NO_SIDE_EFFECTS__*/
function polySymbolKeyFor(sym) {
    if (!sym || !sym[TO_STRING] || strSubstring(sym[TO_STRING](), 0, 6) != SYMBOL) {
        throwTypeError(sym + " is not a symbol");
    }
    const regId = sym[POLYFILL_TAG] && sym[UNIQUE_REGISTRY_ID] && sym[UNIQUE_REGISTRY_ID]();
    return regId ? _globalSymbolRegistry().s[regId] : undefined;
}
/*#__NO_SIDE_EFFECTS__*/
function polyGetKnownSymbol(name) {
    !_wellKnownSymbolCache && (_wellKnownSymbolCache = {});
    let result;
    let knownName = _wellKnownSymbolMap[name];
    if (knownName) {
        result = _wellKnownSymbolCache[knownName] = _wellKnownSymbolCache[knownName] || polyNewSymbol(SYMBOL + "." + knownName);
    }
    return result;
}

const propMap = {
    e: "enumerable",
    c: "configurable",
    v: VALUE,
    w: "writable",
    g: "get",
    s: "set"
};
/*#__NO_SIDE_EFFECTS__*/
function _createProp(value) {
    let prop = {};
    prop[propMap["c"]] = true;
    prop[propMap["e"]] = true;
    if (value.l) {
        prop.get = () => value.l.v;
        let desc = objGetOwnPropertyDescriptor(value.l, "v");
        if (desc && desc.set) {
            prop.set = (newValue) => {
                value.l.v = newValue;
            };
        }
    }
    objForEachKey(value, (key, value) => {
        prop[propMap[key]] = isUndefined(value) ? prop[propMap[key]] : value;
    });
    return prop;
}
const objDefineProp = ObjClass["defineProperty"];
const objDefineProperties = ObjClass["defineProperties"];
function objDefineGet(target, key, value, configurable, enumerable) {
    return objDefineProp(target, key, _createProp({
        e: enumerable,
        c: configurable,
        [isFunction(value) ? "g" : "v"]: value
    }));
}
function objDefineAccessors(target, prop, getProp, setProp, configurable, enumerable) {
    let desc = {
        e: enumerable,
        c: configurable
    };
    if (getProp) {
        desc.g = getProp;
    }
    if (setProp) {
        desc.s = setProp;
    }
    return objDefineProp(target, prop, _createProp(desc));
}
function objDefine(target, key, propDesc) {
    return objDefineProp(target, key, _createProp(propDesc));
}
function objDefineProps(target, propDescMap) {
    let props = {};
    objForEachKey(propDescMap, (key, value) => {
        props[key] = _createProp(value);
    });
    return objDefineProperties(target, props);
}

let _globalLazyTestHooks;
function _initTestHooks() {
    _globalLazyTestHooks = _getGlobalConfig();
}
function getLazy(cb) {
    let lazyValue = {};
    !_globalLazyTestHooks && _initTestHooks();
    lazyValue.b = _globalLazyTestHooks.lzy;
    objDefineProp(lazyValue, "v", {
        configurable: true,
        get: function () {
            let result = cb();
            if (!_globalLazyTestHooks.lzy) {
                objDefineProp(lazyValue, "v", {
                    value: result
                });
            }
            lazyValue.b = _globalLazyTestHooks.lzy;
            return result;
        }
    });
    return lazyValue;
}
function setBypassLazyCache(newValue) {
    !_globalLazyTestHooks && _initTestHooks();
    _globalLazyTestHooks.lzy = newValue;
}

/*#__NO_SIDE_EFFECTS__*/
function safeGetLazy(cb, defValue) {
    return getLazy(() => {
        let result = safe(cb);
        return result.e ? defValue : result.v;
    });
}

/*#__NO_SIDE_EFFECTS__*/
function createCachedValue(value) {
    return objDefineProp({
        toJSON: () => value
    }, "v", { value });
}
/*#__NO_SIDE_EFFECTS__*/
function createDeferredCachedValue(cb) {
    return objDefineProp({}, "v", {
        get: function () {
            let result = cb();
            objDefineProp(this, "v", {
                value: result,
                enumerable: true
            });
            return result;
        },
        enumerable: true,
        configurable: true
    });
}

const WINDOW = "window";
let _cachedGlobal;
let _cachedWindow;
let _cachedDocument;
let _cachedNavigator;
let _cachedHistory;
let _isWebWorker;
let _isNode;
/*#__NO_SIDE_EFFECTS__*/
function lazySafeGetInst(name) {
    return getLazy(() => safe((getInst), [name]).v || UNDEF_VALUE);
}
/*#__NO_SIDE_EFFECTS__*/
function getGlobal(useCached) {
    !_globalLazyTestHooks && _initTestHooks();
    (!_cachedGlobal || useCached === false || _globalLazyTestHooks.lzy) && (_cachedGlobal = createCachedValue(safe(_getGlobalValue).v || NULL_VALUE));
    return _cachedGlobal.v;
}
/*#__NO_SIDE_EFFECTS__*/
function getInst(name, useCached) {
    const gbl = (!_cachedGlobal || useCached === false) ? getGlobal(useCached) : _cachedGlobal.v;
    if (gbl && gbl[name]) {
        return gbl[name];
    }
    if (name === WINDOW && _cachedWindow) {
        return _cachedWindow.v;
    }
    return NULL_VALUE;
}
/*#__NO_SIDE_EFFECTS__*/
function hasDocument() {
    return !!getDocument();
}
/*#__NO_SIDE_EFFECTS__*/
function getDocument() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_cachedDocument || _globalLazyTestHooks.lzy) && (_cachedDocument = createCachedValue(safe((getInst), ["document"]).v));
    return _cachedDocument.v;
}
/*#__NO_SIDE_EFFECTS__*/
function hasWindow() {
    return !!getWindow();
}
/*#__NO_SIDE_EFFECTS__*/
function getWindow() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_cachedWindow || _globalLazyTestHooks.lzy) && (_cachedWindow = createCachedValue(safe((getInst), [WINDOW]).v));
    return _cachedWindow.v;
}
/*#__NO_SIDE_EFFECTS__*/
function hasNavigator() {
    return !!getNavigator();
}
/*#__NO_SIDE_EFFECTS__*/
function getNavigator() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_cachedNavigator || _globalLazyTestHooks.lzy) && (_cachedNavigator = createCachedValue(safe((getInst), ["navigator"]).v));
    return _cachedNavigator.v;
}
/*#__NO_SIDE_EFFECTS__*/
function hasHistory() {
    return !!getHistory();
}
/*#__NO_SIDE_EFFECTS__*/
function getHistory() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_cachedHistory || _globalLazyTestHooks.lzy) && (_cachedHistory = createCachedValue(safe((getInst), ["history"]).v));
    return _cachedHistory.v;
}
function isNode() {
    !_isNode && (_isNode = createCachedValue(!!safe(() => (process && (process.versions || {}).node)).v));
    return _isNode.v;
}
function isWebWorker() {
    !_isWebWorker && (_isWebWorker = createCachedValue(!!safe(() => self && self instanceof WorkerGlobalScope).v));
    return _isWebWorker.v;
}

let _symbol;
let _symbolFor;
let _symbolKeyFor;
function _initSymbol() {
    if (!_symbol || !_symbol.b) {
        _symbol = lazySafeGetInst(SYMBOL);
        _symbolFor = safeGetLazy(() => (_symbol.v ? _symbol.v["for"] : UNDEF_VALUE), UNDEF_VALUE);
        _symbolKeyFor = safeGetLazy(() => (_symbol.v ? _symbol.v["keyFor"] : UNDEF_VALUE), UNDEF_VALUE);
    }
}
const isSymbol = ( /*#__PURE__*/_createIs("symbol"));
/*#__NO_SIDE_EFFECTS__*/
function hasSymbol() {
    return !!getSymbol();
}
/*#__NO_SIDE_EFFECTS__*/
function getSymbol() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_symbol || _globalLazyTestHooks.lzy) && _initSymbol();
    return _symbol.v;
}
/*#__NO_SIDE_EFFECTS__*/
function getKnownSymbol(name, noPoly) {
    let knownName = _wellKnownSymbolMap[name];
    !_globalLazyTestHooks && _initTestHooks();
    (!_symbol || _globalLazyTestHooks.lzy) && _initSymbol();
    return _symbol.v ? _symbol.v[knownName || name] : (!noPoly ? polyGetKnownSymbol(name) : UNDEF_VALUE);
}
/*#__NO_SIDE_EFFECTS__*/
function newSymbol(description, noPoly) {
    !_globalLazyTestHooks && _initTestHooks();
    (!_symbol || _globalLazyTestHooks.lzy) && _initSymbol();
    return _symbol.v ? _symbol.v(description) : (!noPoly ? polyNewSymbol(description) : NULL_VALUE);
}
/*#__NO_SIDE_EFFECTS__*/
function symbolFor(key) {
    !_globalLazyTestHooks && _initTestHooks();
    (!_symbolFor || !_symbol || _globalLazyTestHooks.lzy) && _initSymbol();
    return (_symbolFor.v || polySymbolFor)(key);
}
/*#__NO_SIDE_EFFECTS__*/
function symbolKeyFor(sym) {
    !_globalLazyTestHooks && _initTestHooks();
    (!_symbolKeyFor || !_symbol || _globalLazyTestHooks.lzy) && _initSymbol();
    return (_symbolKeyFor.v || polySymbolKeyFor)(sym);
}

/*#__NO_SIDE_EFFECTS__*/
function isIterator(value) {
    return !!value && isFunction(value.next);
}
/*#__NO_SIDE_EFFECTS__*/
function isIterable(value) {
    return !isStrictNullOrUndefined(value) && isFunction(value[getKnownSymbol(3 )]);
}

let _iterSymbol$1;
function iterForOf(iter, callbackfn, thisArg) {
    if (iter) {
        if (!isIterator(iter)) {
            !_iterSymbol$1 && (_iterSymbol$1 = createCachedValue(getKnownSymbol(3 )));
            iter = iter[_iterSymbol$1.v] ? iter[_iterSymbol$1.v]() : null;
        }
        if (isIterator(iter)) {
            let err;
            let iterResult;
            try {
                let count = 0;
                while (!(iterResult = iter.next())[DONE]) {
                    if (callbackfn[CALL](thisArg || iter, iterResult[VALUE], count, iter) === -1) {
                        break;
                    }
                    count++;
                }
            }
            catch (failed) {
                err = { e: failed };
                if (iter.throw) {
                    iterResult = null;
                    iter.throw(err);
                }
            }
            finally {
                try {
                    if (iterResult && !iterResult[DONE]) {
                        iter.return && iter.return(iterResult);
                    }
                }
                finally {
                    if (err) {
                        throw err.e;
                    }
                }
            }
        }
    }
}

const fnApply = ( /*#__PURE__*/_unwrapInstFunction("apply"));

function arrAppend(target, elms) {
    if (!isUndefined(elms) && target) {
        if (isArray(elms)) {
            fnApply(target.push, target, elms);
        }
        else if (isIterator(elms) || isIterable(elms)) {
            iterForOf(elms, (elm) => {
                target.push(elm);
            });
        }
        else {
            target.push(elms);
        }
    }
    return target;
}

const arrEvery = /*#__PURE__*/ _unwrapFunction("every", ArrProto);
const arrFilter = /*#__PURE__*/ _unwrapFunction("filter", ArrProto);

function arrForEach(theArray, callbackfn, thisArg) {
    if (theArray) {
        const len = theArray[LENGTH] >>> 0;
        for (let idx = 0; idx < len; idx++) {
            if (idx in theArray) {
                if (callbackfn[CALL](thisArg || theArray, theArray[idx], idx, theArray) === -1) {
                    break;
                }
            }
        }
    }
}

const arrIndexOf = ( /*#__PURE__*/_unwrapFunction(INDEX_OF, ArrProto));
const arrLastIndexOf = ( /*#__PURE__*/_unwrapFunction(LAST_INDEX_OF, ArrProto));

const arrMap = ( /*#__PURE__*/_unwrapFunction("map", ArrProto));

const arrSlice = ( /*#__PURE__*/_unwrapFunction(SLICE, ArrProto));

/*#__NO_SIDE_EFFECTS__*/
function polyIsArray(value) {
    if (isNullOrUndefined(value)) {
        return false;
    }
    return objToString(value) === "[object Array]";
}
/*#__NO_SIDE_EFFECTS__*/
function polyArrIncludes(theArray, searchElement, fromIndex) {
    return arrIndexOf(theArray, searchElement, fromIndex) !== -1;
}
function polyArrFind(theArray, callbackFn, thisArg) {
    let result;
    let idx = polyArrFindIndex(theArray, callbackFn, thisArg);
    return idx !== -1 ? theArray[idx] : result;
}
function polyArrFindIndex(theArray, callbackFn, thisArg) {
    let result = -1;
    arrForEach(theArray, (value, index) => {
        if (callbackFn[CALL](thisArg | theArray, value, index, theArray)) {
            result = index;
            return -1;
        }
    });
    return result;
}
function polyArrFindLast(theArray, callbackFn, thisArg) {
    let result;
    let idx = polyArrFindLastIndex(theArray, callbackFn, thisArg);
    return idx !== -1 ? theArray[idx] : result;
}
function polyArrFindLastIndex(theArray, callbackFn, thisArg) {
    let result = -1;
    let len = theArray[LENGTH] >>> 0;
    for (let idx = len - 1; idx >= 0; idx--) {
        if (idx in theArray && callbackFn[CALL](thisArg | theArray, theArray[idx], idx, theArray)) {
            result = idx;
            break;
        }
    }
    return result;
}
function polyArrFrom(theValue, mapFn, thisArg) {
    if (isArray(theValue)) {
        let result = arrSlice(theValue);
        return mapFn ? arrMap(result, mapFn, thisArg) : result;
    }
    let result = [];
    iterForOf(theValue, (value, cnt) => {
        return result.push(mapFn ? mapFn[CALL](thisArg, value, cnt) : value);
    });
    return result;
}

const arrFind = /*#__PURE__*/ _unwrapFunctionWithPoly("find", ArrProto, polyArrFind);
const arrFindIndex = /*#__PURE__*/ _unwrapFunctionWithPoly("findIndex", ArrProto, polyArrFindIndex);
const arrFindLast = /*#__PURE__*/ _unwrapFunctionWithPoly("findLast", ArrProto, polyArrFindLast);
const arrFindLastIndex = /*#__PURE__*/ _unwrapFunctionWithPoly("findLastIndex", ArrProto, polyArrFindLastIndex);

const arrFrom = ArrCls.from || polyArrFrom;

const arrIncludes = ( /*#__PURE__*/_unwrapFunctionWithPoly("includes", ArrProto, polyArrIncludes));
const arrContains = arrIncludes;

const arrReduce = ( /*#__PURE__*/_unwrapFunction("reduce", ArrProto));

const arrSome = ( /*#__PURE__*/_unwrapFunction("some", ArrProto));

const fnBind = ( /*#__PURE__*/_unwrapInstFunction("bind"));

const fnCall = ( /*#__PURE__*/_unwrapInstFunction(CALL));

function createFnDeferredProxy(hostFn, funcName) {
    return function () {
        var theArgs = ArrProto[SLICE][CALL](arguments);
        let theHost = hostFn();
        return fnApply(theHost[funcName], theHost, theArgs);
    };
}
function createProxyFuncs(target, host, funcDefs) {
    if (target && host && isArray(funcDefs)) {
        let isDeferred = isFunction(host);
        arrForEach(funcDefs, (funcDef) => {
            let targetName = (funcDef.as || funcDef.n);
            if (funcDef.rp === false && target[targetName]) {
                return;
            }
            target[targetName] = isDeferred ?
                createFnDeferredProxy(host, funcDef.n) :
                fnBind(host[funcDef.n], host);
        });
    }
    return target;
}

let _iterSymbol;
/*#__NO_SIDE_EFFECTS__*/
function readArgs(theArgs, start, end) {
    if (!objHasOwn(theArgs, LENGTH)) {
        !_iterSymbol && (_iterSymbol = createCachedValue(hasSymbol() && getKnownSymbol(3 )));
        let iterFn = _iterSymbol.v && theArgs[_iterSymbol.v];
        if (iterFn) {
            let values = [];
            let from = (start === UNDEF_VALUE || start < 0) ? 0 : start;
            let to = end < 0 || start < 0 ? UNDEF_VALUE : end;
            iterForOf(iterFn[CALL](theArgs), (value, cnt) => {
                if (to !== UNDEF_VALUE && cnt >= to) {
                    return -1;
                }
                if (cnt >= from) {
                    values.push(value);
                }
            });
            if ((start === UNDEF_VALUE || start >= 0) && (end === UNDEF_VALUE || end >= 0)) {
                return values;
            }
            theArgs = values;
        }
    }
    return arrSlice(theArgs, start, end);
}

const objCreate = ObjClass["create"] || polyObjCreate;
/*#__NO_SIDE_EFFECTS__*/
function polyObjCreate(obj) {
    if (!obj) {
        return {};
    }
    let type = typeof obj;
    if (type !== OBJECT && type !== FUNCTION) {
        throwTypeError("Prototype must be an Object or function: " + dumpObj(obj));
    }
    function tempFunc() { }
    tempFunc[PROTOTYPE] = obj;
    return new tempFunc();
}

let _isProtoArray;
function objSetPrototypeOf(obj, proto) {
    let fn = ObjClass["setPrototypeOf"] ||
        function (d, b) {
            !_isProtoArray && (_isProtoArray = getLazy(() => ({ [__PROTO__]: [] } instanceof Array)));
            _isProtoArray.v ? d[__PROTO__] = b : objForEachKey(b, (key, value) => d[key] = value);
        };
    return fn(obj, proto);
}

function _createCustomError(name, d, b) {
    safe(objDefine, [d, NAME, { v: name, c: true, e: false }]);
    d = objSetPrototypeOf(d, b);
    function __() {
        this.constructor = d;
        safe(objDefine, [this, NAME, { v: name, c: true, e: false }]);
    }
    d[PROTOTYPE] = b === NULL_VALUE ? objCreate(b) : (__[PROTOTYPE] = b[PROTOTYPE], new __());
    return d;
}
function _setName(baseClass, name) {
    name && (baseClass[NAME] = name);
}
/*#__NO_SIDE_EFFECTS__*/
function createCustomError(name, constructCb, errorBase) {
    let theBaseClass = errorBase || Error;
    let orgName = theBaseClass[PROTOTYPE][NAME];
    let captureFn = Error.captureStackTrace;
    return _createCustomError(name, function () {
        let _this = this;
        let theArgs = arguments;
        try {
            safe(_setName, [theBaseClass, name]);
            let _self = fnApply(theBaseClass, _this, ArrProto[SLICE][CALL](theArgs)) || _this;
            if (_self !== _this) {
                let orgProto = objGetPrototypeOf(_this);
                if (orgProto !== objGetPrototypeOf(_self)) {
                    objSetPrototypeOf(_self, orgProto);
                }
            }
            captureFn && captureFn(_self, _this[CONSTRUCTOR]);
            constructCb && constructCb(_self, theArgs);
            return _self;
        }
        finally {
            safe(_setName, [theBaseClass, orgName]);
        }
    }, theBaseClass);
}
let _unsupportedError;
function throwUnsupported(message) {
    if (!_unsupportedError) {
        _unsupportedError = createCustomError("UnsupportedError");
    }
    throw new _unsupportedError(message);
}

/*#__NO_SIDE_EFFECTS__*/
function utcNow() {
    return (Date.now || polyUtcNow)();
}
/*#__NO_SIDE_EFFECTS__*/
function polyUtcNow() {
    return new Date().getTime();
}

/*#__NO_SIDE_EFFECTS__*/
function _createTrimFn(exp) {
    return function _doTrim(value) {
        if (isNullOrUndefined(value)) {
            throwTypeError("strTrim called [" + dumpObj(value) + "]");
        }
        if (value && value.replace) {
            value = value.replace(exp, EMPTY);
        }
        return value;
    };
}
const polyStrTrim = ( /*#__PURE__*/_createTrimFn(/^\s+|(?=\s)\s+$/g));
const polyStrTrimStart = ( /*#__PURE__*/_createTrimFn(/^\s+/g));
const polyStrTrimEnd = ( /*#__PURE__*/_createTrimFn(/(?=\s)\s+$/g));

const strTrim = ( /*#__PURE__*/_unwrapFunctionWithPoly("trim", StrProto, polyStrTrim));
const strTrimStart = ( /*#__PURE__*/_unwrapFunctionWithPoly("trimStart", StrProto, polyStrTrimStart));
const strTrimLeft = strTrimStart;
const strTrimEnd = ( /*#__PURE__*/_unwrapFunctionWithPoly("trimEnd", StrProto, polyStrTrimEnd));
const strTrimRight = strTrimEnd;

const strUpper = ( /*#__PURE__*/_unwrapFunction("toUpperCase", StrProto));
const strLower = ( /*#__PURE__*/_unwrapFunction("toLowerCase", StrProto));

/*#__NO_SIDE_EFFECTS__*/
function _convertCase(value, newPrefix, upperWord) {
    return strTrim(asString(value)).replace(/((_|\W)+(\w){0,1}|([a-z])([A-Z]))/g, (_match, _g1, _g2, wordStart, upperPrefix, upperLetter) => {
        let convertMatch = wordStart || upperLetter || EMPTY;
        if (upperWord) {
            convertMatch = strUpper(convertMatch);
        }
        return (upperPrefix || EMPTY) + newPrefix + convertMatch;
    });
}
/*#__NO_SIDE_EFFECTS__*/
function strLetterCase(value) {
    return asString(value).replace(/(_|\b)\w/g, strUpper);
}
/*#__NO_SIDE_EFFECTS__*/
function strCamelCase(value, upperFirst) {
    let result = _convertCase(value, "", true);
    return result.replace(/^\w/, upperFirst ? strUpper : strLower);
}
/*#__NO_SIDE_EFFECTS__*/
function strKebabCase(value, scream) {
    let result = _convertCase(value, "-");
    return (scream ? strUpper : strLower)(result);
}
/*#__NO_SIDE_EFFECTS__*/
function strSnakeCase(value, scream) {
    let result = _convertCase(value, "_");
    return (scream ? strUpper : strLower)(result);
}

const mathFloor = MathCls.floor;
const mathCeil = MathCls.ceil;

const mathTrunc = MathCls.trunc || polyMathTrunc;
/*#__NO_SIDE_EFFECTS__*/
function polyMathTrunc(value) {
    let theValue = +value;
    return (theValue > 0 ? mathFloor : mathCeil)(theValue);
}

/*#__NO_SIDE_EFFECTS__*/
function mathToInt(value, throwInfinity) {
    let result = +value;
    if (result == Infinity && throwInfinity) {
        throwRangeError("invalid value [" + dumpObj(value) + "]");
    }
    return result !== result || result === 0 ? 0 : mathTrunc(result);
}

const strRepeat = ( /*#__PURE__*/_unwrapFunctionWithPoly("repeat", StrProto, polyStrRepeat));
/*#__NO_SIDE_EFFECTS__*/
function polyStrRepeat(value, count) {
    if (isNullOrUndefined(value)) {
        throwTypeError("can't convert [" + dumpObj(value) + "]");
    }
    count = mathToInt(count, true);
    if (count < 0) {
        throwRangeError("invalid count must be >= 0 && < Infinity");
    }
    let pad = isString(value) ? value : asString(value);
    let result = EMPTY;
    for (; count > 0; (count >>>= 1) && (pad += pad)) {
        if (count & 1) {
            result += pad;
        }
    }
    return result;
}

/*#__NO_SIDE_EFFECTS__*/
function _padValue(value, targetLength, padString) {
    let result = EMPTY;
    targetLength = mathToInt(targetLength, true);
    targetLength >>= 0;
    let len = value[LENGTH];
    if (len < targetLength) {
        result = isNullOrUndefined(padString) ? " " : asString(padString);
        targetLength = targetLength - len;
        if (targetLength > result[LENGTH]) {
            result = strRepeat(result, mathCeil(targetLength / result[LENGTH]));
        }
        if (result[LENGTH] > targetLength) {
            result = strSubstring(result, 0, targetLength);
        }
    }
    return result;
}
const strPadStart = ( /*#__PURE__*/_unwrapFunctionWithPoly("padStart", StrProto, polyStrPadStart));
const strPadEnd = ( /*#__PURE__*/_unwrapFunctionWithPoly("padEnd", StrProto, polyStrPadEnd));
/*#__NO_SIDE_EFFECTS__*/
function polyStrPadStart(value, targetLength, padString) {
    return _padValue(value, targetLength, padString) + value;
}
/*#__NO_SIDE_EFFECTS__*/
function polyStrPadEnd(value, targetLength, padString) {
    return value + _padValue(value, targetLength, padString);
}

const DBL_QUOTE = "\"";
const INVALID_JS_NAME = /([^\w\d_$])/g;
let _htmlEntityCache;
/*#__NO_SIDE_EFFECTS__*/
function normalizeJsName(jsName, camelCase) {
    let result = asString(jsName).replace(INVALID_JS_NAME, "_");
    return !isUndefined(camelCase) ? strCamelCase(result, !camelCase) : result;
}
/*#__NO_SIDE_EFFECTS__*/
function encodeAsJson(value, format) {
    let result;
    if (isString(value)) {
        result = DBL_QUOTE + value.replace(/[^\w .,\-!@#$%\^&*\(\)_+={}\[\]:;|<>?]/g, (match) => {
            if (match === DBL_QUOTE || match === "\\") {
                return "\\" + match;
            }
            var hex = match.charCodeAt(0)[TO_STRING](16);
            return "\\u" + strPadStart(strUpper(hex), 4, "0");
        }) + DBL_QUOTE;
    }
    else {
        try {
            result = JSON.stringify(value, NULL_VALUE, format ? (isNumber(format) ? format : 4) : UNDEF_VALUE);
        }
        catch (e) {
            result = DBL_QUOTE + dumpObj(e) + DBL_QUOTE;
        }
    }
    return result;
}
/*#__NO_SIDE_EFFECTS__*/
function encodeAsHtml(value) {
    !_htmlEntityCache && (_htmlEntityCache = getLazy(() => {
        return {
            "&": "amp",
            "<": "lt",
            ">": "gt",
            [DBL_QUOTE]: "quot",
            "'": "#39"
        };
    }));
    return asString(value).replace(/[&<>"']/g, match => "&" + _htmlEntityCache.v[match] + ";");
}

let _fnToString;
let _objCtrFnString;
let _gblWindow;
/*#__NO_SIDE_EFFECTS__*/
function isPlainObject(value) {
    if (!value || typeof value !== OBJECT) {
        return false;
    }
    if (!_gblWindow) {
        _gblWindow = hasWindow() ? getWindow() : true;
    }
    let result = false;
    if (value !== _gblWindow) {
        if (!_objCtrFnString) {
            _fnToString = Function[PROTOTYPE][TO_STRING];
            _objCtrFnString = _fnToString[CALL](ObjClass);
        }
        try {
            let proto = objGetPrototypeOf(value);
            result = !proto;
            if (!result) {
                if (objHasOwnProperty(proto, CONSTRUCTOR)) {
                    proto = proto[CONSTRUCTOR];
                }
                result = proto && typeof proto === FUNCTION && _fnToString[CALL](proto) === _objCtrFnString;
            }
        }
        catch (ex) {
        }
    }
    return result;
}

const _defaultDeepCopyHandler = (details) => {
    details.value && plainObjDeepCopyHandler(details);
    return true;
};
const defaultDeepCopyHandlers = [
    arrayDeepCopyHandler,
    plainObjDeepCopyHandler,
    functionDeepCopyHandler,
    dateDeepCopyHandler
];
function _getSetVisited(visitMap, source, newPath, cb) {
    let theEntry;
    arrForEach(visitMap, (entry) => {
        if (entry.k === source) {
            theEntry = entry;
            return -1;
        }
    });
    if (!theEntry) {
        theEntry = { k: source, v: source };
        visitMap.push(theEntry);
        cb(theEntry);
    }
    return theEntry.v;
}
function _deepCopy(visitMap, value, ctx, key) {
    let userHandler = ctx.handler;
    let newPath = ctx.path ? (key ? ctx.path.concat(key) : ctx.path) : [];
    let newCtx = {
        handler: ctx.handler,
        src: ctx.src,
        path: newPath
    };
    const theType = typeof value;
    let isPlain = false;
    let isPrim = false;
    if (value && theType === OBJECT) {
        isPlain = isPlainObject(value);
    }
    else {
        isPrim = value === NULL_VALUE || isPrimitiveType(theType);
    }
    let details = {
        type: theType,
        isPrim: isPrim,
        isPlain: isPlain,
        value: value,
        result: value,
        path: newPath,
        origin: ctx.src,
        copy: (source, newKey) => {
            return _deepCopy(visitMap, source, newKey ? newCtx : ctx, newKey);
        },
        copyTo: (target, source) => {
            return _copyProps(visitMap, target, source, newCtx);
        }
    };
    if (!details.isPrim) {
        return _getSetVisited(visitMap, value, newPath, (newEntry) => {
            objDefine(details, "result", {
                g: function () {
                    return newEntry.v;
                },
                s: function (newValue) {
                    newEntry.v = newValue;
                }
            });
            let idx = 0;
            let handler = userHandler;
            while (!(handler || (idx < defaultDeepCopyHandlers.length ? defaultDeepCopyHandlers[idx++] : _defaultDeepCopyHandler))[CALL](ctx, details)) {
                handler = NULL_VALUE;
            }
        });
    }
    if (userHandler && userHandler[CALL](ctx, details)) {
        return details.result;
    }
    return value;
}
function _copyProps(visitMap, target, source, ctx) {
    if (!isNullOrUndefined(source)) {
        for (const key in source) {
            target[key] = _deepCopy(visitMap, source[key], ctx, key);
        }
    }
    return target;
}
function objCopyProps(target, source, handler) {
    let ctx = {
        handler: handler,
        src: source,
        path: []
    };
    return _copyProps([], target, source, ctx);
}
/*#__NO_SIDE_EFFECTS__*/
function objDeepCopy(source, handler) {
    let ctx = {
        handler: handler,
        src: source
    };
    return _deepCopy([], source, ctx);
}
function arrayDeepCopyHandler(details) {
    let value = details.value;
    if (isArray(value)) {
        let target = details.result = [];
        target.length = value.length;
        details.copyTo(target, value);
        return true;
    }
    return false;
}
function dateDeepCopyHandler(details) {
    let value = details.value;
    if (isDate(value)) {
        details.result = new Date(value.getTime());
        return true;
    }
    return false;
}
function functionDeepCopyHandler(details) {
    if (details.type === FUNCTION) {
        return true;
    }
    return false;
}
function plainObjDeepCopyHandler(details) {
    let value = details.value;
    if (value && details.isPlain) {
        let target = details.result = {};
        details.copyTo(target, value);
        return true;
    }
    return false;
}

function _doExtend(target, theArgs) {
    arrForEach(theArgs, (theArg) => {
        objCopyProps(target, theArg);
    });
    return target;
}
function deepExtend(target, obj1, obj2, obj3, obj4, obj5, obj6) {
    return _doExtend(objDeepCopy(target) || {}, ArrProto[SLICE][CALL](arguments));
}
function objExtend(target, obj1, obj2, obj3, obj4, obj5, obj6) {
    return _doExtend(target || {}, ArrProto[SLICE][CALL](arguments));
}

const getLength = ( /*#__PURE__*/_unwrapProp(LENGTH));

/*#__NO_SIDE_EFFECTS__*/
function getIntValue(value, defValue) {
    if (!isNullOrUndefined(value)) {
        if (isNumber(value)) {
            return value;
        }
        let theValue = parseInt(value, 10);
        return isNaN(theValue) ? defValue : theValue;
    }
    return defValue;
}

let _perf;
/*#__NO_SIDE_EFFECTS__*/
function hasPerformance() {
    return !!getPerformance();
}
/*#__NO_SIDE_EFFECTS__*/
function getPerformance() {
    !_globalLazyTestHooks && _initTestHooks();
    (!_perf || (!_perf.b && _globalLazyTestHooks.lzy)) && (_perf = lazySafeGetInst("performance"));
    return _perf.v;
}
/*#__NO_SIDE_EFFECTS__*/
function perfNow() {
    let perf = getPerformance();
    if (perf && perf.now) {
        return perf.now();
    }
    return utcNow();
}
/*#__NO_SIDE_EFFECTS__*/
function elapsedTime(startTime) {
    return perfNow() - startTime;
}

const MATCH_ANY = "(.*)";
const MATCH_SINGLE = "(.)";
function _createRegExp(value, escapeRgx, replaceFn, ignoreCase, fullMatch) {
    return new RegExp((fullMatch ? "^" : EMPTY) + replaceFn(value.replace(escapeRgx, "\\$1")) + (fullMatch ? "$" : EMPTY), ignoreCase ? "i" : "");
}
/*#__NO_SIDE_EFFECTS__*/
function createWildcardRegex(value, ignoreCase, fullMatch) {
    return _createRegExp(asString(value), /([-+|^$#.\?{}()\[\]\\/\"\'])/g, (value) => {
        return value.replace(/\*/g, MATCH_ANY);
    }, !!ignoreCase, fullMatch);
}
/*#__NO_SIDE_EFFECTS__*/
function createFilenameRegex(value, ignoreCase, fullMatch) {
    return _createRegExp(asString(value), /([-+|^$#.{}()\\\/\[\]\"\'])/g, (value) => {
        return value.replace(/(\\\\|\\\/|\*|\?)/g, function (_all, g1) {
            if (g1 == "\\/" || g1 == "\\\\") {
                return "[\\\\\\/]{1}";
            }
            return g1 == "*" ? MATCH_ANY : MATCH_SINGLE;
        });
    }, !!ignoreCase, fullMatch);
}
/*#__NO_SIDE_EFFECTS__*/
function makeGlobRegex(value, ignoreCase, fullMatch) {
    return _createRegExp(asString(value), /([-+|^$#.{}()\\\/\[\]\"\'])/g, (value) => {
        return value.replace(/(\*\*\\[\\\/]|\\\\|\\\/|\*\*|\*|\?)/g, function (_all, g1) {
            if (g1 == "**\\/" || g1 == "**\\\\") {
                return "(.*[\\\\\\/])*";
            }
            if (g1 === "\\/" || g1 == "\\\\") {
                return "[\\\\\\/]{1}";
            }
            if (g1 === "**") {
                return MATCH_ANY;
            }
            return g1 === "*" ? "([^\\\\\\/]*)" : "([^\\\\\\/]{1})";
        });
    }, !!ignoreCase, fullMatch);
}

/*#__NO_SIDE_EFFECTS__*/
function _checkLength(value, props) {
    let result;
    arrForEach(props, (prop) => {
        if (prop in value) {
            let propValue = value[prop];
            result = (isFunction(propValue) ? propValue() : propValue) > 0;
            return -1;
        }
    });
    return result;
}
/*#__NO_SIDE_EFFECTS__*/
function _hasValue(value, depth) {
    let result = value === false || value === 0;
    if (!result && !isNullOrUndefined(value)) {
        if (isArray(value)) {
            result = value[LENGTH] > 0;
        }
        else if (isDate(value)) {
            result = !isNaN(value.getTime());
        }
        else if (isBoolean(value)) {
            return true;
        }
        else if (isObject(value)) {
            try {
                let chkValue = _checkLength(value, [LENGTH, "byteLength", "size", "count"]);
                if (isBoolean(chkValue)) {
                    return chkValue;
                }
                if (isFunction(value.valueOf) && depth < 5) {
                    return _hasValue(value.valueOf(), ++depth);
                }
            }
            catch (e) {
            }
            return !!objKeys(value)[LENGTH];
        }
        else {
            result = isTruthy(value);
        }
    }
    return result;
}
/*#__NO_SIDE_EFFECTS__*/
function hasValue(value) {
    return _hasValue(value, 0);
}

/*#__NO_SIDE_EFFECTS__*/
function createIterable(ctx) {
    return makeIterable({}, ctx);
}
function makeIterable(target, ctx) {
    let itSymbol = getKnownSymbol(3 );
    function _createIterator() {
        return createIterator(ctx);
    }
    target[itSymbol] = _createIterator;
    return target;
}
/*#__NO_SIDE_EFFECTS__*/
function createIterator(ctx) {
    let isDone = false;
    function _value() {
        return ctx.v;
    }
    function _next() {
        isDone = isDone || (ctx.n ? ctx.n(arguments) : true);
        let result = {
            [DONE]: isDone
        };
        if (!isDone) {
            objDefine(result, VALUE, { g: _value });
        }
        return result;
    }
    function _return(value) {
        isDone = true;
        return {
            [DONE]: true,
            [VALUE]: ctx.r && ctx.r(value)
        };
    }
    function _throw(e) {
        isDone = true;
        return {
            [DONE]: true,
            [VALUE]: ctx.t && ctx.t(e)
        };
    }
    let theIterator = {
        next: _next
    };
    if (ctx.r) {
        theIterator.return = _return;
    }
    if (ctx.t) {
        theIterator.throw = _throw;
    }
    return theIterator;
}

/*#__NO_SIDE_EFFECTS__*/
function createArrayIterator(values) {
    let idx = -1;
    let theValues = values ? values.slice() : [];
    let len = theValues[LENGTH];
    function _value() {
        if (idx >= 0 && idx < len) {
            return theValues[idx];
        }
    }
    function _getNext() {
        idx++;
        return idx >= len;
    }
    let ctx = {
        n: _getNext
    };
    objDefine(ctx, "v", { g: _value });
    return createIterator(ctx);
}

/*#__NO_SIDE_EFFECTS__*/
function createRangeIterator(start, end, step) {
    let nextValue = start;
    let theValue = UNDEF_VALUE;
    if (isNullOrUndefined(end)) {
        end = start;
    }
    let theStep = step || ((start <= end) ? 1 : -1);
    function _value() {
        return theValue;
    }
    function _getNext() {
        let isDone = false;
        if (theStep > 0) {
            isDone = nextValue > end;
        }
        else {
            isDone = nextValue < end;
        }
        if (!isDone) {
            theValue = nextValue;
            nextValue += theStep;
        }
        return isDone;
    }
    let ctx = {
        n: _getNext
    };
    objDefine(ctx, "v", { g: _value });
    return createIterator(ctx);
}

/*#__NO_SIDE_EFFECTS__*/
function polyStrSymSplit(value, splitter, limit) {
    let splitFn = splitter && splitter[getKnownSymbol(9 )];
    return splitFn ? splitFn(value, limit) : [value];
}

const strSplit = ( /*#__PURE__*/_unwrapFunction("split", StrProto));
const strSymSplit = ( /*#__PURE__*/_unwrapFunctionWithPoly("split", StrProto, !hasSymbol() ? polyStrSymSplit : null));

/*#__NO_SIDE_EFFECTS__*/
function getValueByKey(target, path, defValue) {
    if (!path || !target) {
        return defValue;
    }
    let parts = strSplit(path, ".");
    let cnt = parts.length;
    for (let lp = 0; lp < cnt && !isNullOrUndefined(target); lp++) {
        target = target[parts[lp]];
    }
    return (!isNullOrUndefined(target) ? target : defValue);
}
/*#__NO_SIDE_EFFECTS__*/
function getValueByIter(target, iter, defValue) {
    if (!iter || !target) {
        return defValue;
    }
    iterForOf(iter, (value) => {
        if (isNullOrUndefined(target)) {
            return -1;
        }
        target = target[value];
    });
    return (!isNullOrUndefined(target) ? target : defValue);
}
function setValueByKey(target, path, value) {
    if (target && path) {
        let parts = strSplit(path, ".");
        let lastKey = parts.pop();
        arrForEach(parts, (key) => {
            if (isNullOrUndefined(target[key])) {
                target[key] = {};
            }
            target = target[key];
        });
        target[lastKey] = value;
    }
}
function setValueByIter(target, iter, value) {
    if (target && iter) {
        let lastKey;
        iterForOf(iter, (key) => {
            if (lastKey) {
                if (isNullOrUndefined(target[lastKey])) {
                    target[lastKey] = {};
                }
                target = target[lastKey];
            }
            lastKey = key;
        });
        target[lastKey] = value;
    }
}

const strEndsWith = ( /*#__PURE__*/_unwrapFunctionWithPoly("endsWith", StrProto, polyStrEndsWith));
/*#__NO_SIDE_EFFECTS__*/
function polyStrEndsWith(value, searchString, length) {
    if (!isString(value)) {
        throwTypeError("'" + dumpObj(value) + "' is not a string");
    }
    let searchValue = isString(searchString) ? searchString : asString(searchString);
    let end = !isUndefined(length) && length < value[LENGTH] ? length : value[LENGTH];
    return strSubstring(value, end - searchValue[LENGTH], end) === searchValue;
}

const strIndexOf = ( /*#__PURE__*/_unwrapFunction(INDEX_OF, StrProto));
const strLastIndexOf = ( /*#__PURE__*/_unwrapFunction(LAST_INDEX_OF, StrProto));

const strIncludes = ( /*#__PURE__*/_unwrapFunctionWithPoly("includes", StrProto, polyStrIncludes));
const strContains = strIncludes;
/*#__NO_SIDE_EFFECTS__*/
function polyStrIncludes(value, searchString, position) {
    if (isRegExp(searchString)) {
        throwTypeError("'searchString' must not be a regular expression" + dumpObj(searchString));
    }
    return strIndexOf(value, asString(searchString), position) !== -1;
}

/*#__NO_SIDE_EFFECTS__*/
function strIsNullOrWhiteSpace(value) {
    if (isString(value)) {
        return value.replace(/[\s\t\r\n\f]+/g, EMPTY) === EMPTY;
    }
    return isNullOrUndefined(value);
}
/*#__NO_SIDE_EFFECTS__*/
function strIsNullOrEmpty(value) {
    if (isString(value)) {
        return value === EMPTY;
    }
    return isNullOrUndefined(value);
}

const strStartsWith = ( /*#__PURE__*/_unwrapFunctionWithPoly("startsWith", StrProto, polyStrStartsWith));
/*#__NO_SIDE_EFFECTS__*/
function polyStrStartsWith(value, searchString, position) {
    if (!isString(value)) {
        throwTypeError("'" + dumpObj(value) + "' is not a string");
    }
    let searchValue = isString(searchString) ? searchString : asString(searchString);
    let pos = position > 0 ? position : 0;
    return strSubstring(value, pos, pos + searchValue[LENGTH]) === searchValue;
}

const REF = "ref";
const UNREF = "un" + REF;
const HAS_REF = "hasRef";
const ENABLED = "enabled";
function _createTimerHandler(startTimer, refreshFn, cancelFn) {
    let ref = true;
    let timerId = startTimer ? refreshFn(NULL_VALUE) : NULL_VALUE;
    let theTimerHandler;
    const _unref = () => {
        ref = false;
        timerId && timerId[UNREF] && timerId[UNREF]();
        return theTimerHandler;
    };
    const _ref = () => {
        ref = true;
        timerId && timerId[REF] && timerId[REF]();
        return theTimerHandler;
    };
    const _hasRef = () => {
        if (timerId && timerId[HAS_REF]) {
            return timerId[HAS_REF]();
        }
        return ref;
    };
    const _refresh = () => {
        timerId = refreshFn(timerId);
        if (!ref) {
            _unref();
        }
        return theTimerHandler;
    };
    const _cancel = () => {
        timerId && cancelFn(timerId);
        timerId = NULL_VALUE;
    };
    const _setEnabled = (value) => {
        !value && timerId && _cancel();
        value && !timerId && _refresh();
    };
    theTimerHandler = {
        cancel: _cancel,
        refresh: _refresh,
        [HAS_REF]: _hasRef,
        [REF]: _ref,
        [UNREF]: _unref,
        [ENABLED]: false
    };
    objDefineProp(theTimerHandler, ENABLED, {
        get: () => !!timerId,
        set: _setEnabled
    });
    return {
        h: theTimerHandler,
        dn: () => {
            timerId = NULL_VALUE;
        }
    };
}

function _createTimeoutWith(self, startTimer, overrideFn, theArgs) {
    let isArr = isArray(overrideFn);
    let len = isArr ? overrideFn.length : 0;
    let setFn = (len > 0 ? overrideFn[0] : (!isArr ? overrideFn : UNDEF_VALUE)) || setTimeout;
    let clearFn = (len > 1 ? overrideFn[1] : UNDEF_VALUE) || clearTimeout;
    let timerFn = theArgs[0];
    theArgs[0] = function () {
        handler.dn();
        fnApply(timerFn, self, ArrProto[SLICE][CALL](arguments));
    };
    let handler = _createTimerHandler(startTimer, (timerId) => {
        if (timerId) {
            if (timerId.refresh) {
                timerId.refresh();
                return timerId;
            }
            fnApply(clearFn, self, [timerId]);
        }
        return fnApply(setFn, self, theArgs);
    }, function (timerId) {
        fnApply(clearFn, self, [timerId]);
    });
    return handler.h;
}
function scheduleTimeout(callback, timeout) {
    return _createTimeoutWith(this, true, UNDEF_VALUE, ArrProto[SLICE][CALL](arguments));
}
function scheduleTimeoutWith(overrideFn, callback, timeout) {
    return _createTimeoutWith(this, true, overrideFn, ArrProto[SLICE][CALL](arguments, 1));
}
function createTimeout(callback, timeout) {
    return _createTimeoutWith(this, false, UNDEF_VALUE, ArrProto[SLICE][CALL](arguments));
}
function createTimeoutWith(overrideFn, callback, timeout) {
    return _createTimeoutWith(this, false, overrideFn, ArrProto[SLICE][CALL](arguments, 1));
}

let _hasIdleCallback;
let _defaultIdleTimeout = 100;
let _maxExecutionTime = 50;
/*#__NO_SIDE_EFFECTS__*/
function hasIdleCallback() {
    !_hasIdleCallback && (_hasIdleCallback = safeGetLazy(() => isDefined(requestIdleCallback), false));
    return !!(_hasIdleCallback.v ? requestIdleCallback : false);
}
function setDefaultIdleTimeout(timeout) {
    _defaultIdleTimeout = timeout;
}
function setDefaultMaxExecutionTime(maxTime) {
    _maxExecutionTime = maxTime;
}
function scheduleIdleCallback(callback, options) {
    function _createDeadline(timedOut) {
        let startTime = perfNow();
        return {
            didTimeout: timedOut,
            timeRemaining: () => {
                return _maxExecutionTime - elapsedTime(startTime);
            }
        };
    }
    if (hasIdleCallback()) {
        let handler = _createTimerHandler(true, (idleId) => {
            idleId && cancelIdleCallback(idleId);
            return requestIdleCallback((deadline) => {
                handler.dn();
                callback(deadline || _createDeadline(false));
            }, options);
        }, (idleId) => {
            cancelIdleCallback(idleId);
        });
        return handler.h;
    }
    let timeout = (options || {}).timeout;
    if (isUndefined(timeout)) {
        timeout = _defaultIdleTimeout;
    }
    return scheduleTimeout(() => {
        callback(_createDeadline(true));
    }, timeout);
}

function scheduleInterval(callback, timeout) {
    let self = this;
    let theArguments = ArrProto[SLICE][CALL](arguments);
    let handler = _createTimerHandler(true, (intervalId) => {
        intervalId && clearInterval(intervalId);
        return fnApply(setInterval, self, theArguments);
    }, (intervalId) => {
        fnApply(clearInterval, self, [intervalId]);
    });
    return handler.h;
}

export { arrAppend, arrContains, arrEvery, arrFilter, arrFind, arrFindIndex, arrFindLast, arrFindLastIndex, arrForEach, arrFrom, arrIncludes, arrIndexOf, arrLastIndexOf, arrMap, arrReduce, arrSlice, arrSome, arrayDeepCopyHandler, asString, createArrayIterator, createCachedValue, createCustomError, createDeferredCachedValue, createEnum, createEnumKeyMap, createEnumValueMap, createFilenameRegex, createFnDeferredProxy, createIterable, createIterator, createProxyFuncs, createRangeIterator, createSimpleMap, createTimeout, createTimeoutWith, createTypeMap, createWildcardRegex, dateDeepCopyHandler, deepExtend, dumpObj, elapsedTime, encodeAsHtml, encodeAsJson, fnApply, fnBind, fnCall, functionDeepCopyHandler, getDocument, getGlobal, getHistory, getInst, getIntValue, getKnownSymbol, getLazy, getLength, getNavigator, getPerformance, getSymbol, getValueByIter, getValueByKey, getWindow, hasDocument, hasHistory, hasIdleCallback, hasNavigator, hasPerformance, hasSymbol, hasValue, hasWindow, isArray, isArrayBuffer, isBlob, isBoolean, isDate, isDefined, isError, isFile, isFormData, isFunction, isIterable, isIterator, isNode, isNotTruthy, isNullOrUndefined, isNumber, isObject, isPlainObject, isPrimitive, isPrimitiveType, isPromise, isPromiseLike, isRegExp, isStrictNullOrUndefined, isStrictUndefined, isString, isSymbol, isThenable, isTruthy, isTypeof, isUndefined, isWebWorker, iterForOf, lazySafeGetInst, makeGlobRegex, makeIterable, mathCeil, mathFloor, mathMax, mathMin, mathToInt, mathTrunc, newSymbol, normalizeJsName, objAssign, objCopyProps, objCreate, objDeepCopy, objDeepFreeze, objDefine, objDefineAccessors, objDefineGet, objDefineProp, objDefineProperties, objDefineProps, objEntries, objExtend, objForEachKey, objFreeze, objGetOwnPropertyDescriptor, objGetPrototypeOf, objHasOwn, objHasOwnProperty, objKeys, objSeal, objSetPrototypeOf, objToString, perfNow, plainObjDeepCopyHandler, polyArrFind, polyArrFindIndex, polyArrFindLast, polyArrFindLastIndex, polyArrFrom, polyArrIncludes, polyGetKnownSymbol, polyIsArray, polyNewSymbol, polyObjEntries, polyObjHasOwn, polyObjKeys, polyStrIncludes, polyStrSubstr, polyStrSymSplit, polyStrTrim, polyStrTrimEnd, polyStrTrimStart, polySymbolFor, polySymbolKeyFor, polyUtcNow, readArgs, safe, safeGet, safeGetLazy, scheduleIdleCallback, scheduleInterval, scheduleTimeout, scheduleTimeoutWith, setBypassLazyCache, setDefaultIdleTimeout, setDefaultMaxExecutionTime, setValueByIter, setValueByKey, strCamelCase, strContains, strEndsWith, strIncludes, strIndexOf, strIsNullOrEmpty, strIsNullOrWhiteSpace, strKebabCase, strLastIndexOf, strLeft, strLetterCase, strLower, strPadEnd, strPadStart, strRepeat, strRight, strSlice, strSnakeCase, strSplit, strStartsWith, strSubstr, strSubstring, strSymSplit, strTrim, strTrimEnd, strTrimLeft, strTrimRight, strTrimStart, strUpper, symbolFor, symbolKeyFor, throwError, throwRangeError, throwTypeError, throwUnsupported, utcNow };
//# sourceMappingURL=ts-utils.js.map
