"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.beacon = beacon;
exports.getSubmitMethod = getSubmitMethod;
exports.xhr = xhr;
var _runtime = require("../constants/runtime");
/**
 * @file Contains common methods used to transmit harvested data.
 * @copyright 2023 New Relic Corporation. All rights reserved.
 * @license Apache-2.0
 */

/**
 * @typedef {xhr|beacon} NetworkMethods
 */

/**
 * Determines the submit method to use based on options.
 * @param {object} opts Options used to determine submit method.
 * @param {boolean} opts.isFinalHarvest Indicates if the data submission is due to
 * a final harvest within the agent.
 */
function getSubmitMethod() {
  let {
    isFinalHarvest = false
  } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  return isFinalHarvest && _runtime.isBrowserScope && _runtime.supportsSendBeacon
  // Use sendBeacon for final harvest
  ? beacon
  // Only IE does not support sendBeacon for final harvest
  // If not final harvest, or not browserScope, always use xhr post
  : xhr;
}

/**
 * Send via XHR
 * @param {Object} args - The args.
 * @param {string} args.url - The URL to send to.
 * @param {string=} args.body - The Stringified body. Default null to prevent IE11 from breaking.
 * @param {boolean=} args.sync - Run XHR synchronously.
 * @param {string=} [args.method=POST] - The XHR method to use.
 * @param {{key: string, value: string}[]} [args.headers] - The headers to attach.
 * @returns {XMLHttpRequest}
 */
function xhr(_ref) {
  let {
    url,
    body = null,
    sync,
    method = 'POST',
    headers = [{
      key: 'content-type',
      value: 'text/plain'
    }]
  } = _ref;
  const request = new XMLHttpRequest();
  request.open(method, url, !sync);
  try {
    // Set cookie
    if ('withCredentials' in request) request.withCredentials = true;
  } catch (e) {
    // do nothing
  }
  headers.forEach(header => {
    request.setRequestHeader(header.key, header.value);
  });
  request.send(body);
  return request;
}

/**
 * Send via sendBeacon. Do NOT call this function outside of a guaranteed web window environment.
 * @param {Object} args - The args
 * @param {string} args.url - The URL to send to
 * @param {string=} args.body - The Stringified body
 * @returns {boolean}
 */
function beacon(_ref2) {
  let {
    url,
    body
  } = _ref2;
  try {
    // Navigator has to be bound to ensure it does not error in some browsers
    // https://xgwang.me/posts/you-may-not-know-beacon/#it-may-throw-error%2C-be-sure-to-catch
    const send = window.navigator.sendBeacon.bind(window.navigator);
    return send(url, body);
  } catch (err) {
    // if sendBeacon still trys to throw an illegal invocation error,
    // we can catch here and return
    return false;
  }
}