"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Aggregate = void 0;
var _canonicalFunctionName = require("./canonical-function-name");
var _computeStackTrace = require("./compute-stack-trace");
var _stringHashCode = require("./string-hash-code");
var _formatStackTrace = require("./format-stack-trace");
var _registerHandler = require("../../../common/event-emitter/register-handler");
var _harvestScheduler = require("../../../common/harvest/harvest-scheduler");
var _stringify = require("../../../common/util/stringify");
var _handle = require("../../../common/event-emitter/handle");
var _mapOwn = require("../../../common/util/map-own");
var _config = require("../../../common/config/config");
var _now = require("../../../common/timing/now");
var _runtime = require("../../../common/constants/runtime");
var _constants = require("../constants");
var _features = require("../../../loaders/features/features");
var _aggregateBase = require("../../utils/aggregate-base");
var _nreum = require("../../../common/window/nreum");
var _drain = require("../../../common/drain/drain");
/*
 * Copyright 2020 New Relic Corporation. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @typedef {import('./compute-stack-trace.js').StackInfo} StackInfo
 */

class Aggregate extends _aggregateBase.AggregateBase {
  static featureName = _constants.FEATURE_NAME;
  constructor(agentIdentifier, aggregator) {
    var _this;
    super(agentIdentifier, aggregator, _constants.FEATURE_NAME);
    _this = this;
    this.stackReported = {};
    this.observedAt = {};
    this.pageviewReported = {};
    this.bufferedErrorsUnderSpa = {};
    this.currentBody = undefined;
    this.errorOnPage = false;

    // this will need to change to match whatever ee we use in the instrument
    this.ee.on('interactionDone', (interaction, wasSaved) => this.onInteractionDone(interaction, wasSaved));
    (0, _registerHandler.registerHandler)('err', function () {
      return _this.storeError(...arguments);
    }, this.featureName, this.ee);
    (0, _registerHandler.registerHandler)('ierr', function () {
      return _this.storeError(...arguments);
    }, this.featureName, this.ee);
    (0, _registerHandler.registerHandler)('softNavFlush', (interactionId, wasFinished, softNavAttrs) => this.onSoftNavNotification(interactionId, wasFinished, softNavAttrs), this.featureName, this.ee); // when an ixn is done or cancelled

    const harvestTimeSeconds = (0, _config.getConfigurationValue)(this.agentIdentifier, 'jserrors.harvestTimeSeconds') || 10;

    // 0 == off, 1 == on
    this.waitForFlags(['err']).then(_ref => {
      let [errFlag] = _ref;
      if (errFlag) {
        const scheduler = new _harvestScheduler.HarvestScheduler('jserrors', {
          onFinished: function () {
            return _this.onHarvestFinished(...arguments);
          }
        }, this);
        scheduler.harvest.on('jserrors', function () {
          return _this.onHarvestStarted(...arguments);
        });
        scheduler.startTimer(harvestTimeSeconds);
        this.drain();
      } else {
        this.blocked = true; // if rum response determines that customer lacks entitlements for spa endpoint, this feature shouldn't harvest
        (0, _drain.deregisterDrain)(this.agentIdentifier, this.featureName);
      }
    });
  }
  onHarvestStarted(options) {
    // this gets rid of dependency in AJAX module
    var body = this.aggregator.take(['err', 'ierr', 'xhr']);
    if (options.retry) {
      this.currentBody = body;
    }
    var payload = {
      body,
      qs: {}
    };
    var releaseIds = (0, _stringify.stringify)((0, _config.getRuntime)(this.agentIdentifier).releaseIds);
    if (releaseIds !== '{}') {
      payload.qs.ri = releaseIds;
    }
    if (body && body.err && body.err.length && !this.errorOnPage) {
      payload.qs.pve = '1';
      this.errorOnPage = true;
    }
    return payload;
  }
  onHarvestFinished(result) {
    if (result.retry && this.currentBody) {
      (0, _mapOwn.mapOwn)(this.currentBody, (key, value) => {
        for (var i = 0; i < value.length; i++) {
          var bucket = value[i];
          var name = this.getBucketName(key, bucket.params, bucket.custom);
          this.aggregator.merge(key, name, bucket.metrics, bucket.params, bucket.custom);
        }
      });
      this.currentBody = null;
    }
  }
  nameHash(params) {
    return (0, _stringHashCode.stringHashCode)("".concat(params.exceptionClass, "_").concat(params.message, "_").concat(params.stack_trace || params.browser_stack_hash));
  }
  getBucketName(objType, params, customParams) {
    if (objType === 'xhr') {
      return (0, _stringHashCode.stringHashCode)((0, _stringify.stringify)(params)) + ':' + (0, _stringHashCode.stringHashCode)((0, _stringify.stringify)(customParams));
    }
    return this.nameHash(params) + ':' + (0, _stringHashCode.stringHashCode)((0, _stringify.stringify)(customParams));
  }

  /**
   * Builds a standardized stack trace string from the frames in the given `stackInfo` object, with each frame separated
   * by a newline character. Lines take the form `<functionName>@<url>:<lineNumber>`.
   *
   * @param {StackInfo} stackInfo - An object specifying a stack string and individual frames.
   * @returns {string} A canonical stack string built from the URLs and function names in the given `stackInfo` object.
   */
  buildCanonicalStackString(stackInfo) {
    var canonicalStackString = '';
    for (var i = 0; i < stackInfo.frames.length; i++) {
      var frame = stackInfo.frames[i];
      var func = (0, _canonicalFunctionName.canonicalFunctionName)(frame.func);
      if (canonicalStackString) canonicalStackString += '\n';
      if (func) canonicalStackString += func + '@';
      if (typeof frame.url === 'string') canonicalStackString += frame.url;
      if (frame.line) canonicalStackString += ':' + frame.line;
    }
    return canonicalStackString;
  }
  storeError(err, time, internal, customAttributes) {
    // are we in an interaction
    time = time || (0, _now.now)();
    const agentRuntime = (0, _config.getRuntime)(this.agentIdentifier);
    let filterOutput;
    if (!internal && agentRuntime.onerror) {
      filterOutput = agentRuntime.onerror(err);
      if (filterOutput && !(typeof filterOutput.group === 'string' && filterOutput.group.length)) {
        // All truthy values mean don't report (store) the error, per backwards-compatible usage,
        // - EXCEPT if a fingerprinting label is returned, via an object with key of 'group' and value of non-empty string
        return;
      }
      // Again as with previous usage, all falsey values would include the error.
    }
    var stackInfo = (0, _computeStackTrace.computeStackTrace)(err);
    var canonicalStackString = this.buildCanonicalStackString(stackInfo);
    const params = {
      stackHash: (0, _stringHashCode.stringHashCode)(canonicalStackString),
      exceptionClass: stackInfo.name,
      request_uri: _runtime.globalScope?.location.pathname
    };
    if (stackInfo.message) params.message = '' + stackInfo.message;
    // Notice if filterOutput isn't false|undefined OR our specified object, this func would've returned already (so it's unnecessary to req-check group).
    // Do not modify the name ('errorGroup') of params without DEM approval!
    if (filterOutput?.group) params.errorGroup = filterOutput.group;

    /**
     * The bucketHash is different from the params.stackHash because the params.stackHash is based on the canonicalized
     * stack trace and is used downstream in NR1 to attempt to group the same errors across different browsers. However,
     * the canonical stack trace excludes items like the column number increasing the hit-rate of different errors potentially
     * bucketing and ultimately resulting in the loss of data in NR1.
     */
    var bucketHash = (0, _stringHashCode.stringHashCode)("".concat(stackInfo.name, "_").concat(stackInfo.message, "_").concat(stackInfo.stackString));
    if (!this.stackReported[bucketHash]) {
      this.stackReported[bucketHash] = true;
      params.stack_trace = (0, _formatStackTrace.truncateSize)(stackInfo.stackString);
      this.observedAt[bucketHash] = agentRuntime.offset + time;
    } else {
      params.browser_stack_hash = (0, _stringHashCode.stringHashCode)(stackInfo.stackString);
    }
    params.releaseIds = (0, _stringify.stringify)(agentRuntime.releaseIds);

    // When debugging stack canonicalization/hashing, uncomment these lines for
    // more output in the test logs
    // params.origStack = err.stack
    // params.canonicalStack = canonicalStack

    if (!this.pageviewReported[bucketHash]) {
      params.pageview = 1;
      this.pageviewReported[bucketHash] = true;
    }
    if (agentRuntime?.session?.state?.sessionReplayMode) params.hasReplay = true;
    params.firstOccurrenceTimestamp = this.observedAt[bucketHash];
    var type = internal ? 'ierr' : 'err';
    var newMetrics = {
      time
    };

    // Trace sends the error in its payload, and both trace & replay simply listens for any error to occur.
    const jsErrorEvent = [type, bucketHash, params, newMetrics, customAttributes];
    (0, _handle.handle)('errorAgg', jsErrorEvent, undefined, _features.FEATURE_NAMES.sessionTrace, this.ee);
    (0, _handle.handle)('errorAgg', jsErrorEvent, undefined, _features.FEATURE_NAMES.sessionReplay, this.ee);
    // still send EE events for other features such as above, but stop this one from aggregating internal data
    if (this.blocked) return;
    const softNavInUse = Boolean((0, _nreum.getNREUMInitializedAgent)(this.agentIdentifier)?.features[_features.FEATURE_NAMES.softNav]);
    // Note: the following are subject to potential race cond wherein if the other feature aren't fully initialized, it'll be treated as there being no associated interaction.
    // They each will also tack on their respective properties to the params object as part of the decision flow.
    if (softNavInUse) (0, _handle.handle)('jserror', [params, time], undefined, _features.FEATURE_NAMES.softNav, this.ee);else (0, _handle.handle)('errorAgg', jsErrorEvent, undefined, _features.FEATURE_NAMES.spa, this.ee);
    if (params.browserInteractionId && !params._softNavFinished) {
      // hold onto the error until the in-progress interaction is done, eithered saved or discarded
      this.bufferedErrorsUnderSpa[params.browserInteractionId] ??= [];
      this.bufferedErrorsUnderSpa[params.browserInteractionId].push(jsErrorEvent);
    } else if (params._interactionId != null) {
      // same as above, except tailored for the way old spa does it
      this.bufferedErrorsUnderSpa[params._interactionId] = this.bufferedErrorsUnderSpa[params._interactionId] || [];
      this.bufferedErrorsUnderSpa[params._interactionId].push(jsErrorEvent);
    } else {
      // Either there is no interaction (then all these params properties will be undefined) OR there's a related soft navigation that's already completed.
      // The old spa does not look up completed interactions at all, so there's no need to consider it.
      this.#storeJserrorForHarvest(jsErrorEvent, params.browserInteractionId !== undefined, params._softNavAttributes);
    }
  }
  #storeJserrorForHarvest(errorInfoArr, softNavOccurredFinished) {
    let softNavCustomAttrs = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    let [type, bucketHash, params, newMetrics, localAttrs] = errorInfoArr;
    const allCustomAttrs = {};
    if (softNavOccurredFinished) {
      Object.entries(softNavCustomAttrs).forEach(_ref2 => {
        let [k, v] = _ref2;
        return setCustom(k, v);
      }); // when an ixn finishes, it'll include stuff in jsAttributes + attrs specific to the ixn
      bucketHash += params.browserInteractionId;
      delete params._softNavAttributes; // cleanup temp properties from synchronous evaluation; this is harmless when async from soft nav (properties DNE)
      delete params._softNavFinished;
    } else {
      // interaction was cancelled -> error should not be associated OR there was no interaction
      Object.entries((0, _config.getInfo)(this.agentIdentifier).jsAttributes).forEach(_ref3 => {
        let [k, v] = _ref3;
        return setCustom(k, v);
      });
      delete params.browserInteractionId;
    }
    if (localAttrs) Object.entries(localAttrs).forEach(_ref4 => {
      let [k, v] = _ref4;
      return setCustom(k, v);
    }); // local custom attrs are applied in either case with the highest precedence

    const jsAttributesHash = (0, _stringHashCode.stringHashCode)((0, _stringify.stringify)(allCustomAttrs));
    const aggregateHash = bucketHash + ':' + jsAttributesHash;
    this.aggregator.store(type, aggregateHash, params, newMetrics, allCustomAttrs);
    function setCustom(key, val) {
      allCustomAttrs[key] = val && typeof val === 'object' ? (0, _stringify.stringify)(val) : val;
    }
  }

  // TO-DO: Remove this function when old spa is taken out. #storeJserrorForHarvest handles the work with the softnav feature.
  onInteractionDone(interaction, wasSaved) {
    if (!this.bufferedErrorsUnderSpa[interaction.id] || this.blocked) return;
    this.bufferedErrorsUnderSpa[interaction.id].forEach(item => {
      var allCustomAttrs = {};
      const localCustomAttrs = item[4];
      (0, _mapOwn.mapOwn)(interaction.root.attrs.custom, setCustom); // tack on custom attrs from the interaction
      (0, _mapOwn.mapOwn)(localCustomAttrs, setCustom);
      var params = item[2];
      if (wasSaved) {
        params.browserInteractionId = interaction.root.attrs.id;
        if (params._interactionNodeId) params.parentNodeId = params._interactionNodeId.toString();
      }
      delete params._interactionId;
      delete params._interactionNodeId;
      var hash = wasSaved ? item[1] + interaction.root.attrs.id : item[1];
      var jsAttributesHash = (0, _stringHashCode.stringHashCode)((0, _stringify.stringify)(allCustomAttrs));
      var aggregateHash = hash + ':' + jsAttributesHash;
      this.aggregator.store(item[0], aggregateHash, params, item[3], allCustomAttrs);
      function setCustom(key, val) {
        allCustomAttrs[key] = val && typeof val === 'object' ? (0, _stringify.stringify)(val) : val;
      }
    });
    delete this.bufferedErrorsUnderSpa[interaction.id];
  }
  onSoftNavNotification(interactionId, wasFinished, softNavAttrs) {
    if (this.blocked) return;
    this.bufferedErrorsUnderSpa[interactionId]?.forEach(jsErrorEvent => this.#storeJserrorForHarvest(jsErrorEvent, wasFinished, softNavAttrs) // this should not modify the re-used softNavAttrs contents
    );
    delete this.bufferedErrorsUnderSpa[interactionId]; // wipe the list of jserrors so they aren't duplicated by another call to the same id
  }
}
exports.Aggregate = Aggregate;