"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Instrument = void 0;
var _handle = require("../../../common/event-emitter/handle");
var _now = require("../../../common/timing/now");
var _instrumentBase = require("../../utils/instrument-base");
var _constants = require("../constants");
var _features = require("../../../loaders/features/features");
var _runtime = require("../../../common/constants/runtime");
var _eventListenerOpts = require("../../../common/event-listener/event-listener-opts");
var _stringify = require("../../../common/util/stringify");
var _uncaughtError = require("./uncaught-error");
/*
 * Copyright 2020 New Relic Corporation. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

class Instrument extends _instrumentBase.InstrumentBase {
  static featureName = _constants.FEATURE_NAME;
  #seenErrors = new Set();
  constructor(agentIdentifier, aggregator) {
    let auto = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
    super(agentIdentifier, aggregator, _constants.FEATURE_NAME, auto);
    try {
      // this try-catch can be removed when IE11 is completely unsupported & gone
      this.removeOnAbort = new AbortController();
    } catch (e) {}

    // Capture function errors early in case the spa feature is loaded
    this.ee.on('fn-err', (args, obj, error) => {
      if (!this.abortHandler || this.#seenErrors.has(error)) return;
      this.#seenErrors.add(error);
      (0, _handle.handle)('err', [this.#castError(error), (0, _now.now)()], undefined, _features.FEATURE_NAMES.jserrors, this.ee);
    });
    this.ee.on('internal-error', error => {
      if (!this.abortHandler) return;
      (0, _handle.handle)('ierr', [this.#castError(error), (0, _now.now)(), true], undefined, _features.FEATURE_NAMES.jserrors, this.ee);
    });
    _runtime.globalScope.addEventListener('unhandledrejection', promiseRejectionEvent => {
      if (!this.abortHandler) return;
      (0, _handle.handle)('err', [this.#castPromiseRejectionEvent(promiseRejectionEvent), (0, _now.now)(), false, {
        unhandledPromiseRejection: 1
      }], undefined, _features.FEATURE_NAMES.jserrors, this.ee);
    }, (0, _eventListenerOpts.eventListenerOpts)(false, this.removeOnAbort?.signal));
    _runtime.globalScope.addEventListener('error', errorEvent => {
      if (!this.abortHandler) return;

      /**
       * If the spa feature is loaded, errors may already have been captured in the `fn-err` listener above.
       * This ensures those errors are not captured twice.
       */
      if (this.#seenErrors.has(errorEvent.error)) {
        this.#seenErrors.delete(errorEvent.error);
        return;
      }
      (0, _handle.handle)('err', [this.#castErrorEvent(errorEvent), (0, _now.now)()], undefined, _features.FEATURE_NAMES.jserrors, this.ee);
    }, (0, _eventListenerOpts.eventListenerOpts)(false, this.removeOnAbort?.signal));
    this.abortHandler = this.#abort; // we also use this as a flag to denote that the feature is active or on and handling errors
    this.importAggregator();
  }

  /** Restoration and resource release tasks to be done if JS error loader is being aborted. Unwind changes to globals. */
  #abort() {
    this.removeOnAbort?.abort();
    this.#seenErrors.clear();
    this.abortHandler = undefined; // weakly allow this abort op to run only once
  }

  /**
   * Any value can be used with the `throw` keyword. This function ensures that the value is
   * either a proper Error instance or attempts to convert it to an UncaughtError instance.
   * @param {any} error The value thrown
   * @returns {Error|UncaughtError} The converted error instance
   */
  #castError(error) {
    if (error instanceof Error) {
      return error;
    }

    /**
     * The thrown value may contain a message property. If it does, try to treat the thrown
     * value as an Error-like object.
     */
    if (typeof error?.message !== 'undefined') {
      return new _uncaughtError.UncaughtError(error.message, error.filename || error.sourceURL, error.lineno || error.line, error.colno || error.col);
    }
    return new _uncaughtError.UncaughtError(typeof error === 'string' ? error : (0, _stringify.stringify)(error));
  }

  /**
   * Attempts to convert a PromiseRejectionEvent object to an Error object
   * @param {PromiseRejectionEvent} unhandledRejectionEvent The unhandled promise rejection event
   * @returns {Error} An Error object with the message as the casted reason
   */
  #castPromiseRejectionEvent(promiseRejectionEvent) {
    let prefix = 'Unhandled Promise Rejection: ';
    if (promiseRejectionEvent?.reason instanceof Error) {
      try {
        promiseRejectionEvent.reason.message = prefix + promiseRejectionEvent.reason.message;
        return promiseRejectionEvent.reason;
      } catch (e) {
        return promiseRejectionEvent.reason;
      }
    }
    if (typeof promiseRejectionEvent.reason === 'undefined') return new _uncaughtError.UncaughtError(prefix);
    const error = this.#castError(promiseRejectionEvent.reason);
    error.message = prefix + error.message;
    return error;
  }

  /**
   * Attempts to convert an ErrorEvent object to an Error object
   * @param {ErrorEvent} errorEvent The error event
   * @returns {Error|UncaughtError} The error event converted to an Error object
   */
  #castErrorEvent(errorEvent) {
    if (errorEvent.error instanceof SyntaxError && !/:\d+$/.test(errorEvent.error.stack?.trim())) {
      const error = new _uncaughtError.UncaughtError(errorEvent.message, errorEvent.filename, errorEvent.lineno, errorEvent.colno);
      error.name = SyntaxError.name;
      return error;
    }
    if (errorEvent.error instanceof Error) {
      return errorEvent.error;
    }

    /**
     * Older browsers do not contain the `error` property on the ErrorEvent instance.
     * https://caniuse.com/mdn-api_errorevent_error
     */
    return new _uncaughtError.UncaughtError(errorEvent.message, errorEvent.filename, errorEvent.lineno, errorEvent.colno);
  }
}
exports.Instrument = Instrument;