"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Aggregate = void 0;
var _config = require("../../../common/config/config");
var _registerHandler = require("../../../common/event-emitter/register-handler");
var _harvestScheduler = require("../../../common/harvest/harvest-scheduler");
var _constants = require("../constants");
var _frameworkDetection = require("./framework-detection");
var _protocol = require("../../../common/url/protocol");
var _obfuscate = require("../../../common/util/obfuscate");
var _load = require("../../../common/window/load");
var _eventListenerOpts = require("../../../common/event-listener/event-listener-opts");
var _runtime = require("../../../common/constants/runtime");
var _aggregateBase = require("../../utils/aggregate-base");
var _drain = require("../../../common/drain/drain");
class Aggregate extends _aggregateBase.AggregateBase {
  static featureName = _constants.FEATURE_NAME;
  constructor(agentIdentifier, aggregator) {
    super(agentIdentifier, aggregator, _constants.FEATURE_NAME);
    this.waitForFlags(['err']).then(_ref => {
      let [errFlag] = _ref;
      if (errFlag) {
        // *cli, Mar 23 - Per NR-94597, this feature should only harvest ONCE at the (potential) EoL time of the page.
        const scheduler = new _harvestScheduler.HarvestScheduler('jserrors', {
          onUnload: () => this.unload()
        }, this);
        // this is needed to ensure EoL is "on" and sent
        scheduler.harvest.on('jserrors', () => ({
          body: this.aggregator.take(['cm', 'sm'])
        }));
        this.drain();
      } else {
        this.blocked = true; // if rum response determines that customer lacks entitlements for spa endpoint, this feature shouldn't harvest
        (0, _drain.deregisterDrain)(this.agentIdentifier, this.featureName);
      }
    });

    // Allow features external to the metrics feature to capture SMs and CMs through the event emitter
    (0, _registerHandler.registerHandler)(_constants.SUPPORTABILITY_METRIC_CHANNEL, this.storeSupportabilityMetrics.bind(this), this.featureName, this.ee);
    (0, _registerHandler.registerHandler)(_constants.CUSTOM_METRIC_CHANNEL, this.storeEventMetrics.bind(this), this.featureName, this.ee);
    this.singleChecks(); // checks that are run only one time, at script load
    this.eachSessionChecks(); // the start of every time user engages with page
  }
  storeSupportabilityMetrics(name, value) {
    if (this.blocked) return;
    const type = _constants.SUPPORTABILITY_METRIC;
    const params = {
      name
    };
    this.aggregator.storeMetric(type, name, params, value);
  }
  storeEventMetrics(name, metrics) {
    if (this.blocked) return;
    const type = _constants.CUSTOM_METRIC;
    const params = {
      name
    };
    this.aggregator.store(type, name, params, metrics);
  }
  singleChecks() {
    // report loaderType
    const {
      distMethod,
      loaderType
    } = (0, _config.getRuntime)(this.agentIdentifier);
    const {
      proxy,
      privacy,
      page_view_timing
    } = (0, _config.getConfiguration)(this.agentIdentifier);
    if (loaderType) this.storeSupportabilityMetrics("Generic/LoaderType/".concat(loaderType, "/Detected"));
    if (distMethod) this.storeSupportabilityMetrics("Generic/DistMethod/".concat(distMethod, "/Detected"));
    if (_runtime.isBrowserScope) {
      this.storeSupportabilityMetrics('Generic/Runtime/Browser/Detected');
      const nonce = document?.currentScript?.nonce;
      if (nonce && nonce !== '') {
        this.storeSupportabilityMetrics('Generic/Runtime/Nonce/Detected');
      }

      // These SMs are used by the AppExp team
      (0, _load.onDOMContentLoaded)(() => {
        (0, _frameworkDetection.getFrameworks)().forEach(framework => {
          this.storeSupportabilityMetrics('Framework/' + framework + '/Detected');
        });
      });
      if (!privacy.cookies_enabled) this.storeSupportabilityMetrics('Config/SessionTracking/Disabled');
      if (page_view_timing.long_task) this.storeSupportabilityMetrics('Config/LongTask/Enabled');
    } else if (_runtime.isWorkerScope) {
      this.storeSupportabilityMetrics('Generic/Runtime/Worker/Detected');
    } else {
      this.storeSupportabilityMetrics('Generic/Runtime/Unknown/Detected');
    }

    // Track if the agent is being loaded using a file protocol such as is the case in some
    // set-top box applications or Electron applications
    if ((0, _protocol.isFileProtocol)()) {
      this.storeSupportabilityMetrics('Generic/FileProtocol/Detected');
    }

    // Capture SMs to assess customer engagement with the obfuscation config
    const rules = (0, _obfuscate.getRules)(this.agentIdentifier);
    if (rules.length > 0) this.storeSupportabilityMetrics('Generic/Obfuscate/Detected');
    if (rules.length > 0 && !(0, _obfuscate.validateRules)(rules)) this.storeSupportabilityMetrics('Generic/Obfuscate/Invalid');

    // Check if proxy for either chunks or beacon is being used
    if (proxy.assets) this.storeSupportabilityMetrics('Config/AssetsUrl/Changed');
    if (proxy.beacon) this.storeSupportabilityMetrics('Config/BeaconUrl/Changed');
  }
  eachSessionChecks() {
    if (!_runtime.isBrowserScope) return;

    // [Temporary] Report restores from BFCache to NR1 while feature flag is in place in lieu of sending pageshow events.
    (0, _eventListenerOpts.windowAddEventListener)('pageshow', evt => {
      if (evt?.persisted) {
        this.storeSupportabilityMetrics('Generic/BFCache/PageRestored');
      }
    });
  }
  unload() {
    try {
      if (this.resourcesSent) return;
      this.resourcesSent = true; // make sure this only gets sent once

      const agentRuntime = (0, _config.getRuntime)(this.agentIdentifier);

      // Capture SMs around network resources using the performance API to assess
      // work to split this out from the ST nodes
      // differentiate between internal+external and ajax+non-ajax
      const ajaxResources = ['beacon', 'fetch', 'xmlhttprequest'];
      const internalUrls = ['nr-data.net', 'newrelic.com', 'nr-local.net', 'localhost'];
      function isInternal(x) {
        return internalUrls.some(y => x.name.indexOf(y) >= 0);
      }
      function isAjax(x) {
        return ajaxResources.includes(x.initiatorType);
      }
      const allResources = performance?.getEntriesByType('resource') || [];
      allResources.forEach(entry => {
        if (isInternal(entry)) {
          if (isAjax(entry)) this.storeSupportabilityMetrics('Generic/Resources/Ajax/Internal');else this.storeSupportabilityMetrics('Generic/Resources/Non-Ajax/Internal');
        } else {
          if (isAjax(entry)) this.storeSupportabilityMetrics('Generic/Resources/Ajax/External');else this.storeSupportabilityMetrics('Generic/Resources/Non-Ajax/External');
        }
      });

      // Capture SMs for session trace if active (`ptid` is set when returned by replay ingest).
      // Retain these SMs while we are working through the session_replay feature
      if (agentRuntime.ptid) {
        this.storeSupportabilityMetrics('PageSession/Feature/SessionTrace/DurationMs', Math.round(performance.now()));
      }

      // Capture SMs for performance markers and measures to assess the usage and possible inclusion of this
      // data in the agent for use in NR
      if (typeof performance !== 'undefined') {
        const markers = performance.getEntriesByType('mark');
        const measures = performance.getEntriesByType('measure');
        this.storeSupportabilityMetrics('Generic/Performance/Mark/Seen', markers.length);
        this.storeSupportabilityMetrics('Generic/Performance/Measure/Seen', measures.length);
      }
    } catch (e) {
      // do nothing
    }
  }
}
exports.Aggregate = Aggregate;