"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Aggregate = void 0;
var _mapOwn = require("../../../common/util/map-own");
var _stringify = require("../../../common/util/stringify");
var _registerHandler = require("../../../common/event-emitter/register-handler");
var _harvestScheduler = require("../../../common/harvest/harvest-scheduler");
var _cleanUrl = require("../../../common/url/clean-url");
var _config = require("../../../common/config/config");
var _constants = require("../constants");
var _runtime = require("../../../common/constants/runtime");
var _aggregateBase = require("../../utils/aggregate-base");
var _drain = require("../../../common/drain/drain");
/*
 * Copyright 2020 New Relic Corporation. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

class Aggregate extends _aggregateBase.AggregateBase {
  static featureName = _constants.FEATURE_NAME;
  constructor(agentIdentifier, aggregator) {
    var _this;
    super(agentIdentifier, aggregator, _constants.FEATURE_NAME);
    _this = this;
    this.eventsPerMinute = 240;
    this.harvestTimeSeconds = (0, _config.getConfigurationValue)(this.agentIdentifier, 'page_action.harvestTimeSeconds') || (0, _config.getConfigurationValue)(this.agentIdentifier, 'ins.harvestTimeSeconds') || 30;
    this.eventsPerHarvest = this.eventsPerMinute * this.harvestTimeSeconds / 60;
    this.referrerUrl = undefined;
    this.currentEvents = undefined;
    this.events = [];
    this.att = (0, _config.getInfo)(this.agentIdentifier).jsAttributes; // per-agent, aggregators-shared info context

    if (_runtime.isBrowserScope && document.referrer) this.referrerUrl = (0, _cleanUrl.cleanURL)(document.referrer);
    (0, _registerHandler.registerHandler)('api-addPageAction', function () {
      return _this.addPageAction(...arguments);
    }, this.featureName, this.ee);
    this.waitForFlags(['ins']).then(_ref => {
      let [insFlag] = _ref;
      if (insFlag) {
        const scheduler = new _harvestScheduler.HarvestScheduler('ins', {
          onFinished: function () {
            return _this.onHarvestFinished(...arguments);
          }
        }, this);
        scheduler.harvest.on('ins', function () {
          return _this.onHarvestStarted(...arguments);
        });
        scheduler.startTimer(this.harvestTimeSeconds, 0);
        this.drain();
      } else {
        this.blocked = true; // if rum response determines that customer lacks entitlements for ins endpoint, this feature shouldn't harvest
        (0, _drain.deregisterDrain)(this.agentIdentifier, this.featureName);
      }
    });
  }
  onHarvestStarted(options) {
    const {
      userAttributes,
      atts
    } = (0, _config.getInfo)(this.agentIdentifier);
    var payload = {
      qs: {
        ua: userAttributes,
        at: atts
      },
      body: {
        ins: this.events
      }
    };
    if (options.retry) {
      this.currentEvents = this.events;
    }
    this.events = [];
    return payload;
  }
  onHarvestFinished(result) {
    if (result && result.sent && result.retry && this.currentEvents) {
      this.events = this.events.concat(this.currentEvents);
      this.currentEvents = null;
    }
  }

  // WARNING: Insights times are in seconds. EXCEPT timestamp, which is in ms.
  addPageAction(t, name, attributes) {
    if (this.events.length >= this.eventsPerHarvest || this.blocked) return;
    var width;
    var height;
    var eventAttributes = {};
    if (_runtime.isBrowserScope && window.document.documentElement) {
      // Doesn't include the nav bar when it disappears in mobile safari
      // https://github.com/jquery/jquery/blob/10399ddcf8a239acc27bdec9231b996b178224d3/src/dimensions.js#L23
      width = window.document.documentElement.clientWidth;
      height = window.document.documentElement.clientHeight;
    }
    var defaults = {
      timestamp: t + (0, _config.getRuntime)(this.agentIdentifier).offset,
      timeSinceLoad: t / 1000,
      browserWidth: width,
      browserHeight: height,
      referrerUrl: this.referrerUrl,
      currentUrl: (0, _cleanUrl.cleanURL)('' + location),
      pageUrl: (0, _cleanUrl.cleanURL)((0, _config.getRuntime)(this.agentIdentifier).origin),
      eventType: 'PageAction'
    };
    (0, _mapOwn.mapOwn)(defaults, set);
    (0, _mapOwn.mapOwn)((0, _config.getInfo)(this.agentIdentifier).jsAttributes, set);
    if (attributes && typeof attributes === 'object') {
      (0, _mapOwn.mapOwn)(attributes, set);
    }
    eventAttributes.actionName = name || '';
    this.events.push(eventAttributes);
    function set(key, val) {
      eventAttributes[key] = val && typeof val === 'object' ? (0, _stringify.stringify)(val) : val;
    }
  }
}
exports.Aggregate = Aggregate;