"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Aggregate = void 0;
var _belSerializer = require("../../../common/serialize/bel-serializer");
var _mapOwn = require("../../../common/util/map-own");
var _harvestScheduler = require("../../../common/harvest/harvest-scheduler");
var _registerHandler = require("../../../common/event-emitter/register-handler");
var _handle = require("../../../common/event-emitter/handle");
var _config = require("../../../common/config/config");
var _constants = require("../constants");
var _features = require("../../../loaders/features/features");
var _aggregateBase = require("../../utils/aggregate-base");
var _cumulativeLayoutShift = require("../../../common/vitals/cumulative-layout-shift");
var _firstContentfulPaint = require("../../../common/vitals/first-contentful-paint");
var _firstInputDelay = require("../../../common/vitals/first-input-delay");
var _firstPaint = require("../../../common/vitals/first-paint");
var _interactionToNextPaint = require("../../../common/vitals/interaction-to-next-paint");
var _largestContentfulPaint = require("../../../common/vitals/largest-contentful-paint");
var _timeToFirstByte = require("../../../common/vitals/time-to-first-byte");
var _longTask = require("../../../common/vitals/long-task");
/*
 * Copyright 2020 New Relic Corporation. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 */

class Aggregate extends _aggregateBase.AggregateBase {
  static featureName = _constants.FEATURE_NAME;
  #handleVitalMetric = _ref => {
    let {
      name,
      value,
      attrs
    } = _ref;
    this.addTiming(name, value, attrs);
  };
  constructor(agentIdentifier, aggregator) {
    var _this;
    super(agentIdentifier, aggregator, _constants.FEATURE_NAME);
    _this = this;
    this.timings = [];
    this.timingsSent = [];
    this.curSessEndRecorded = false;
    if ((0, _config.getConfigurationValue)(this.agentIdentifier, 'page_view_timing.long_task') === true) _longTask.longTask.subscribe(this.#handleVitalMetric);

    /* It's important that CWV api, like "onLCP", is called before this scheduler is initialized. The reason is because they listen to the same
      on vis change or pagehide events, and we'd want ex. onLCP to record the timing (win the race) before we try to send "final harvest". */

    (0, _registerHandler.registerHandler)('docHidden', msTimestamp => this.endCurrentSession(msTimestamp), this.featureName, this.ee);
    (0, _registerHandler.registerHandler)('winPagehide', msTimestamp => this.recordPageUnload(msTimestamp), this.featureName, this.ee);
    const initialHarvestSeconds = (0, _config.getConfigurationValue)(this.agentIdentifier, 'page_view_timing.initialHarvestSeconds') || 10;
    const harvestTimeSeconds = (0, _config.getConfigurationValue)(this.agentIdentifier, 'page_view_timing.harvestTimeSeconds') || 30;
    this.waitForFlags([]).then(() => {
      _firstPaint.firstPaint.subscribe(this.#handleVitalMetric);
      _firstContentfulPaint.firstContentfulPaint.subscribe(this.#handleVitalMetric);
      _firstInputDelay.firstInputDelay.subscribe(this.#handleVitalMetric);
      _largestContentfulPaint.largestContentfulPaint.subscribe(this.#handleVitalMetric);
      _interactionToNextPaint.interactionToNextPaint.subscribe(this.#handleVitalMetric);
      _timeToFirstByte.timeToFirstByte.subscribe(_ref2 => {
        let {
          entries
        } = _ref2;
        this.addTiming('load', Math.round(entries[0].loadEventEnd));
      });
      const scheduler = new _harvestScheduler.HarvestScheduler('events', {
        onFinished: function () {
          return _this.onHarvestFinished(...arguments);
        },
        getPayload: function () {
          return _this.prepareHarvest(...arguments);
        }
      }, this);
      scheduler.startTimer(harvestTimeSeconds, initialHarvestSeconds);
      this.drain();
    });
  }

  /**
   * Add the time of _document visibilitychange to hidden_ to the next PVT harvest == NRDB pageHide attr.
   * @param {number} timestamp
   */
  endCurrentSession(timestamp) {
    if (!this.curSessEndRecorded) {
      // TO DO: stage 2 - we don't want to capture this timing twice on page navigating away, but it should run again if we return to page and away *again*
      this.addTiming('pageHide', timestamp, null);
      this.curSessEndRecorded = true;
    }
  }

  /**
   * Add the time of _window pagehide event_ firing to the next PVT harvest == NRDB windowUnload attr.
   */
  recordPageUnload(timestamp) {
    this.addTiming('unload', timestamp, null);
    /*
    Issue: Because window's pageHide commonly fires BEFORE vis change and "final" harvest would happen at the former in this case, we also have to add our vis-change event now or it may not be sent.
    Affected: Safari < v14.1/.5 ; versions that don't support 'visiilitychange' event
    Impact: For affected w/o this, NR 'pageHide' attribute may not be sent. For other browsers w/o this, NR 'pageHide' gets fragmented into its own harvest call on page unloading because of dual EoL logic.
    Mitigation: NR 'unload' and 'pageHide' are both recorded when window pageHide fires, rather than only recording 'unload'.
    Future: When EoL can become the singular subscribeToVisibilityChange, it's likely endCurrentSession isn't needed here as 'unload'-'pageHide' can be untangled.
    */
    this.endCurrentSession(timestamp);
  }
  addTiming(name, value, attrs) {
    attrs = attrs || {};
    addConnectionAttributes(attrs); // network conditions may differ from the actual for VitalMetrics when they were captured

    // If cls was set to another value by `onCLS`, then it's supported and is attached onto any timing but is omitted until such time.
    /*
    *cli Apr'23 - Convert attach-to-all -> attach-if-not-null. See NEWRELIC-6143.
    Issue: Because NR 'pageHide' was only sent once with what is considered the "final" CLS value, in the case that 'pageHide' fires before 'load' happens, we incorrectly a final CLS of 0 for that page.
    Mitigation: We've set initial CLS to null so that it's omitted from timings like 'pageHide' in that edge case. It should only be included if onCLS callback was executed at least once.
    Future: onCLS value changes should be reported directly & CLS separated into its own timing node so it's not beholden to 'pageHide' firing. It'd also be possible to report the real final CLS.
    */
    if (_cumulativeLayoutShift.cumulativeLayoutShift.current.value >= 0) {
      attrs.cls = _cumulativeLayoutShift.cumulativeLayoutShift.current.value;
    }
    this.timings.push({
      name,
      value,
      attrs
    });
    (0, _handle.handle)('pvtAdded', [name, value, attrs], undefined, _features.FEATURE_NAMES.sessionTrace, this.ee);
  }
  onHarvestFinished(result) {
    if (result.retry && this.timingsSent.length > 0) {
      this.timings.unshift(...this.timingsSent);
      this.timingsSent = [];
    }
  }
  appendGlobalCustomAttributes(timing) {
    var timingAttributes = timing.attrs || {};
    var customAttributes = (0, _config.getInfo)(this.agentIdentifier).jsAttributes || {};
    var reservedAttributes = ['size', 'eid', 'cls', 'type', 'fid', 'elTag', 'elUrl', 'net-type', 'net-etype', 'net-rtt', 'net-dlink'];
    (0, _mapOwn.mapOwn)(customAttributes, function (key, val) {
      if (reservedAttributes.indexOf(key) < 0) {
        timingAttributes[key] = val;
      }
    });
  }

  // serialize and return current timing data, clear and save current data for retry
  prepareHarvest(options) {
    if (this.timings.length === 0) return;
    var payload = this.getPayload(this.timings);
    if (options.retry) {
      for (var i = 0; i < this.timings.length; i++) {
        this.timingsSent.push(this.timings[i]);
      }
    }
    this.timings = [];
    return {
      body: {
        e: payload
      }
    };
  }

  // serialize array of timing data
  getPayload(data) {
    var addString = (0, _belSerializer.getAddStringContext)(this.agentIdentifier);
    var payload = 'bel.6;';
    for (var i = 0; i < data.length; i++) {
      var timing = data[i];
      payload += 'e,';
      payload += addString(timing.name) + ',';
      payload += (0, _belSerializer.nullable)(timing.value, _belSerializer.numeric, false) + ',';
      this.appendGlobalCustomAttributes(timing);
      var attrParts = (0, _belSerializer.addCustomAttributes)(timing.attrs, addString);
      if (attrParts && attrParts.length > 0) {
        payload += (0, _belSerializer.numeric)(attrParts.length) + ';' + attrParts.join(';');
      }
      if (i + 1 < data.length) payload += ';';
    }
    return payload;
  }
}
exports.Aggregate = Aggregate;
function addConnectionAttributes(obj) {
  var connection = navigator.connection || navigator.mozConnection || navigator.webkitConnection; // to date, both window & worker shares the same support for connection
  if (!connection) return;
  if (connection.type) obj['net-type'] = connection.type;
  if (connection.effectiveType) obj['net-etype'] = connection.effectiveType;
  if (connection.rtt) obj['net-rtt'] = connection.rtt;
  if (connection.downlink) obj['net-dlink'] = connection.downlink;
}