"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Interaction = void 0;
var _config = require("../../../common/config/config");
var _runtime = require("../../../common/constants/runtime");
var _uniqueId = require("../../../common/ids/unique-id");
var _belSerializer = require("../../../common/serialize/bel-serializer");
var _cleanUrl = require("../../../common/url/clean-url");
var _constants = require("../constants");
var _belNode = require("./bel-node");
/**
 * link https://github.com/newrelic/nr-querypack/blob/main/schemas/bel/7.qpschema
 **/
class Interaction extends _belNode.BelNode {
  id = (0, _uniqueId.generateUuid)(); // unique id that is serialized and used to link interactions with errors
  initialPageURL = _runtime.initialLocation;
  oldURL = '' + _runtime.globalScope?.location;
  newURL = '' + _runtime.globalScope?.location;
  customName;
  customAttributes = {};
  customDataByApi = {};
  queueTime; // only used by initialPageLoad interactions
  appTime; // only used by initialPageLoad interactions
  newRoute;
  /** Internal state of this interaction: in-progress, finished, or cancelled. */
  status = _constants.INTERACTION_STATUS.IP;
  domTimestamp = 0;
  historyTimestamp = 0;
  createdByApi = false;
  keepOpenUntilEndApi = false;
  onDone = [];
  cancellationTimer;
  constructor(agentIdentifier, uiEvent, uiEventTimestamp, currentRouteKnown) {
    super(agentIdentifier);
    this.belType = _constants.NODE_TYPE.INTERACTION;
    this.trigger = uiEvent;
    this.start = uiEventTimestamp;
    this.oldRoute = currentRouteKnown;
    this.eventSubscription = new Map([['finished', []], ['cancelled', []]]);
    this.forceSave = this.forceIgnore = false;
    if (this.trigger === _constants.API_TRIGGER_NAME) this.createdByApi = true;
  }
  updateDom(timestamp) {
    this.domTimestamp = timestamp || performance.now(); // default timestamp should be precise for accurate isActiveDuring calculations
  }
  updateHistory(timestamp, newUrl) {
    this.newURL = newUrl || '' + _runtime.globalScope?.location;
    this.historyTimestamp = timestamp || performance.now();
  }
  seenHistoryAndDomChange() {
    return this.historyTimestamp > 0 && this.domTimestamp > this.historyTimestamp; // URL must change before DOM does
  }
  on(event, cb) {
    if (!this.eventSubscription.has(event)) throw new Error('Cannot subscribe to non pre-defined events.');
    if (typeof cb !== 'function') throw new Error('Must supply function as callback.');
    this.eventSubscription.get(event).push(cb);
  }
  done(customEndTime) {
    // User could've mark this interaction--regardless UI or api started--as "don't close until .end() is called on it". Only .end provides a timestamp; the default flows do not.
    if (this.keepOpenUntilEndApi && customEndTime === undefined) return false;
    this.onDone.forEach(apiProvidedCb => apiProvidedCb(this.customDataByApi)); // this interaction's .save or .ignore can still be set by these user provided callbacks for example

    if (this.forceIgnore) this.#cancel(); // .ignore() always has precedence over save actions
    else if (this.seenHistoryAndDomChange()) this.#finish(customEndTime); // then this should've already finished while it was the interactionInProgress, with a natural end time
    else if (this.forceSave) this.#finish(customEndTime || performance.now()); // a manually saved ixn (did not fulfill conditions) must have a specified end time, if one wasn't provided
    else this.#cancel();
    return true;
  }
  #finish() {
    let customEndTime = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
    if (this.status !== _constants.INTERACTION_STATUS.IP) return; // disallow this call if the ixn is already done aka not in-progress
    clearTimeout(this.cancellationTimer);
    this.end = Math.max(this.domTimestamp, this.historyTimestamp, customEndTime);
    this.customAttributes = {
      ...(0, _config.getInfo)(this.agentIdentifier).jsAttributes,
      ...this.customAttributes
    }; // attrs specific to this interaction should have precedence over the general custom attrs
    this.status = _constants.INTERACTION_STATUS.FIN;

    // Run all the callbacks awaiting this interaction to finish.
    const callbacks = this.eventSubscription.get('finished');
    callbacks.forEach(fn => fn());
  }
  #cancel() {
    if (this.status !== _constants.INTERACTION_STATUS.IP) return; // disallow this call if the ixn is already done aka not in-progress
    clearTimeout(this.cancellationTimer);
    this.status = _constants.INTERACTION_STATUS.CAN;

    // Run all the callbacks listening to this interaction's potential cancellation.
    const callbacks = this.eventSubscription.get('cancelled');
    callbacks.forEach(fn => fn());
  }

  /**
   * Given a timestamp, determine if it falls within this interaction's span, i.e. if this was the active interaction during that time.
   * For in-progress interactions, this only compares the time with the start of span. Cancelled interactions are not considered active at all.
   * @param {DOMHighResTimeStamp} timestamp
   * @returns True or false boolean.
   */
  isActiveDuring(timestamp) {
    if (this.status === _constants.INTERACTION_STATUS.IP) return this.start <= timestamp;
    return this.status === _constants.INTERACTION_STATUS.FIN && this.start <= timestamp && this.end >= timestamp;
  }

  // Following are virtual properties overridden by a subclass:
  get firstPaint() {}
  get firstContentfulPaint() {}
  get navTiming() {}
  serialize(firstStartTimeOfPayload) {
    const addString = (0, _belSerializer.getAddStringContext)(this.agentIdentifier);
    const nodeList = [];
    let ixnType;
    if (this.trigger === 'initialPageLoad') ixnType = _constants.INTERACTION_TYPE.INITIAL_PAGE_LOAD;else if (this.newURL !== this.oldURL) ixnType = _constants.INTERACTION_TYPE.ROUTE_CHANGE;else ixnType = _constants.INTERACTION_TYPE.UNSPECIFIED;

    // IMPORTANT: The order in which addString is called matters and correlates to the order in which string shows up in the harvest payload. Do not re-order the following code.
    const fields = [(0, _belSerializer.numeric)(this.belType), 0,
    // this will be overwritten below with number of attached nodes
    (0, _belSerializer.numeric)(Math.floor(this.start - firstStartTimeOfPayload)),
    // relative to first node
    (0, _belSerializer.numeric)(Math.floor(this.end - this.start)),
    // end -- relative to start
    (0, _belSerializer.numeric)(this.callbackEnd),
    // cbEnd -- relative to start; not used by BrowserInteraction events
    (0, _belSerializer.numeric)(this.callbackDuration),
    // not relative
    addString(this.trigger), addString((0, _cleanUrl.cleanURL)(this.initialPageURL, true)), addString((0, _cleanUrl.cleanURL)(this.oldURL, true)), addString((0, _cleanUrl.cleanURL)(this.newURL, true)), addString(this.customName), ixnType, (0, _belSerializer.nullable)(this.queueTime, _belSerializer.numeric, true) + (0, _belSerializer.nullable)(this.appTime, _belSerializer.numeric, true) + (0, _belSerializer.nullable)(this.oldRoute, addString, true) + (0, _belSerializer.nullable)(this.newRoute, addString, true) + addString(this.id), addString(this.nodeId), (0, _belSerializer.nullable)(this.firstPaint, _belSerializer.numeric, true) + (0, _belSerializer.nullable)(this.firstContentfulPaint, _belSerializer.numeric)];
    const allAttachedNodes = (0, _belSerializer.addCustomAttributes)(this.customAttributes || {}, addString); // start with all custom attributes
    if ((0, _config.getInfo)(this.agentIdentifier).atts) allAttachedNodes.push('a,' + addString((0, _config.getInfo)(this.agentIdentifier).atts)); // add apm provided attributes
    /* Querypack encoder+decoder quirkiness:
       - If first ixn node of payload is being processed, we use this node's start to offset. (firstStartTime should be 0--or undefined.)
       - Else for subsequent ixn nodes, we use the first ixn node's start to offset. */
    this.children.forEach(node => allAttachedNodes.push(node.serialize(firstStartTimeOfPayload || this.start))); // recursively add the serialized string of every child of this (ixn) bel node

    fields[1] = (0, _belSerializer.numeric)(allAttachedNodes.length);
    nodeList.push(fields);
    if (allAttachedNodes.length) nodeList.push(allAttachedNodes.join(';'));
    if (this.navTiming) nodeList.push(this.navTiming);else nodeList.push('');
    // nodeList = [<fields array>, <serialized string of all attributes and children>, <serialized nav timing info> || '']

    return nodeList.join(';');
  }
}
exports.Interaction = Interaction;