"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InstrumentBase = void 0;
var _drain = require("../../common/drain/drain");
var _featureBase = require("./feature-base");
var _load = require("../../common/window/load");
var _runtime = require("../../common/constants/runtime");
var _console = require("../../common/util/console");
var _features = require("../../loaders/features/features");
var _config = require("../../common/config/config");
var _utils = require("../session_replay/shared/utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /**
 * @file Defines `InstrumentBase` to be used as the super of the Instrument classes implemented by each feature.
 * Inherits and executes the `checkConfiguration` method from [FeatureBase]{@link ./feature-base}, which also
 * exposes the `blocked` property.
 */
/**
 * Base class for instrumenting a feature.
 * @extends FeatureBase
 */
class InstrumentBase extends _featureBase.FeatureBase {
  /**
   * Instantiate InstrumentBase.
   * @param {string} agentIdentifier - The unique ID of the instantiated agent (relative to global scope).
   * @param {import('../../common/aggregate/aggregator').Aggregator} aggregator - The shared Aggregator that will handle batching and reporting of data.
   * @param {string} featureName - The name of the feature module (used to construct file path).
   * @param {boolean} [auto=true] - Determines whether the feature should automatically register to have the draining
   *     of its pooled instrumentation data handled by the agent's centralized drain functionality, rather than draining
   *     immediately. Primarily useful for fine-grained control in tests.
   */
  constructor(agentIdentifier, aggregator, featureName) {
    let auto = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : true;
    super(agentIdentifier, aggregator, featureName);
    this.auto = auto;

    /** @type {Function | undefined} This should be set by any derived Instrument class if it has things to do when feature fails or is killed. */
    this.abortHandler = undefined;

    /**
     * @type {import('./aggregate-base').AggregateBase} Holds the reference to the feature's aggregate module counterpart, if and after it has been initialized. This may not be assigned until after page loads!
     * The only purpose of this for now is to expose it to the NREUM interface, as the feature's instrument instance is already exposed.
    */
    this.featAggregate = undefined;

    /**
     * @type {Promise} Assigned immediately after @see importAggregator runs. Serves as a signal for when the inner async fn finishes execution. Useful for features to await
     * one another if there are inter-features dependencies.
    */
    this.onAggregateImported = undefined;

    /** used in conjunction with newrelic.start() to defer harvesting in features */
    if ((0, _config.getConfigurationValue)(this.agentIdentifier, "".concat(this.featureName, ".autoStart")) === false) this.auto = false;
    /** if the feature requires opt-in (!auto-start), it will get registered once the api has been called */
    if (this.auto) (0, _drain.registerDrain)(agentIdentifier, featureName);
  }

  /**
   * Lazy-load the latter part of the feature: its aggregator. This method is called by the first part of the feature
   * (the instrumentation) when instrumentation is complete.
   * @param {Object} [argsObjFromInstrument] - any values or references to pass down to aggregate
   * @returns void
   */
  importAggregator() {
    let argsObjFromInstrument = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    if (this.featAggregate) return;
    if (!this.auto) {
      // this feature requires an opt in...
      // wait for API to be called
      this.ee.on("".concat(this.featureName, "-opt-in"), () => {
        // register the feature to drain only once the API has been called, it will drain when importAggregator finishes for all the features
        // called by the api in that cycle
        (0, _drain.registerDrain)(this.agentIdentifier, this.featureName);
        this.auto = true;
        this.importAggregator();
      });
      return;
    }
    let loadedSuccessfully;
    this.onAggregateImported = new Promise(resolve => {
      loadedSuccessfully = resolve;
    });
    const importLater = async () => {
      let session;
      try {
        if ((0, _utils.enableSessionTracking)(this.agentIdentifier)) {
          // would require some setup before certain features start
          const {
            setupAgentSession
          } = await Promise.resolve().then(() => _interopRequireWildcard(require( /* webpackChunkName: "session-manager" */'./agent-session')));
          session = setupAgentSession(this.agentIdentifier);
        }
      } catch (e) {
        (0, _console.warn)('A problem occurred when starting up session manager. This page will not start or extend any session.', e);
        if (this.featureName === _features.FEATURE_NAMES.sessionReplay) this.abortHandler?.(); // SR should stop recording if session DNE
      }

      /**
       * Note this try-catch differs from the one in Agent.run() in that it's placed later in a page's lifecycle and
       * it's only responsible for aborting its one specific feature, rather than all.
       */
      try {
        if (!this.#shouldImportAgg(this.featureName, session)) {
          (0, _drain.drain)(this.agentIdentifier, this.featureName);
          loadedSuccessfully(false); // aggregate module isn't loaded at all
          return;
        }
        const {
          lazyFeatureLoader
        } = await Promise.resolve().then(() => _interopRequireWildcard(require( /* webpackChunkName: "lazy-feature-loader" */'./lazy-feature-loader')));
        const {
          Aggregate
        } = await lazyFeatureLoader(this.featureName, 'aggregate');
        this.featAggregate = new Aggregate(this.agentIdentifier, this.aggregator, argsObjFromInstrument);
        loadedSuccessfully(true);
      } catch (e) {
        (0, _console.warn)("Downloading and initializing ".concat(this.featureName, " failed..."), e);
        this.abortHandler?.(); // undo any important alterations made to the page
        // not supported yet but nice to do: "abort" this agent's EE for this feature specifically
        (0, _drain.drain)(this.agentIdentifier, this.featureName, true);
        loadedSuccessfully(false);
        if (this.ee) this.ee.abort();
      }
    };

    // For regular web pages, we want to wait and lazy-load the aggregator only after all page resources are loaded.
    // Non-browser scopes (i.e. workers) have no `window.load` event, so the aggregator can be lazy-loaded immediately.
    if (!_runtime.isBrowserScope) importLater();else (0, _load.onWindowLoad)(() => importLater(), true);
  }

  /**
  * Make a determination if an aggregate class should even be imported
  * @param {string} featureName
  * @param {import('../../common/session/session-entity').SessionEntity} session
  * @returns
  */
  #shouldImportAgg(featureName, session) {
    if (featureName === _features.FEATURE_NAMES.sessionReplay) return (0, _utils.canImportReplayAgg)(this.agentIdentifier, session);
    return true;
  }
}
exports.InstrumentBase = InstrumentBase;