"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Agent = void 0;
require("./configure/public-path");
require("./configure/nonce");
var _agentBase = require("./agent-base");
var _enabledFeatures = require("./features/enabled-features");
var _configure = require("./configure/configure");
var _featureDependencies = require("./features/featureDependencies");
var _features = require("./features/features");
var _instrument = require("../features/page_view_event/instrument");
var _aggregator = require("../common/aggregate/aggregator");
var _nreum = require("../common/window/nreum");
var _console = require("../common/util/console");
var _stringify = require("../common/util/stringify");
var _runtime = require("../common/constants/runtime");
// important side effects

// loader files

// required features

// common files

/**
 * A flexible class that may be used to compose an agent from a select subset of feature modules. In applications
 * sensitive to network load, this may result in smaller builds with slightly lower performance impact.
 */
class Agent extends _agentBase.AgentBase {
  constructor(options, agentIdentifier) {
    super(agentIdentifier);
    if (!_runtime.globalScope) {
      // We could not determine the runtime environment. Short-circuite the agent here
      // to avoid possible exceptions later that may cause issues with customer's application.
      (0, _console.warn)('Failed to initial the agent. Could not determine the runtime environment.');
      return;
    }
    this.sharedAggregator = new _aggregator.Aggregator({
      agentIdentifier: this.agentIdentifier
    });
    this.features = {};
    (0, _nreum.setNREUMInitializedAgent)(this.agentIdentifier, this); // append this agent onto the global NREUM.initializedAgents

    this.desiredFeatures = new Set(options.features || []); // expected to be a list of static Instrument/InstrumentBase classes, see "spa.js" for example
    // For Now... ALL agents must make the rum call whether the page_view_event feature was enabled or not.
    // NR1 creates an index on the rum call, and if not seen for a few days, will remove the browser app!
    // Future work is being planned to evaluate removing this behavior from the backend, but for now we must ensure this call is made
    this.desiredFeatures.add(_instrument.Instrument);
    this.runSoftNavOverSpa = [...this.desiredFeatures].some(instr => instr.featureName === _features.FEATURE_NAMES.softNav);
    (0, _configure.configure)(this, options, options.loaderType || 'agent'); // add api, exposed, and other config properties

    this.run();
  }
  get config() {
    return {
      info: this.info,
      init: this.init,
      loader_config: this.loader_config,
      runtime: this.runtime
    };
  }
  run() {
    // Attempt to initialize all the requested features (sequentially in prio order & synchronously), with any failure aborting the whole process.
    try {
      const enabledFeatures = (0, _enabledFeatures.getEnabledFeatures)(this.agentIdentifier);
      const featuresToStart = [...this.desiredFeatures];
      featuresToStart.sort((a, b) => _features.featurePriority[a.featureName] - _features.featurePriority[b.featureName]);
      featuresToStart.forEach(InstrumentCtor => {
        if (!enabledFeatures[InstrumentCtor.featureName] && InstrumentCtor.featureName !== _features.FEATURE_NAMES.pageViewEvent) return; // PVE is required to run even if it's marked disabled
        if (this.runSoftNavOverSpa && InstrumentCtor.featureName === _features.FEATURE_NAMES.spa) return;
        if (!this.runSoftNavOverSpa && InstrumentCtor.featureName === _features.FEATURE_NAMES.softNav) return;
        const dependencies = (0, _featureDependencies.getFeatureDependencyNames)(InstrumentCtor.featureName);
        const hasAllDeps = dependencies.every(featName => featName in this.features); // any other feature(s) this depends on should've been initialized on prior iterations by priority order
        if (!hasAllDeps) (0, _console.warn)("".concat(InstrumentCtor.featureName, " is enabled but one or more dependent features has not been initialized (").concat((0, _stringify.stringify)(dependencies), "). This may cause unintended consequences or missing data..."));
        this.features[InstrumentCtor.featureName] = new InstrumentCtor(this.agentIdentifier, this.sharedAggregator);
      });
    } catch (err) {
      (0, _console.warn)('Failed to initialize all enabled instrument classes (agent aborted) -', err);
      for (const featName in this.features) {
        // this.features hold only features that have been instantiated
        this.features[featName].abortHandler?.();
      }
      const newrelic = (0, _nreum.gosNREUM)();
      delete newrelic.initializedAgents[this.agentIdentifier]?.api; // prevent further calls to agent-specific APIs (see "configure.js")
      delete newrelic.initializedAgents[this.agentIdentifier]?.features; // GC mem used internally by features
      delete this.sharedAggregator;
      // Keep the initialized agent object with its configs for troubleshooting purposes.
      const thisEE = newrelic.ee.get(this.agentIdentifier);
      thisEE.abort(); // set flag and clear backlog
      return false;
    }
  }
}
exports.Agent = Agent;