"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MicroAgent = void 0;
var _instrument = require("../features/page_view_event/instrument");
var _enabledFeatures = require("./features/enabled-features");
var _configure = require("./configure/configure");
var _aggregator = require("../common/aggregate/aggregator");
var _nreum = require("../common/window/nreum");
var _uniqueId = require("../common/ids/unique-id");
var _config = require("../common/config/config");
var _features = require("./features/features");
var _console = require("../common/util/console");
var _load = require("../common/window/load");
var _agentBase = require("./agent-base");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } // loader files
// core files
const nonAutoFeatures = [_features.FEATURE_NAMES.jserrors, _features.FEATURE_NAMES.pageAction, _features.FEATURE_NAMES.metrics];

/**
 * A minimal agent class designed to only respond to manual user input. As such, this class does not
 * automatically instrument. Instead, each MicroAgent instance will lazy load the required features and can support loading multiple instances on one page.
 * Out of the box, it can manually handle and report Page View, Page Action, and Error events.
 */
class MicroAgent extends _agentBase.AgentBase {
  /**
   * @param {Object} options - Specifies features and runtime configuration,
   * @param {string=} agentIdentifier - The optional unique ID of the agent.
   */
  constructor(options) {
    let agentIdentifier = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : (0, _uniqueId.generateRandomHexString)(16);
    super();
    this.agentIdentifier = agentIdentifier;
    this.sharedAggregator = new _aggregator.Aggregator({
      agentIdentifier: this.agentIdentifier
    });
    this.features = {};
    (0, _nreum.setNREUMInitializedAgent)(agentIdentifier, this);
    (0, _configure.configure)(this, {
      ...options,
      runtime: {
        isolatedBacklog: true
      }
    }, options.loaderType || 'micro-agent');
    Object.assign(this, this.api); // the APIs should be available at the class level for micro-agent

    /**
     * Starts a set of agent features if not running in "autoStart" mode
     * {@link https://docs.newrelic.com/docs/browser/new-relic-browser/browser-apis/start/}
     * @param {string|string[]|undefined} name The feature name(s) to start.  If no name(s) are passed, all features will be started
     */
    this.start = features => this.run(features);
    this.run(nonAutoFeatures.filter(featureName => (0, _config.getConfigurationValue)(agentIdentifier, "".concat(featureName, ".autoStart"))));
  }
  get config() {
    return {
      info: (0, _config.getInfo)(this.agentIdentifier),
      init: (0, _config.getConfiguration)(this.agentIdentifier),
      loader_config: (0, _config.getLoaderConfig)(this.agentIdentifier),
      runtime: (0, _config.getRuntime)(this.agentIdentifier)
    };
  }
  run(features) {
    try {
      const featNames = nonAutoFeatures;
      if (features === undefined) features = featNames;else {
        features = Array.isArray(features) && features.length ? features : [features];
        if (features.some(f => !featNames.includes(f))) return (0, _console.warn)("Invalid feature name supplied. Acceptable feature names are: ".concat(featNames));
        if (!features.includes(_features.FEATURE_NAMES.pageViewEvent)) features.push(_features.FEATURE_NAMES.pageViewEvent);
      }
    } catch (err) {
      (0, _console.warn)('An unexpected issue occurred', err);
    }
    try {
      const enabledFeatures = (0, _enabledFeatures.getEnabledFeatures)(this.agentIdentifier);
      try {
        // a biproduct of doing this is that the "session manager" is automatically handled through importing this feature
        this.features.page_view_event = new _instrument.Instrument(this.agentIdentifier, this.sharedAggregator);
      } catch (err) {
        (0, _console.warn)('Something prevented the agent from instrumenting.', err);
      }
      (0, _load.onWindowLoad)(() => {
        // these features do not import an "instrument" file, meaning they are only hooked up to the API.
        nonAutoFeatures.forEach(f => {
          if (enabledFeatures[f] && features.includes(f)) {
            Promise.resolve().then(() => _interopRequireWildcard(require( /* webpackChunkName: "lazy-feature-loader" */'../features/utils/lazy-feature-loader'))).then(_ref => {
              let {
                lazyFeatureLoader
              } = _ref;
              return lazyFeatureLoader(f, 'aggregate');
            }).then(_ref2 => {
              let {
                Aggregate
              } = _ref2;
              this.features[f] = new Aggregate(this.agentIdentifier, this.sharedAggregator);
            }).catch(err => (0, _console.warn)('Something prevented the agent from being downloaded.', err));
          }
        });
      });
      return true;
    } catch (err) {
      (0, _console.warn)('Failed to initialize instrument classes.', err);
      return false;
    }
  }
}
exports.MicroAgent = MicroAgent;