/*global jasmine,angular,describe,it,expect*/
var Application = require('admin-config/lib/Application'),
    Entity = require('admin-config/lib/Entity/Entity'),
    DashboardView = require('admin-config/lib/View/DashboardView'),
    CreateView = require('admin-config/lib/View/CreateView');

describe("Service: Application config", function () {
    'use strict';

    describe('entity', function () {
        it('should store entity by name.', function () {
            var app = new Application(),
                entity = new Entity('myEntity');
            app.addEntity(entity);

            expect(app.getEntity('myEntity').name()).toBe('myEntity');
            expect(app.getEntity('myEntity').order()).toBe(0);
            expect(app.hasEntity('myEntity')).toBe(true);
        });

        it('should return all entity names.', function () {
            var app = new Application();
            app.addEntity(new Entity('myEntity1'));
            app.addEntity(new Entity('myEntity2'));

            expect(app.getEntityNames()).toEqual(['myEntity1', 'myEntity2']);
        });
    });

    describe('view', function () {
        it('should returns all view of a certain type.', function () {
            var app = new Application(),
                entity1 = new Entity('myEntity1'),
                entity2 = new Entity('myEntity2'),
                dashboard = entity1.dashboardView().enable(),
                dashboard2 = entity2.dashboardView().enable(),
                createView = entity2.creationView().enable();

            app.addEntity(entity1);
            app.addEntity(entity2);

            var dashboards = app.getViewsOfType('DashboardView'),
                forms = app.getViewsOfType('CreateView'),
                lists = app.getViewsOfType('ListView');

            expect(dashboards.length).toBe(2);
            expect(forms.length).toBe(1);
            expect(lists.length).toBe(0);

            expect(dashboards[0].getEntity().name()).toBe('myEntity1');
            expect(dashboards[1].getEntity().name()).toBe('myEntity2');
        });
    });

    describe('getRouteFor', function () {
        it('should return the url specified in a view', function () {
            var app = new Application(),
                entity1 = new Entity('myEntity1'),
                view = entity1.dashboardView();

            view.url('http://localhost/dashboard');
            app.addEntity(entity1);

            expect(app.getRouteFor(entity1, view.getUrl(), view.type)).toBe('http://localhost/dashboard');
        });

        it('should not consider protocol relative URL as a relative path', function () {
            var app = new Application(),
                entity1 = new Entity('myEntity1'),
                view = entity1.dashboardView();

            view.url('//localhost/dashboard');
            app.addEntity(entity1);

            expect(app.getRouteFor(entity1, view.getUrl(), view.type)).toBe('//localhost/dashboard');
        });

        it('should return the url specified in the entity when the URL is not specified in the view', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.dashboardView();

            entity1.baseApiUrl('http://api.com/');
            app.addEntity(entity1);

            expect(app.getRouteFor(entity1, view.getUrl(), view.type)).toBe('http://api.com/comments');
        });

        it('should return the url specified in the entity when the app also define a base URL', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.dashboardView();

            entity1.baseApiUrl('//api.com/');
            app.baseApiUrl('http://api-entity.com/');
            app.addEntity(entity1);

            expect(app.getRouteFor(entity1, view.getUrl(), view.type)).toBe('//api.com/comments');
        });

        it('should return the url specified in the app when the URL is not specified in the view nor in the entity', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.dashboardView();

            app.baseApiUrl('https://elastic.local/');
            app.addEntity(entity1);

            expect(app.getRouteFor(entity1, view.getUrl(), view.type)).toBe('https://elastic.local/comments');
        });

        it('should call url() defined in the view if it\'s a function', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.editionView();

            app.baseApiUrl('http://api.local');

            view.url(function (entityId) {
                return '/post/:' + entityId;
            });

            expect(app.getRouteFor(entity1, view.getUrl(1), view.type, 1, view.identifier())).toBe('http://api.local/post/:1');
        });

        it('should call url() defined in the entity if it\'s a function', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.editionView();

            app.baseApiUrl('http://api.local');

            entity1.url(function (entityName, viewType, identifierValue) {
                return '/' + entityName + '_' + viewType + '/:' + identifierValue;
            });

            expect(app.getRouteFor(entity1, view.getUrl(1), view.type, 1, view.identifier())).toBe('http://api.local/comments_EditView/:1');
        });

        it('should not prepend baseApiUrl when the URL begins with http', function () {
            var app = new Application(),
                entity1 = new Entity('comments'),
                view = entity1.editionView();

            app.baseApiUrl('http://api.local');

            entity1.url('http://mock.local/entity');

            expect(app.getRouteFor(entity1, view.getUrl(1), view.type, 1, view.identifier())).toBe('http://mock.local/entity');
        });
    });

    describe('getErrorMessageFor', function () {
        it('should return the global message with a simple string as body', function () {
            var app = new Application(),
                entity = new Entity(),
                response = {
                    status: 500,
                    data: 'myBody'
                };

            app.addEntity(entity);

            expect(app.getErrorMessageFor(entity.creationView(), response)).toBe('Oops, an error occured : (code: 500) myBody');
        });

        it('should return the global message with an object as body', function () {
            var app = new Application(),
                entity = new Entity(),
                response = {
                    status: 500,
                    data: {
                        error: 'Internal error'
                    }
                };

            app.addEntity(entity);

            expect(app.getErrorMessageFor(entity.listView(), response)).toBe('Oops, an error occured : (code: 500) {"error":"Internal error"}');
        });

        it('should return the error message defined globally', function () {
            var app = new Application(),
                entity = new Entity(),
                response = {
                    status: 500
                };

            app.errorMessage(function (response) {
                return 'Global error: ' + response.status;
            });

            app.addEntity(entity);

            expect(app.getErrorMessageFor(entity.listView(), response)).toBe('Global error: 500');
        });

        it('should return the message defined by the entity', function () {
            var app = new Application(),
                entity = new Entity(),
                response = {
                    status: 500,
                    data: {
                        error: 'Internal error'
                    }
                };

            entity.errorMessage(function (response) {
                return 'error: ' + response.status;
            });

            app.addEntity(entity);

            expect(app.getErrorMessageFor(entity.listView(), response)).toBe('error: 500');
        });

        it('should return the message defined by the view', function () {
            var app = new Application(),
                entity = new Entity(),
                response = {
                    status: 500,
                    data: {
                        error: 'Internal error'
                    }
                };

            entity.listView().errorMessage(function (response) {
                return 'Error during listing: ' + response.status;
            });

            app.addEntity(entity);

            expect(app.getErrorMessageFor(entity.listView(), response)).toBe('Error during listing: 500');
        });
    });
});
