import { Directive, Input, ElementRef, ViewContainerRef, Renderer, ComponentFactoryResolver, NgZone, forwardRef } from '@angular/core';
import { NG_VALUE_ACCESSOR } from '@angular/forms';
import { NgbDate } from './ngb-date';
import { NgbDatepicker } from './datepicker';
import { NgbDateParserFormatter } from './ngb-date-parser-formatter';
import { positionElements } from '../util/positioning';
var NGB_DATEPICKER_VALUE_ACCESSOR = {
    provide: NG_VALUE_ACCESSOR,
    useExisting: forwardRef(function () { return NgbInputDatepicker; }),
    multi: true
};
/**
 * A directive that makes it possible to have datepickers on input fields.
 * Manages integration with the input field itself (data entry) and ngModel (validation etc.).
 */
export var NgbInputDatepicker = (function () {
    function NgbInputDatepicker(_parserFormatter, _elRef, _vcRef, _renderer, _cfr, ngZone) {
        var _this = this;
        this._parserFormatter = _parserFormatter;
        this._elRef = _elRef;
        this._vcRef = _vcRef;
        this._renderer = _renderer;
        this._cfr = _cfr;
        this._cRef = null;
        this._onChange = function (_) { };
        this._onTouched = function () { };
        this._zoneSubscription = ngZone.onStable.subscribe(function () {
            if (_this._cRef) {
                positionElements(_this._elRef.nativeElement, _this._cRef.location.nativeElement, 'bottom-left');
            }
        });
    }
    /**
     * @internal
     */
    NgbInputDatepicker.prototype.registerOnChange = function (fn) { this._onChange = fn; };
    /**
     * @internal
     */
    NgbInputDatepicker.prototype.registerOnTouched = function (fn) { this._onTouched = fn; };
    /**
     * @internal
     */
    NgbInputDatepicker.prototype.writeValue = function (value) {
        this._model = value ? new NgbDate(value.year, value.month, value.day) : null;
        this._writeModelValue(this._model);
    };
    /**
     * @internal
     */
    NgbInputDatepicker.prototype.setDisabledState = function (isDisabled) {
        this._renderer.setElementProperty(this._elRef.nativeElement, 'disabled', isDisabled);
        if (this.isOpen()) {
            this._cRef.instance.setDisabledState(isDisabled);
        }
    };
    /**
     * @internal
     */
    NgbInputDatepicker.prototype.manualDateChange = function (value) {
        this._model = this._parserFormatter.parse(value);
        this._onChange(this._model ? { year: this._model.year, month: this._model.month, day: this._model.day } : null);
        this._writeModelValue(this._model);
    };
    NgbInputDatepicker.prototype.isOpen = function () { return !!this._cRef; };
    /**
     * Opens the datepicker with the selected date indicated by the ngModel value.
     */
    NgbInputDatepicker.prototype.open = function () {
        var _this = this;
        if (!this.isOpen()) {
            var cf = this._cfr.resolveComponentFactory(NgbDatepicker);
            this._cRef = this._vcRef.createComponent(cf);
            this._applyPopupStyling(this._cRef.location.nativeElement);
            this._cRef.instance.writeValue(this._model);
            this._applyDatepickerInputs(this._cRef.instance);
            this._cRef.instance.ngOnInit();
            // date selection event handling
            this._cRef.instance.registerOnChange(function (selectedDate) {
                _this.writeValue(selectedDate);
                _this._onChange(selectedDate);
                _this.close();
            });
        }
    };
    /**
     * Closes the datepicker popup.
     */
    NgbInputDatepicker.prototype.close = function () {
        if (this.isOpen()) {
            this._vcRef.remove(this._vcRef.indexOf(this._cRef.hostView));
            this._cRef = null;
        }
    };
    /**
     * Toggles the datepicker popup (opens when closed and closes when opened).
     */
    NgbInputDatepicker.prototype.toggle = function () {
        if (this.isOpen()) {
            this.close();
        }
        else {
            this.open();
        }
    };
    /**
     * Navigates current view to provided date. If nothing provided calendar will open current month.
     * Use 'startDate' input as an alternative
     */
    NgbInputDatepicker.prototype.navigateTo = function (date) {
        if (this.isOpen()) {
            this._cRef.instance.navigateTo(date);
        }
    };
    NgbInputDatepicker.prototype._applyDatepickerInputs = function (datepickerInstance) {
        var _this = this;
        ['dayTemplate', 'firstDayOfWeek', 'markDisabled', 'minDate', 'maxDate', 'showNavigation', 'showWeekdays',
            'showWeekNumbers', 'startDate']
            .forEach(function (optionName) {
            if (_this[optionName] !== undefined) {
                datepickerInstance[optionName] = _this[optionName];
            }
        });
    };
    NgbInputDatepicker.prototype._applyPopupStyling = function (nativeElement) {
        this._renderer.setElementClass(nativeElement, 'dropdown-menu', true);
        this._renderer.setElementStyle(nativeElement, 'display', 'block');
        this._renderer.setElementStyle(nativeElement, 'padding', '0.40rem');
    };
    NgbInputDatepicker.prototype._writeModelValue = function (model) {
        this._renderer.setElementProperty(this._elRef.nativeElement, 'value', this._parserFormatter.format(model));
        if (this.isOpen()) {
            this._cRef.instance.writeValue(model);
        }
    };
    NgbInputDatepicker.decorators = [
        { type: Directive, args: [{
                    selector: 'input[ngbDatepicker]',
                    exportAs: 'ngbDatepicker',
                    host: { '(change)': 'manualDateChange($event.target.value)', '(keyup.esc)': 'close()' },
                    providers: [NGB_DATEPICKER_VALUE_ACCESSOR]
                },] },
    ];
    /** @nocollapse */
    NgbInputDatepicker.ctorParameters = [
        { type: NgbDateParserFormatter, },
        { type: ElementRef, },
        { type: ViewContainerRef, },
        { type: Renderer, },
        { type: ComponentFactoryResolver, },
        { type: NgZone, },
    ];
    NgbInputDatepicker.propDecorators = {
        'dayTemplate': [{ type: Input },],
        'firstDayOfWeek': [{ type: Input },],
        'markDisabled': [{ type: Input },],
        'minDate': [{ type: Input },],
        'maxDate': [{ type: Input },],
        'showNavigation': [{ type: Input },],
        'showWeekdays': [{ type: Input },],
        'showWeekNumbers': [{ type: Input },],
        'startDate': [{ type: Input },],
    };
    return NgbInputDatepicker;
}());
//# sourceMappingURL=datepicker-input.js.map