import { Directive, Input, Output, EventEmitter, ComponentFactoryResolver, ViewContainerRef, Injector, Renderer, ElementRef, forwardRef, NgZone } from '@angular/core';
import { NG_VALUE_ACCESSOR } from '@angular/forms';
import { Subject } from 'rxjs/Rx';
import 'rxjs/add/operator/do';
import 'rxjs/add/operator/let';
import { positionElements } from '../util/positioning';
import { NgbTypeaheadWindow } from './typeahead-window';
import { PopupService } from '../util/popup';
import { toString } from '../util/util';
import { NgbTypeaheadConfig } from './typeahead-config';
var Key;
(function (Key) {
    Key[Key["Tab"] = 9] = "Tab";
    Key[Key["Enter"] = 13] = "Enter";
    Key[Key["Escape"] = 27] = "Escape";
    Key[Key["ArrowUp"] = 38] = "ArrowUp";
    Key[Key["ArrowDown"] = 40] = "ArrowDown";
})(Key || (Key = {}));
var NGB_TYPEAHEAD_VALUE_ACCESSOR = {
    provide: NG_VALUE_ACCESSOR,
    useExisting: forwardRef(function () { return NgbTypeahead; }),
    multi: true
};
/**
 * NgbTypeahead directive provides a simple way of creating powerful typeaheads from any text input
 */
export var NgbTypeahead = (function () {
    function NgbTypeahead(_elementRef, _viewContainerRef, _renderer, _injector, componentFactoryResolver, config, ngZone) {
        var _this = this;
        this._elementRef = _elementRef;
        this._viewContainerRef = _viewContainerRef;
        this._renderer = _renderer;
        this._injector = _injector;
        this._valueChanges = new Subject();
        /**
         * An event emitted when a match is selected. Event payload is of type NgbTypeaheadSelectItemEvent.
         */
        this.selectItem = new EventEmitter();
        this.onChange = function (value) {
            _this._onChangeNoEmit(value);
            _this._valueChanges.next(value);
        };
        this.onTouched = function () { };
        this.showHint = config.showHint;
        this._popupService = new PopupService(NgbTypeaheadWindow, _injector, _viewContainerRef, _renderer, componentFactoryResolver);
        this._onChangeNoEmit = function (_) { };
        this._zoneSubscription = ngZone.onStable.subscribe(function () {
            if (_this._windowRef) {
                positionElements(_this._elementRef.nativeElement, _this._windowRef.location.nativeElement, 'bottom-left');
            }
        });
    }
    NgbTypeahead.prototype.ngOnDestroy = function () {
        this._subscription.unsubscribe();
        this._zoneSubscription.unsubscribe();
    };
    NgbTypeahead.prototype.ngOnInit = function () {
        var _this = this;
        this._subscription =
            this._valueChanges.do(function (value) { return _this._userInput = value; }).let(this.ngbTypeahead).subscribe(function (results) {
                if (!results || results.length === 0) {
                    _this._closePopup();
                }
                else {
                    _this._openPopup();
                    _this._windowRef.instance.results = results;
                    _this._windowRef.instance.term = _this._elementRef.nativeElement.value;
                    if (_this.resultFormatter) {
                        _this._windowRef.instance.formatter = _this.resultFormatter;
                    }
                    if (_this.resultTemplate) {
                        _this._windowRef.instance.resultTemplate = _this.resultTemplate;
                    }
                    _this._showHint();
                }
            });
    };
    NgbTypeahead.prototype.registerOnChange = function (fn) { this._onChangeNoEmit = fn; };
    NgbTypeahead.prototype.registerOnTouched = function (fn) { this.onTouched = fn; };
    NgbTypeahead.prototype.writeValue = function (value) { this._writeInputValue(this._formatItemForInput(value)); };
    NgbTypeahead.prototype.setDisabledState = function (isDisabled) {
        this._renderer.setElementProperty(this._elementRef.nativeElement, 'disabled', isDisabled);
    };
    /**
     * @internal
     */
    NgbTypeahead.prototype.dismissPopup = function () {
        if (this.isPopupOpen()) {
            this._closePopup();
            this._writeInputValue(this._userInput);
        }
    };
    /**
     * @internal
     */
    NgbTypeahead.prototype.isPopupOpen = function () { return this._windowRef != null; };
    /**
     * @internal
     */
    NgbTypeahead.prototype.handleKeyDown = function (event) {
        if (!this._windowRef) {
            return;
        }
        if (Key[toString(event.which)]) {
            event.preventDefault();
            switch (event.which) {
                case Key.ArrowDown:
                    this._windowRef.instance.next();
                    this._showHint();
                    break;
                case Key.ArrowUp:
                    this._windowRef.instance.prev();
                    this._showHint();
                    break;
                case Key.Enter:
                case Key.Tab:
                    var result = this._windowRef.instance.getActive();
                    this._selectResult(result);
                    break;
                case Key.Escape:
                    this.dismissPopup();
                    break;
            }
        }
    };
    NgbTypeahead.prototype._openPopup = function () {
        var _this = this;
        if (!this._windowRef) {
            this._windowRef = this._popupService.open();
            this._windowRef.instance.selectEvent.subscribe(function (result) { return _this._selectResult(result); });
        }
    };
    NgbTypeahead.prototype._closePopup = function () {
        this._popupService.close();
        this._windowRef = null;
    };
    NgbTypeahead.prototype._selectResult = function (result) {
        var defaultPrevented = false;
        this.selectItem.emit({ item: result, preventDefault: function () { defaultPrevented = true; } });
        if (!defaultPrevented) {
            this.writeValue(result);
            this._onChangeNoEmit(result);
        }
        this._closePopup();
    };
    NgbTypeahead.prototype._showHint = function () {
        if (this.showHint) {
            var userInputLowerCase = this._userInput.toLowerCase();
            var formattedVal = this._formatItemForInput(this._windowRef.instance.getActive());
            if (userInputLowerCase === formattedVal.substr(0, this._userInput.length).toLowerCase()) {
                this._writeInputValue(this._userInput + formattedVal.substr(this._userInput.length));
                this._renderer.invokeElementMethod(this._elementRef.nativeElement, 'setSelectionRange', [this._userInput.length, formattedVal.length]);
            }
            else {
                this.writeValue(this._windowRef.instance.getActive());
            }
        }
    };
    NgbTypeahead.prototype._formatItemForInput = function (item) {
        return item && this.inputFormatter ? this.inputFormatter(item) : toString(item);
    };
    NgbTypeahead.prototype._writeInputValue = function (value) {
        this._renderer.setElementProperty(this._elementRef.nativeElement, 'value', value);
    };
    NgbTypeahead.decorators = [
        { type: Directive, args: [{
                    selector: 'input[ngbTypeahead]',
                    host: {
                        '(blur)': 'onTouched()',
                        '[class.open]': 'isPopupOpen()',
                        '(document:click)': 'dismissPopup()',
                        '(input)': 'onChange($event.target.value)',
                        '(keydown)': 'handleKeyDown($event)',
                        'autocomplete': 'off',
                        'autocapitalize': 'off',
                        'autocorrect': 'off'
                    },
                    providers: [NGB_TYPEAHEAD_VALUE_ACCESSOR]
                },] },
    ];
    /** @nocollapse */
    NgbTypeahead.ctorParameters = [
        { type: ElementRef, },
        { type: ViewContainerRef, },
        { type: Renderer, },
        { type: Injector, },
        { type: ComponentFactoryResolver, },
        { type: NgbTypeaheadConfig, },
        { type: NgZone, },
    ];
    NgbTypeahead.propDecorators = {
        'inputFormatter': [{ type: Input },],
        'ngbTypeahead': [{ type: Input },],
        'resultFormatter': [{ type: Input },],
        'resultTemplate': [{ type: Input },],
        'showHint': [{ type: Input },],
        'selectItem': [{ type: Output },],
    };
    return NgbTypeahead;
}());
//# sourceMappingURL=typeahead.js.map