import { EMPTY, fromEvent, of, race, Subject, timer } from 'rxjs';
import { endWith, filter, takeUntil } from 'rxjs/operators';
import { getTransitionDurationMs } from './util';
import { environment } from '../../environment';
import { runInZone } from '../util';
const noopFn = () => { };
const ɵ0 = noopFn;
const { transitionTimerDelayMs } = environment;
const runningTransitions = new Map();
export const ngbRunTransition = (zone, element, startFn, options) => {
    // Getting initial context from options
    let context = options.context || {};
    // Checking if there are already running transitions on the given element.
    const running = runningTransitions.get(element);
    if (running) {
        switch (options.runningTransition) {
            // If there is one running and we want for it to 'continue' to run, we have to cancel the new one.
            // We're not emitting any values, but simply completing the observable (EMPTY).
            case 'continue':
                return EMPTY;
            // If there is one running and we want for it to 'stop', we have to complete the running one.
            // We're simply completing the running one and not emitting any values and merging newly provided context
            // with the one coming from currently running transition.
            case 'stop':
                zone.run(() => running.transition$.complete());
                context = Object.assign(running.context, context);
                runningTransitions.delete(element);
        }
    }
    // Running the start function
    const endFn = startFn(element, options.animation, context) || noopFn;
    // If 'prefer-reduced-motion' is enabled, the 'transition' will be set to 'none'.
    // If animations are disabled, we have to emit a value and complete the observable
    // In this case we have to call the end function, but can finish immediately by emitting a value,
    // completing the observable and executing end functions synchronously.
    if (!options.animation || window.getComputedStyle(element).transitionProperty === 'none') {
        zone.run(() => endFn());
        return of(undefined).pipe(runInZone(zone));
    }
    // Starting a new transition
    const transition$ = new Subject();
    const finishTransition$ = new Subject();
    const stop$ = transition$.pipe(endWith(true));
    runningTransitions.set(element, {
        transition$,
        complete: () => {
            finishTransition$.next();
            finishTransition$.complete();
        },
        context
    });
    const transitionDurationMs = getTransitionDurationMs(element);
    // 1. We have to both listen for the 'transitionend' event and have a 'just-in-case' timer,
    // because 'transitionend' event might not be fired in some browsers, if the transitioning
    // element becomes invisible (ex. when scrolling, making browser tab inactive, etc.). The timer
    // guarantees, that we'll release the DOM element and complete 'ngbRunTransition'.
    // 2. We need to filter transition end events, because they might bubble from shorter transitions
    // on inner DOM elements. We're only interested in the transition on the 'element' itself.
    zone.runOutsideAngular(() => {
        const transitionEnd$ = fromEvent(element, 'transitionend').pipe(takeUntil(stop$), filter(({ target }) => target === element));
        const timer$ = timer(transitionDurationMs + transitionTimerDelayMs).pipe(takeUntil(stop$));
        race(timer$, transitionEnd$, finishTransition$).pipe(takeUntil(stop$)).subscribe(() => {
            runningTransitions.delete(element);
            zone.run(() => {
                endFn();
                transition$.next();
                transition$.complete();
            });
        });
    });
    return transition$.asObservable();
};
export const ngbCompleteTransition = (element) => {
    var _a;
    (_a = runningTransitions.get(element)) === null || _a === void 0 ? void 0 : _a.complete();
};
export { ɵ0 };
//# sourceMappingURL=data:application/json;base64,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