/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
import { NgbDate } from '../ngb-date';
/** @type {?} */
var PARTS_PER_HOUR = 1080;
/** @type {?} */
var PARTS_PER_DAY = 24 * PARTS_PER_HOUR;
/** @type {?} */
var PARTS_FRACTIONAL_MONTH = 12 * PARTS_PER_HOUR + 793;
/** @type {?} */
var PARTS_PER_MONTH = 29 * PARTS_PER_DAY + PARTS_FRACTIONAL_MONTH;
/** @type {?} */
var BAHARAD = 11 * PARTS_PER_HOUR + 204;
/** @type {?} */
var HEBREW_DAY_ON_JAN_1_1970 = 2092591;
/** @type {?} */
var GREGORIAN_EPOCH = 1721425.5;
/**
 * @param {?} year
 * @return {?}
 */
function isGregorianLeapYear(year) {
    return year % 4 === 0 && year % 100 !== 0 || year % 400 === 0;
}
/**
 * @param {?} year
 * @return {?}
 */
function numberOfFirstDayInYear(year) {
    /** @type {?} */
    var monthsBeforeYear = Math.floor((235 * year - 234) / 19);
    /** @type {?} */
    var fractionalMonthsBeforeYear = monthsBeforeYear * PARTS_FRACTIONAL_MONTH + BAHARAD;
    /** @type {?} */
    var dayNumber = monthsBeforeYear * 29 + Math.floor(fractionalMonthsBeforeYear / PARTS_PER_DAY);
    /** @type {?} */
    var timeOfDay = fractionalMonthsBeforeYear % PARTS_PER_DAY;
    /** @type {?} */
    var dayOfWeek = dayNumber % 7; // 0 == Monday
    if (dayOfWeek === 2 || dayOfWeek === 4 || dayOfWeek === 6) {
        dayNumber++;
        dayOfWeek = dayNumber % 7;
    }
    if (dayOfWeek === 1 && timeOfDay > 15 * PARTS_PER_HOUR + 204 && !isHebrewLeapYear(year)) {
        dayNumber += 2;
    }
    else if (dayOfWeek === 0 && timeOfDay > 21 * PARTS_PER_HOUR + 589 && isHebrewLeapYear(year - 1)) {
        dayNumber++;
    }
    return dayNumber;
}
/**
 * @param {?} month
 * @param {?} year
 * @return {?}
 */
function getDaysInGregorianMonth(month, year) {
    /** @type {?} */
    var days = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    if (isGregorianLeapYear(year)) {
        days[1]++;
    }
    return days[month - 1];
}
/**
 * @param {?} year
 * @return {?}
 */
function getHebrewMonths(year) {
    return isHebrewLeapYear(year) ? 13 : 12;
}
/**
 * Returns the number of days in a specific Hebrew year.
 * `year` is any Hebrew year.
 * @param {?} year
 * @return {?}
 */
function getDaysInHebrewYear(year) {
    return numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
}
/**
 * @param {?} year
 * @return {?}
 */
export function isHebrewLeapYear(year) {
    /** @type {?} */
    var b = (year * 12 + 17) % 19;
    return b >= ((b < 0) ? -7 : 12);
}
/**
 * Returns the number of days in a specific Hebrew month.
 * `month` is 1 for Nisan, 2 for Iyar etc. Note: Hebrew leap year contains 13 months.
 * `year` is any Hebrew year.
 * @param {?} month
 * @param {?} year
 * @return {?}
 */
export function getDaysInHebrewMonth(month, year) {
    /** @type {?} */
    var yearLength = numberOfFirstDayInYear(year + 1) - numberOfFirstDayInYear(year);
    /** @type {?} */
    var yearType = (yearLength <= 380 ? yearLength : (yearLength - 30)) - 353;
    /** @type {?} */
    var leapYear = isHebrewLeapYear(year);
    /** @type {?} */
    var daysInMonth = leapYear ? [30, 29, 29, 29, 30, 30, 29, 30, 29, 30, 29, 30, 29] :
        [30, 29, 29, 29, 30, 29, 30, 29, 30, 29, 30, 29];
    if (yearType > 0) {
        daysInMonth[2]++; // Kislev gets an extra day in normal or complete years.
    }
    if (yearType > 1) {
        daysInMonth[1]++; // Heshvan gets an extra day in complete years only.
    }
    return daysInMonth[month - 1];
}
/**
 * @param {?} date
 * @return {?}
 */
export function getDayNumberInHebrewYear(date) {
    /** @type {?} */
    var numberOfDay = 0;
    for (var i = 1; i < date.month; i++) {
        numberOfDay += getDaysInHebrewMonth(i, date.year);
    }
    return numberOfDay + date.day;
}
/**
 * @param {?} date
 * @param {?} val
 * @return {?}
 */
export function setHebrewMonth(date, val) {
    /** @type {?} */
    var after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getHebrewMonths(date.year) - date.month) {
                val -= getHebrewMonths(date.year) - date.month + 1;
                date.year++;
                date.month = 1;
            }
            else {
                date.month += val;
                val = 0;
            }
        }
        else {
            if (val >= date.month) {
                date.year--;
                val -= date.month;
                date.month = getHebrewMonths(date.year);
            }
            else {
                date.month -= val;
                val = 0;
            }
        }
    }
    return date;
}
/**
 * @param {?} date
 * @param {?} val
 * @return {?}
 */
export function setHebrewDay(date, val) {
    /** @type {?} */
    var after = val >= 0;
    if (!after) {
        val = -val;
    }
    while (val > 0) {
        if (after) {
            if (val > getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date)) {
                val -= getDaysInHebrewYear(date.year) - getDayNumberInHebrewYear(date) + 1;
                date.year++;
                date.month = 1;
                date.day = 1;
            }
            else if (val > getDaysInHebrewMonth(date.month, date.year) - date.day) {
                val -= getDaysInHebrewMonth(date.month, date.year) - date.day + 1;
                date.month++;
                date.day = 1;
            }
            else {
                date.day += val;
                val = 0;
            }
        }
        else {
            if (val >= date.day) {
                val -= date.day;
                date.month--;
                if (date.month === 0) {
                    date.year--;
                    date.month = getHebrewMonths(date.year);
                }
                date.day = getDaysInHebrewMonth(date.month, date.year);
            }
            else {
                date.day -= val;
                val = 0;
            }
        }
    }
    return date;
}
/**
 * Returns the equivalent Hebrew date value for a give input Gregorian date.
 * `gdate` is a JS Date to be converted to Hebrew date.
 * @param {?} gdate
 * @return {?}
 */
export function fromGregorian(gdate) {
    /** @type {?} */
    var date = new Date(gdate);
    /** @type {?} */
    var gYear = date.getFullYear();
    /** @type {?} */
    var gMonth = date.getMonth();
    /** @type {?} */
    var gDay = date.getDate();
    /** @type {?} */
    var julianDay = GREGORIAN_EPOCH - 1 + 365 * (gYear - 1) + Math.floor((gYear - 1) / 4) -
        Math.floor((gYear - 1) / 100) + Math.floor((gYear - 1) / 400) +
        Math.floor((367 * (gMonth + 1) - 362) / 12 + (gMonth + 1 <= 2 ? 0 : isGregorianLeapYear(gYear) ? -1 : -2) + gDay);
    julianDay = Math.floor(julianDay + 0.5);
    /** @type {?} */
    var daysSinceHebEpoch = julianDay - 347997;
    /** @type {?} */
    var monthsSinceHebEpoch = Math.floor(daysSinceHebEpoch * PARTS_PER_DAY / PARTS_PER_MONTH);
    /** @type {?} */
    var hYear = Math.floor((monthsSinceHebEpoch * 19 + 234) / 235) + 1;
    /** @type {?} */
    var firstDayOfThisYear = numberOfFirstDayInYear(hYear);
    /** @type {?} */
    var dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    while (dayOfYear < 1) {
        hYear--;
        firstDayOfThisYear = numberOfFirstDayInYear(hYear);
        dayOfYear = daysSinceHebEpoch - firstDayOfThisYear;
    }
    /** @type {?} */
    var hMonth = 1;
    /** @type {?} */
    var hDay = dayOfYear;
    while (hDay > getDaysInHebrewMonth(hMonth, hYear)) {
        hDay -= getDaysInHebrewMonth(hMonth, hYear);
        hMonth++;
    }
    return new NgbDate(hYear, hMonth, hDay);
}
/**
 * Returns the equivalent JS date value for a given Hebrew date.
 * `hebrewDate` is an Hebrew date to be converted to Gregorian.
 * @param {?} hebrewDate
 * @return {?}
 */
export function toGregorian(hebrewDate) {
    /** @type {?} */
    var hYear = hebrewDate.year;
    /** @type {?} */
    var hMonth = hebrewDate.month;
    /** @type {?} */
    var hDay = hebrewDate.day;
    /** @type {?} */
    var days = numberOfFirstDayInYear(hYear);
    for (var i = 1; i < hMonth; i++) {
        days += getDaysInHebrewMonth(i, hYear);
    }
    days += hDay;
    /** @type {?} */
    var diffDays = days - HEBREW_DAY_ON_JAN_1_1970;
    /** @type {?} */
    var after = diffDays >= 0;
    if (!after) {
        diffDays = -diffDays;
    }
    /** @type {?} */
    var gYear = 1970;
    /** @type {?} */
    var gMonth = 1;
    /** @type {?} */
    var gDay = 1;
    while (diffDays > 0) {
        if (after) {
            if (diffDays >= (isGregorianLeapYear(gYear) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear) ? 366 : 365;
                gYear++;
            }
            else if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                gMonth++;
            }
            else {
                gDay += diffDays;
                diffDays = 0;
            }
        }
        else {
            if (diffDays >= (isGregorianLeapYear(gYear - 1) ? 366 : 365)) {
                diffDays -= isGregorianLeapYear(gYear - 1) ? 366 : 365;
                gYear--;
            }
            else {
                if (gMonth > 1) {
                    gMonth--;
                }
                else {
                    gMonth = 12;
                    gYear--;
                }
                if (diffDays >= getDaysInGregorianMonth(gMonth, gYear)) {
                    diffDays -= getDaysInGregorianMonth(gMonth, gYear);
                }
                else {
                    gDay = getDaysInGregorianMonth(gMonth, gYear) - diffDays + 1;
                    diffDays = 0;
                }
            }
        }
    }
    return new Date(gYear, gMonth - 1, gDay);
}
/**
 * @param {?} numerals
 * @return {?}
 */
export function hebrewNumerals(numerals) {
    if (!numerals) {
        return '';
    }
    /** @type {?} */
    var hArray0_9 = ['', 'א', 'ב', 'ג', 'ד', 'ה', 'ו', 'ז', 'ח', 'ט'];
    /** @type {?} */
    var hArray10_19 = ['י', 'יא‬', 'יב‬', 'יג‬', 'יד‬', 'טו', 'טז‬', 'יז‬', 'יח‬', 'יט‬'];
    /** @type {?} */
    var hArray20_90 = ['', '', 'כ', 'ל', 'מ', 'נ', 'ס', 'ע', 'פ', 'צ'];
    /** @type {?} */
    var hArray100_900 = ['', 'ק', 'ר', 'ש', 'ת', 'תק', 'תר', 'תש', 'תת', 'תתק'];
    /** @type {?} */
    var hArray1000_9000 = ['', 'א', 'ב', 'אב', 'בב', 'ה', 'אה', 'בה', 'אבה', 'בבה'];
    /** @type {?} */
    var geresh = '׳';
    /** @type {?} */
    var gershaim = '״';
    /** @type {?} */
    var mem = 0;
    /** @type {?} */
    var result = [];
    /** @type {?} */
    var step = 0;
    while (numerals > 0) {
        /** @type {?} */
        var m = numerals % 10;
        if (step === 0) {
            mem = m;
        }
        else if (step === 1) {
            if (m !== 1) {
                result.unshift(hArray20_90[m], hArray0_9[mem]);
            }
            else {
                result.unshift(hArray10_19[mem]);
            }
        }
        else if (step === 2) {
            result.unshift(hArray100_900[m]);
        }
        else {
            if (m !== 5) {
                result.unshift(hArray1000_9000[m], geresh, ' ');
            }
            break;
        }
        numerals = Math.floor(numerals / 10);
        if (step === 0 && numerals === 0) {
            result.unshift(hArray0_9[m]);
        }
        step++;
    }
    result = result.join('').split('');
    if (result.length === 1) {
        result.push(geresh);
    }
    else if (result.length > 1) {
        result.splice(result.length - 1, 0, gershaim);
    }
    return result.join('');
}

//# sourceMappingURL=data:application/json;base64,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