import { DatePickerViewModes } from './bs-date-picker-options.provider';
import * as moment from 'moment';
export var DatePickerBase = (function () {
    function DatePickerBase(datePickerState, options) {
        var _this = this;
        this.subscriptions = [];
        this.datePickerState = datePickerState;
        this.options = options;
        if (!datePickerState.viewDate) {
            datePickerState.viewDate = moment();
        }
        this.refresh(datePickerState.viewDate);
        this.subscriptions.push(options.onUpdate.subscribe(function () {
            _this.refresh(datePickerState.viewDate);
        }));
        this.subscriptions.push(datePickerState.viewDateChange.subscribe(function (v) {
            _this.refresh(datePickerState.viewDate);
        }));
        this.subscriptions.push(datePickerState.activeDateChange.subscribe(function (v) {
            _this.markActive();
        }));
        this.subscriptions.push(datePickerState.selectedDateChange.subscribe(function (v) {
            _this.markSelected();
        }));
        this.subscriptions.push(datePickerState.selectedEndDateChange.subscribe(function (v) {
            _this.markSelected();
            _this.markActive();
        }));
    }
    DatePickerBase.prototype.ngOnInit = function () {
        if (this.options.date) {
            var selected = this.datePickerState.selectedDate || this.options.date.selected;
            var selectedEnd = this.datePickerState.selectedEndDate || this.options.date.selectedEnd;
            this.datePickerState.viewDate = this.datePickerState.viewDate || moment(this.options.date.initial);
            this.datePickerState.selectedDate = selected ? moment(selected) : void 0;
            this.datePickerState.selectedEndDate = selectedEnd ? moment(selectedEnd) : void 0;
        }
    };
    DatePickerBase.prototype.ngOnDestroy = function () {
        this.subscriptions.forEach(function (sub) { return sub.unsubscribe(); });
    };
    DatePickerBase.prototype.viewPrev = function (unitOfTime, step) {
        if (step === void 0) { step = 1; }
        this.datePickerState.viewDate = this.datePickerState.viewDate.clone().subtract(step, unitOfTime);
    };
    DatePickerBase.prototype.viewNext = function (unitOfTime, step) {
        if (step === void 0) { step = 1; }
        this.datePickerState.viewDate = this.datePickerState.viewDate.clone().add(step, unitOfTime);
    };
    /**
     * Selects new view mode
     * do nothing if mode <> min/max modes
     */
    DatePickerBase.prototype.viewMode = function (mode) {
        if (DatePickerViewModes[mode] >= DatePickerViewModes[this.options.ui.minMode] &&
            DatePickerViewModes[mode] <= DatePickerViewModes[this.options.ui.maxMode]) {
            this.options.viewMode = mode;
        }
    };
    DatePickerBase.prototype.viewDate = function (date, _opts) {
        var opts = Object.assign({}, { degrade: false }, _opts);
        this.datePickerState.viewDate = date;
        // fixme: triple if, oh really?
        if (this.options.viewMode && opts.degrade) {
            if (this.options.viewMode === 'years') {
                if (DatePickerViewModes.months >= DatePickerViewModes[this.options.ui.minMode]) {
                    this.options.viewMode = 'months';
                }
                else {
                    this.selectDate(date);
                }
            }
            else if (this.options.viewMode === 'months') {
                if (DatePickerViewModes.days >= DatePickerViewModes[this.options.ui.minMode]) {
                    this.options.viewMode = 'days';
                }
                else {
                    this.selectDate(date);
                }
            }
        }
    };
    DatePickerBase.prototype.activeDate = function (date) {
        if (date && this.isDisabled(date)) {
            return;
        }
        // todo: add range check
        this.datePickerState.activeDate = date;
    };
    DatePickerBase.prototype.selectDate = function (date) {
        if (this.isDisabled(date)) {
            return;
        }
        if (this.options.isDatePicker) {
            // select date
            this.datePickerState.selectedDate = date;
            this.datePickerState.selectedEndDate = void 0;
            return;
        }
        if (this.options.isDateRangePicker) {
            // if no selected then set start date
            if (!this.datePickerState.selectedDate) {
                this.datePickerState.selectedDate = date;
                return;
            }
            // if end date lesser then the start date
            if (moment(date).isBefore(this.datePickerState.selectedDate, this.viewGranularity)) {
                this.datePickerState.selectedDate = date;
                this.datePickerState.selectedEndDate = void 0;
                return;
            }
            // allow to select one date at range picker
            if (moment(date).isSame(this.datePickerState.selectedDate, this.viewGranularity)) {
                this.datePickerState.selectedEndDate = date;
                return;
            }
            // select new range start
            if (this.datePickerState.selectedEndDate) {
                this.datePickerState.selectedDate = date;
                this.datePickerState.selectedEndDate = void 0;
                return;
            }
            // don't allow to select range with disabled dates in the middle
            if (this.isDisabledDateInRange(date)) {
                return;
            }
            // if start date is selected than select end date
            if (this.isSame(this.datePickerState.selectedDate, date)) {
                this.datePickerState.selectedDate = date;
                this.datePickerState.selectedEndDate = void 0;
                return;
            }
            this.datePickerState.selectedEndDate = date;
        }
    };
    DatePickerBase.prototype.resetSelection = function () {
        this.datePickerState.selectedDate = void 0;
        this.datePickerState.selectedEndDate = void 0;
    };
    DatePickerBase.prototype.isSelected = function (date) {
        if (!date) {
            return false;
        }
        if (this.options.isDatePicker) {
            return this.isSame(this.datePickerState.selectedDate, date);
        }
        return this.isSame(this.datePickerState.selectedDate, date) ||
            this.isSame(this.datePickerState.selectedEndDate, date);
    };
    DatePickerBase.prototype.isActive = function (currDate) {
        if (this.options.isDatePicker) {
            return false;
        }
        var selectedDate = this.datePickerState.selectedDate;
        var selectedEndDate = this.datePickerState.selectedEndDate;
        var activeDate = this.datePickerState.activeDate;
        if (!selectedDate || !currDate) {
            return false;
        }
        if (selectedDate && !activeDate && !selectedEndDate) {
            return false;
        }
        if (selectedEndDate) {
            // fixme: makes sense only for day selection
            if (this.isDisabledDateInRange(selectedEndDate)) {
                return false;
            }
            return moment(currDate).isAfter(selectedDate, this.viewGranularity) &&
                moment(currDate).isBefore(selectedEndDate, this.viewGranularity);
        }
        if (this.isDisabledDateInRange(activeDate)) {
            return false;
        }
        return moment(currDate).isAfter(selectedDate, this.viewGranularity) &&
            moment(currDate).isBefore(activeDate, this.viewGranularity);
    };
    DatePickerBase.prototype.isDisabled = function (date, granularity) {
        if (granularity === void 0) { granularity = 'day'; }
        if (!date) {
            return true;
        }
        var minDate = this.options.date && this.options.date.min;
        var maxDate = this.options.date && this.options.date.max;
        if (minDate && moment(date).isSameOrBefore(minDate, granularity)) {
            return true;
        }
        if (maxDate && moment(date).isSameOrAfter(maxDate, granularity)) {
            return true;
        }
        var customDates = this.options.customDates;
        if (customDates) {
            for (var i = 0; i < customDates.length; i++) {
                if (customDates[i].isDisabled && this.isSame(customDates[i].date, date)) {
                    return true;
                }
            }
        }
        // todo: check dates options
        return false;
    };
    DatePickerBase.prototype.isSelectionStart = function (date) {
        if (!this.options.isDateRangePicker) {
            return false;
        }
        return this.isSame(date, this.datePickerState.selectedDate);
    };
    DatePickerBase.prototype.isSelectionEnd = function (date) {
        if (!this.options.isDateRangePicker) {
            return false;
        }
        return this.isSame(date, this.datePickerState.selectedEndDate);
    };
    DatePickerBase.prototype.isOtherMonth = function (date, viewDate) {
        return !moment(date).isSame(viewDate, 'month');
    };
    DatePickerBase.prototype.isHighlighted = function (date) {
        if (this.isDisabledDateInRange(date)) {
            return false;
        }
        if (!this.datePickerState.activeDate || !date) {
            return false;
        }
        return moment(date).isSame(this.datePickerState.activeDate, this.viewGranularity);
    };
    DatePickerBase.prototype.isDisabledDateInRange = function (date) {
        if (!this.options.isDateRangePicker) {
            return false;
        }
        var customDates = this.options.customDates;
        if (customDates) {
            for (var i = 0; i < customDates.length; i++) {
                if (customDates[i].isDisabled &&
                    moment(customDates[i].date).isSameOrAfter(this.datePickerState.selectedDate, this.viewGranularity) &&
                    moment(customDates[i].date).isSameOrBefore(date, this.viewGranularity)) {
                    return true;
                }
            }
        }
        return false;
    };
    DatePickerBase.prototype.markActive = function () {
        if (!this.calendar || !this.calendar.length) {
            return;
        }
        // mark proper dates as active
        for (var i = 0; i < this.calendar.length; i++) {
            for (var j = 0; j < this.calendar[i].length; j++) {
                if (this.calendar[i][j].isSelected) {
                    continue;
                }
                if (this.calendar[i][j].isDisabled) {
                    continue;
                }
                this.calendar[i][j].isActive = this.isActive(this.calendar[i][j].date);
                this.calendar[i][j].isHighlighted = this.isHighlighted(this.calendar[i][j].date);
            }
        }
    };
    DatePickerBase.prototype.markSelected = function () {
        if (!this.calendar || !this.calendar.length) {
            return;
        }
        // mark proper dates as selected
        for (var i = 0; i < this.calendar.length; i++) {
            for (var j = 0; j < this.calendar[i].length; j++) {
                var isSelected = this.isSelected(this.calendar[i][j].date);
                this.calendar[i][j].isSelected = isSelected;
                this.calendar[i][j].isSelectionStart = this.isSelectionStart(this.calendar[i][j].date);
                this.calendar[i][j].isSelectionEnd = this.isSelectionEnd(this.calendar[i][j].date);
                if (isSelected) {
                    this.calendar[i][j].isActive = false;
                    this.calendar[i][j].isHighlighted = false;
                }
            }
        }
    };
    DatePickerBase.prototype.getDaysCalendarMatrix = function (viewDate) {
        //
        // Build the matrix of dates that will populate the calendar
        //
        // current date
        var month = viewDate.month();
        var year = viewDate.year();
        // const date = viewDate.date();
        var hour = viewDate.hour();
        var minute = viewDate.minute();
        var second = viewDate.second();
        // month range
        var firstDay = moment([year, month, 1]);
        // prev
        var lastMonth = moment(firstDay).subtract(1, 'month').month();
        var lastYear = moment(firstDay).subtract(1, 'month').year();
        // initialize a 6 rows x 7 columns array for the calendar
        var calendarW = this.options.ui.dayColums;
        var calendarH = this.options.ui.dayRows;
        var calendar = new Array(calendarW);
        for (var j = 0; j < calendarW; j++) {
            calendar[j] = new Array(calendarH);
        }
        var startDay = this.getStartingDay(viewDate).date();
        // fixme: take in account time picker
        var curDate = moment([lastYear, lastMonth, startDay, 12, minute, second]);
        // where the f*** 42 came from
        for (var _a = [0, 0, 0], i = _a[0], col = _a[1], row = _a[2]; i < calendarH * calendarW; i++, col++, curDate = moment(curDate)
            .add(24, 'hour')) {
            if (i > 0 && col % 7 === 0) {
                col = 0;
                row++;
            }
            calendar[row][col] = {
                date: curDate.clone().hour(hour).minute(minute).second(second),
                label: curDate.format(this.options.format.day),
                isActive: this.isActive(curDate),
                isSelected: this.isSelected(curDate),
                isDisabled: this.isDisabled(curDate),
                isSelectionStart: this.isSelectionStart(curDate),
                isSelectionEnd: this.isSelectionEnd(curDate),
                isOtherMonth: this.isOtherMonth(curDate, viewDate),
                isHighlighted: this.isHighlighted(curDate)
            };
            curDate.hour(12);
        }
        return calendar;
    };
    DatePickerBase.prototype.getMonthsCalendarMatrix = function (viewDate /*, options:any*/) {
        var w = 3;
        var h = 4;
        var months = new Array(h);
        for (var row = 0; row < h; row++) {
            months[row] = new Array(w);
            for (var coll = 0; coll < w; coll++) {
                var monthNum = row * w + coll;
                var date = moment([viewDate.year(), monthNum, 1]);
                months[row][coll] = {
                    date: date,
                    label: date.format(this.options.format.month),
                    isActive: this.isActive(date),
                    isSelected: this.isSelected(date),
                    isDisabled: this.isDisabled(date),
                    isSelectionStart: this.isSelectionStart(date),
                    isSelectionEnd: this.isSelectionEnd(date),
                    isHighlighted: this.isHighlighted(date)
                };
            }
        }
        return months;
    };
    DatePickerBase.prototype.getYearsCalendarMatrix = function (viewDate /*, options:any*/) {
        var year = this.getStartingYear(viewDate.year());
        var cols = this.options.ui.yearColumns;
        var rows = this.options.ui.yearRows;
        var yearsMatrix = new Array(rows);
        for (var row = 0; row < rows; row++) {
            yearsMatrix[row] = new Array(cols);
            for (var coll = 0; coll < cols; coll++, year++) {
                var date = moment([year, viewDate.month()]);
                yearsMatrix[row][coll] = {
                    date: date,
                    label: date.format(this.options.format.year),
                    isActive: this.isActive(date),
                    isSelected: this.isSelected(date),
                    isDisabled: this.isDisabled(date),
                    isSelectionStart: this.isSelectionStart(date),
                    isSelectionEnd: this.isSelectionEnd(date),
                    isHighlighted: this.isHighlighted(date)
                };
            }
        }
        return yearsMatrix;
    };
    DatePickerBase.prototype.getWeeksNumbers = function (viewDate) {
        // initialize weeks row
        var calendarH = this.options.ui.dayRows;
        var startDay = this.getStartingDay(viewDate);
        var weeks = new Array(calendarH);
        var currWeek = viewDate;
        for (var i = 0; i < calendarH; i++) {
            weeks[i] = this.options.ui.showISOWeekNumbers ? startDay.format('ww') : startDay.format('WW');
            currWeek = viewDate.clone().add(1, 'week');
        }
        return weeks;
    };
    DatePickerBase.prototype.getLocale = function () {
        var localeData = moment.localeData();
        return {
            direction: 'ltr',
            format: localeData.longDateFormat('L'),
            separator: ' - ',
            applyLabel: 'Apply',
            cancelLabel: 'Cancel',
            weekLabel: 'W',
            customRangeLabel: 'Custom Range',
            weekdays: moment.weekdays(true),
            weekdaysShort: moment.weekdaysMin(true),
            monthNames: moment.monthsShort(),
            firstDay: localeData.firstDayOfWeek()
        };
    };
    DatePickerBase.prototype.getStartingDay = function (viewDate) {
        var locale = this.getLocale();
        var month = viewDate.month();
        var year = viewDate.year();
        var firstDay = moment([year, month, 1]);
        // prev
        var lastMonth = moment(firstDay).subtract(1, 'month').month();
        var lastYear = moment(firstDay).subtract(1, 'month').year();
        var daysInLastMonth = moment([lastYear, lastMonth]).daysInMonth();
        var dayOfWeek = firstDay.day();
        // populate the calendar with date objects
        var startDay = daysInLastMonth - dayOfWeek + locale.firstDay + 1;
        if (startDay > daysInLastMonth) {
            startDay -= 7;
        }
        if (dayOfWeek === locale.firstDay) {
            startDay = daysInLastMonth - 6;
        }
        return moment([year, lastMonth, startDay]);
    };
    DatePickerBase.prototype.getStartingYear = function (year) {
        var yearsStep = this.options.ui.yearColumns * this.options.ui.yearRows;
        // return ((year - 1) / this.yearsStep) * this.yearsStep + 1;
        return year - year % yearsStep;
    };
    DatePickerBase.prototype.isSame = function (date1, date2) {
        if (!date1 || !date2) {
            return false;
        }
        return moment(date1).isSame(date2, this.viewGranularity);
    };
    Object.defineProperty(DatePickerBase.prototype, "viewGranularity", {
        get: function () {
            switch (this.options.viewMode) {
                case 'days':
                    return 'day';
                case 'months':
                    return 'month';
                case 'years':
                    return 'years';
                default:
                    throw new Error('Unexpected view mode');
            }
        },
        enumerable: true,
        configurable: true
    });
    return DatePickerBase;
}());
//# sourceMappingURL=bs-date-picker-base.class.js.map