"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var proj4_1 = __importDefault(require("proj4"));
/**
 * Create a new bounding box
 * @class BoundingBox
 */
var BoundingBox = /** @class */ (function () {
    /**
     * @param  {BoundingBox | Number} minLongitudeOrBoundingBox minimum longitude or bounding box to copy (west)
     * @param  {Number} [maxLongitude]              maximum longitude (east)
     * @param  {Number} [minLatitude]               Minimum latitude (south)
     * @param  {Number} [maxLatitude]               Maximum latitude (north)
     */
    function BoundingBox(minLongitudeOrBoundingBox, maxLongitude, minLatitude, maxLatitude) {
        // if there is a second argument the first argument is the minLongitude
        if (minLongitudeOrBoundingBox instanceof BoundingBox) {
            var boundingBox = minLongitudeOrBoundingBox;
            this.minLongitude = boundingBox.minLongitude;
            this.maxLongitude = boundingBox.maxLongitude;
            this.minLatitude = boundingBox.minLatitude;
            this.maxLatitude = boundingBox.maxLatitude;
        }
        else {
            this.minLongitude = minLongitudeOrBoundingBox;
            this.maxLongitude = maxLongitude;
            this.minLatitude = minLatitude;
            this.maxLatitude = maxLatitude;
        }
    }
    /**
     * Build a Geometry Envelope from the bounding box
     *
     * @return geometry envelope
     */
    BoundingBox.prototype.buildEnvelope = function () {
        return {
            minY: this.minLatitude,
            minX: this.minLongitude,
            maxY: this.maxLatitude,
            maxX: this.maxLongitude,
        };
    };
    BoundingBox.prototype.toGeoJSON = function () {
        return {
            type: 'Feature',
            properties: {},
            geometry: {
                type: 'Polygon',
                coordinates: [
                    [
                        [this.minLongitude, this.minLatitude],
                        [this.maxLongitude, this.minLatitude],
                        [this.maxLongitude, this.maxLatitude],
                        [this.minLongitude, this.maxLatitude],
                        [this.minLongitude, this.minLatitude],
                    ],
                ],
            },
        };
    };
    /**
     * Determine if equal to the provided bounding box
     * @param  {BoundingBox} boundingBox bounding boundingBox
     * @return {Boolean}             true if equal, false if not
     */
    BoundingBox.prototype.equals = function (boundingBox) {
        if (!boundingBox) {
            return false;
        }
        if (this === boundingBox) {
            return true;
        }
        return (this.maxLatitude === boundingBox.maxLatitude &&
            this.minLatitude === boundingBox.minLatitude &&
            this.maxLongitude === boundingBox.maxLongitude &&
            this.maxLatitude === boundingBox.maxLatitude);
    };
    BoundingBox.prototype.isConverter = function (x) {
        return x.forward !== undefined;
    };
    /**
     * Project the bounding box into a new projection
     *
     * @param {string} from
     * @param {string} to
     * @return {BoundingBox}
     */
    BoundingBox.prototype.projectBoundingBox = function (from, to) {
        if (from && from !== 'undefined' && to && to !== 'undefined') {
            if (!this.isConverter(to) &&
                to.toUpperCase() === 'EPSG:3857' &&
                !this.isConverter(from) &&
                from.toUpperCase() === 'EPSG:4326') {
                this.maxLatitude = Math.min(this.maxLatitude, 85.0511);
                this.minLatitude = Math.max(this.minLatitude, -85.0511);
                this.minLongitude = Math.max(this.minLongitude, -180.0);
                this.maxLongitude = Math.min(this.maxLongitude, 180.0);
            }
            // if they are both strings, let proj4 do it
            if (!this.isConverter(to) && !this.isConverter(from)) {
                var min = proj4_1.default(from, to, [this.minLongitude, this.minLatitude]);
                var max = proj4_1.default(from, to, [this.maxLongitude, this.maxLatitude]);
                var projected = new BoundingBox(min[0], max[0], min[1], max[1]);
                return projected;
            }
            // if they are both projections to from.inverse then to.forward
            else {
                var toConverter = void 0;
                if (this.isConverter(to)) {
                    toConverter = to;
                }
                else {
                    toConverter = proj4_1.default(to);
                }
                var fromConverter = void 0;
                if (this.isConverter(from)) {
                    fromConverter = from;
                }
                else {
                    fromConverter = proj4_1.default(from);
                }
                var min = toConverter.forward(fromConverter.inverse([this.minLongitude, this.minLatitude]));
                var max = toConverter.forward(fromConverter.inverse([this.maxLongitude, this.maxLatitude]));
                var projected = new BoundingBox(min[0], max[0], min[1], max[1]);
                return projected;
            }
        }
        return this;
    };
    return BoundingBox;
}());
exports.BoundingBox = BoundingBox;
//# sourceMappingURL=boundingBox.js.map