/// <reference types="proj4" />
import { Dao } from '../../dao/dao';
import { Contents } from './contents';
import { GeometryColumns } from '../../features/columns/geometryColumns';
import { TileMatrixSet } from '../../tiles/matrixset/tileMatrixSet';
import { TileMatrix } from '../../tiles/matrix/tileMatrix';
import { SpatialReferenceSystem } from '../srs/spatialReferenceSystem';
import { DBValue } from '../../db/dbAdapter';
/**
 * Contents object. Provides identifying and descriptive information that an
 * application can display to a user in a menu of geospatial data that is
 * available for access and/or update.
 * @class ContentsDao
 * @extends Dao
 */
export declare class ContentsDao extends Dao<Contents> {
    static readonly TABLE_NAME: string;
    static readonly COLUMN_PK: string;
    static readonly COLUMN_TABLE_NAME: string;
    static readonly COLUMN_DATA_TYPE: string;
    static readonly COLUMN_IDENTIFIER: string;
    static readonly COLUMN_DESCRIPTION: string;
    static readonly COLUMN_LAST_CHANGE: string;
    static readonly COLUMN_MIN_X: string;
    static readonly COLUMN_MIN_Y: string;
    static readonly COLUMN_MAX_X: string;
    static readonly COLUMN_MAX_Y: string;
    static readonly COLUMN_SRS_ID: string;
    static readonly GPKG_CDT_FEATURES_NAME: string;
    static readonly GPKG_CDT_TILES_NAME: string;
    static readonly GPKG_CDT_ATTRIBUTES_NAME: string;
    readonly gpkgTableName: string;
    readonly idColumns: string[];
    /**
     * Creates a new Contents object
     * @return {module:core/contents~Contents} new Contents object
     */
    createObject(results?: Record<string, DBValue>): Contents;
    /**
     * Get table names by table type
     * @param  {string} [tableType] table type to query for
     * @return {string[]}           Array of table names
     */
    getTables(tableType?: string): string[];
    getContentsForTableType(tableType?: string, reprojectTo4326?: boolean): Contents[];
    /**
     * Returns the proj4 projection for the Contents
     * @param  {module:core/contents~Contents} contents Contents to get the projection from
     * @return {*}          proj4 projection
     */
    getProjection(contents: Contents): proj4.Converter;
    /**
     * Get the SpatialReferenceSystemDao for the Contents
     * @param  {module:core/contents~Contents} contents Contents to get the SpatialReferenceSystemDao from
     * @return {module:core/srs~SpatialReferenceSystemDao}
     */
    getSrs(contents: Contents): SpatialReferenceSystem;
    /**
     * Get the GeometryColumns for the Contents
     * @param  {module:core/contents~Contents} contents Contents
     * @return {module:features/columns~GeometryColumns}
     */
    getGeometryColumns(contents: Contents): GeometryColumns;
    /**
     * Get the TileMatrixSet for the Contents
     * @param  {module:core/contents~Contents} contents Contents
     * @return {module:tiles/matrixset~TileMatrixSet}
     */
    getTileMatrixSet(contents: Contents): TileMatrixSet;
    /**
     * Get the TileMatrix for the Contents
     * @param  {module:core/contents~Contents} contents Contents
     * @return {module:tiles/matrix~TileMatrix}
     */
    getTileMatrix(contents: Contents): TileMatrix[];
}
