import { DBValue } from './dbAdapter';
/**
 * SQLite query builder module.
 * @module db/sqliteQueryBuilder
 */
/**
 * Utility class to build sql queries
 * @class
 */
export declare class SqliteQueryBuilder {
    /**
     * Replaces all whitespace in a column name with underscores
     * @param  {string} columnName column name to fix
     * @return {string}
     */
    static fixColumnName(columnName: string): string;
    /**
     * Builds a query
     * @param  {Boolean} distinct whether query should be distinct or not
     * @param  {string} tables   table names to query, added to the query from clause
     * @param  {string[]} [columns]  columns to query for
     * @param  {string} [where]    where clause
     * @param  {string} [join]     join clause
     * @param  {string} [groupBy]  group by clause
     * @param  {string} [having]   having clause
     * @param  {string} [orderBy]  order by clause
     * @param  {Number} [limit]    limit
     * @param  {Number} [offset]   offset
     * @return {string}
     */
    static buildQuery(distinct: boolean, tables: string, columns?: string[], where?: string, join?: string, groupBy?: string, having?: string, orderBy?: string, limit?: number, offset?: number): string;
    /**
     * Builds a count statement
     * @param  {string} tables table names to query for
     * @param  {string} [where]  where clause
     * @return {string} count statement
     */
    static buildCount(tables: string, where?: string): string;
    /**
     * Builds an insert statement using the properties of the object
     * @param  {string} table  table to insert into
     * @param  {Object} object object to insert
     * @return {string} insert statement
     */
    static buildInsert(table: string, object: any): string;
    /**
     * Builds an insert statement from the object.getColumnNames method
     * @param  {string} table  table to insert into
     * @param  {Object} object object with a getColumnNames method
     * @return {string} insert statement
     */
    static buildInsertFromColumnNames(table: string, object: any): string;
    /**
     * Builds an update or insert object to bind to a statement
     * @param  {Object} object object to create bind parameters from
     * @return {Object} bind parameters
     */
    static buildUpdateOrInsertObject(object: any): any;
    /**
     * Builds an update statement
     * @param  {string} table     table to update
     * @param  {Object} values    object with values to update
     * @param  {string} [where]     where clause
     * @param  {Array|Object} [whereArgs] where bind parameters
     * @return {Object} object with a sql property containing the update statement and an args property with bind arguments
     */
    static buildUpdate(table: string, values: Record<string, DBValue>, where: string, whereArgs: DBValue[] | DBValue): {
        sql: string;
        args: DBValue[];
    };
    /**
     * Builds an update from an object
     * @param  {string} table  table name to update
     * @param  {Object} object object with values to update
     * @return {string} update statement
     */
    static buildObjectUpdate(table: string, object: any): string;
    private static appendClauseToString;
    private static appendColumnsToString;
    private static columnToAppend;
    private static isEmpty;
}
