/// <reference types="proj4" />
/**
 * Feature Table Index
 * @module extension/index
 */
import { RTreeIndex } from '../rtree/rtreeIndex';
import { BaseExtension } from '../baseExtension';
import { GeoPackage } from '../../geoPackage';
import { Extension } from '../extension';
import { TableIndex } from './tableIndex';
import { FeatureDao } from '../../features/user/featureDao';
import { GeometryIndexDao } from './geometryIndexDao';
import { RTreeIndexDao } from '../rtree/rtreeIndexDao';
import { TableIndexDao } from './tableIndexDao';
import { Envelope } from '../../geom/envelope';
import { FeatureRow } from '../../features/user/featureRow';
import { GeometryData } from '../../geom/geometryData';
import { BoundingBox } from '../../boundingBox';
/**
 * This class will either use the RTree index if it exists, or the
 * Feature Table Index NGA Extension implementation. This extension is used to
 * index Geometries within a feature table by their minimum bounding box for
 * bounding box queries.
 * @class
 * @extends BaseExtension
 */
export declare class FeatureTableIndex extends BaseExtension {
    featureDao: FeatureDao<FeatureRow>;
    static readonly EXTENSION_GEOMETRY_INDEX_AUTHOR: string;
    static readonly EXTENSION_GEOMETRY_INDEX_NAME_NO_AUTHOR: string;
    static readonly EXTENSION_GEOMETRY_INDEX_DEFINITION: string;
    progress: Function;
    tableName: string;
    columnName: string;
    tableIndexDao: TableIndexDao;
    geometryIndexDao: GeometryIndexDao;
    rtreeIndexDao: RTreeIndexDao;
    rtreeIndex: RTreeIndex;
    rtreeIndexed: boolean;
    /**
     *
     * @param geoPackage GeoPackage object
     * @param featureDao FeatureDao to index
     */
    constructor(geoPackage: GeoPackage, featureDao: FeatureDao<FeatureRow>);
    /**
     * Index the table if not already indexed
     * @param  {Function} progress function which is called with progress while indexing
     * @return {Promise<Boolean>} promise resolved when the indexing is complete
     */
    index(progress?: Function): Promise<boolean>;
    /**
     * Index the table if not already indexed or force is true
     * @param  {Boolean} force force index even if the table is already indexed
     * @param  {Function} progress function which is called with progress while indexing
     * @return {Promise<Boolean>} promise resolved when the indexing is complete
     */
    indexWithForce(force?: false, progress?: Function): Promise<boolean>;
    /**
     * Index the table using the NGA index and Rtree if not already indexed or force is true
     * @param  {Boolean} force force index even if the table is already indexed
     * @param  {Function} progress function which is called with progress while indexing
     * @return {Promise<Boolean>} promise resolved when the indexing is complete
     */
    ngaIndexWithForce(force?: false, progress?: Function): Promise<boolean>;
    /**
     * Check if the table is indexed either with an RTree or the NGA Feature Table Index
     * @return {Boolean}
     */
    isIndexed(checkOnlyNGA?: boolean): boolean;
    /**
     * Returns the feature table index extension for this table and column name if exists
     * @return {module:extension~Extension}
     */
    getFeatureTableIndexExtension(): Extension;
    /**
     * Get or create the extension for this table name and column name
     * @return {module:extension~Extension}
     */
    getOrCreateExtension(): Promise<Extension>;
    /**
     * Get or create if needed the table index
     * @return {Promise<TableIndex>}
     */
    getOrCreateTableIndex(): Promise<TableIndex>;
    /**
     * Create the table index
     * @return {module:extension/index~TableIndex}
     */
    createTableIndex(): number;
    /**
     * Get the table index
     * @return {module:extension/index~TableIndex}
     */
    getTableIndex(): TableIndex;
    /**
     * Clear the geometry indices or create the table if needed
     * @return {Promise} resolved when complete
     */
    createOrClearGeometryIndicies(): Promise<number>;
    /**
     * Clears the geometry indices
     * @return {Number} number of rows deleted
     */
    clearGeometryIndicies(): number;
    /**
     * Indexes the table
     * @param  {module:extension/index~TableIndex} tableIndex TableIndex
     * @return {Promise} resolved when complete
     */
    indexTable(tableIndex: TableIndex): Promise<boolean>;
    /**
     * Indexes a chunk of 100 rows
     * @param  {Number} page       page to start on
     * @param  {module:extension/index~TableIndex} tableIndex TableIndex
     * @param  {Function} resolve    function to call when all chunks are indexed
     * @param  {Function} reject     called if there is an error
     */
    indexChunk(page: number, tableIndex: TableIndex, resolve: Function, reject: Function): void;
    /**
     * Indexes a row
     * @param  {ableIndex} tableIndex TableIndex`
     * @param  {Number} geomId     id of the row
     * @param  {GeometryData} geomData   GeometryData to index
     * @return {Boolean} success
     */
    indexRow(tableIndex: TableIndex, geomId: number, geomData: GeometryData): boolean;
    /**
     * Update the last time this feature table was indexed
     * @param  {module:extension/index~TableIndex} tableIndex TableIndex
     * @return {Object} update status
     */
    updateLastIndexed(tableIndex: TableIndex): number;
    /**
     * Query the index with the specified bounding box and projection
     * @param  {module:boundingBox~BoundingBox} boundingBox bounding box to query for
     * @param  {string} projection  projection the boundingBox is in
     * @return {IterableIterator}
     */
    queryWithBoundingBox(boundingBox: BoundingBox, projection: string | proj4.Converter): IterableIterator<any>;
    /**
     * Query witha geometry envelope
     * @param  {any} envelope envelope
     * @return {IterableIterator<any>}
     */
    queryWithGeometryEnvelope(envelope: Envelope): IterableIterator<any>;
    /**
     * Count the index with the specified bounding box and projection
     * @param  {module:boundingBox~BoundingBox} boundingBox bounding box to query for
     * @param  {string} projection  projection the boundingBox is in
     * @return {Number}
     */
    countWithBoundingBox(boundingBox: BoundingBox, projection: string): number;
    /**
     * Count with a geometry envelope
     * @param  {any} envelope envelope
     * @return {Number}
     */
    countWithGeometryEnvelope(envelope: Envelope): number;
}
