/**
 * RelatedTablesExtension module.
 * @module extension/relatedTables
 * @see module:extension/BaseExtension
 */
import { BaseExtension } from '../baseExtension';
import { Extension } from '../extension';
import { MediaDao } from './mediaDao';
import { MediaTable } from './mediaTable';
import { SimpleAttributesDao } from './simpleAttributesDao';
import { SimpleAttributesTable } from './simpleAttributesTable';
import { UserMappingTable } from './userMappingTable';
import { UserMappingDao } from './userMappingDao';
import { UserCustomDao } from '../../user/custom/userCustomDao';
import { ExtendedRelationDao } from './extendedRelationDao';
import { RelationType } from './relationType';
import { ExtendedRelation } from './extendedRelation';
import { GeoPackage } from '../../geoPackage';
import { UserRelatedTable } from './userRelatedTable';
import { UserRow } from '../../user/userRow';
import { UserMappingRow } from './userMappingRow';
import { MediaRow } from './mediaRow';
import { SimpleAttributesRow } from './simpleAttributesRow';
/**
 * Related Tables Extension
 * @param  {module:geoPackage~GeoPackage} geoPackage the GeoPackage object
 * @class
 * @extends BaseExtension
 */
export declare class RelatedTablesExtension extends BaseExtension {
    extendedRelationDao: ExtendedRelationDao;
    static readonly EXTENSION_NAME: string;
    static readonly EXTENSION_RELATED_TABLES_AUTHOR: string;
    static readonly EXTENSION_RELATED_TABLES_NAME_NO_AUTHOR: string;
    static readonly EXTENSION_RELATED_TABLES_DEFINITION: string;
    constructor(geoPackage: GeoPackage);
    /**
     * Get or create the extension
     * @return {Promise}
     */
    getOrCreateExtension(): Promise<Extension>;
    /**
     * Get or create the extension for the mapping table
     * @param  {string} mappingTableName user mapping table
     * @return {Promise}
     */
    getOrCreateMappingTable(mappingTableName: string): Promise<Extension>;
    /**
     * Set the contents in the UserRelatedTable
     * @param  {module:extension/relatedTables~UserRelatedTable} userRelatedTable user related table
     */
    setContents(userRelatedTable: UserRelatedTable): boolean;
    /**
     * Reads the user table and creates a UserCustomDao
     * @param  {string} tableName       table name to reader
     * @param  {string[]} requiredColumns required columns
     * @return {module:user/custom~UserCustomDao}
     */
    getUserDao(tableName: string, requiredColumns?: string[]): UserCustomDao<UserRow>;
    /**
     * Gets the UserMappingDao from the mapping table name
     * @param  {string | ExtendedRelation} tableName user mapping table name or ExtendedRelation object
     * @return {module:extension/relatedTables~UserMappingDao}
     */
    getMappingDao(tableName: string | ExtendedRelation): UserMappingDao<UserMappingRow>;
    /**
     * Gets all relationships in the GeoPackage with an optional base table name and an optional base id
     * @param {String} [baseTableName] base table name
     * @return {module:extension/relatedTables~ExtendedRelation[]}
     */
    getRelationships(baseTableName?: string): ExtendedRelation[];
    /**
     * Gets all relationships in the GeoPackage with an optional base table name and an optional base id
     * @param {String} [baseTableName] base table name
     * @param {String} [relatedTableName] related table name
     * @param {String} [mappingTableName] mapping table name
     * @return {Boolean}
     */
    hasRelations(baseTableName?: string, relatedTableName?: string, mappingTableName?: string): boolean;
    getRelatedRows(baseTableName: string, baseId: number): ExtendedRelation[];
    /**
     * Convience object to build a Relationship object for querying and adding
     * @typedef {Object} module:extension/relatedTables~Relationship
     * @property  {module:extension/relatedTables~RelationType} relationType type of relationship
     * @property  {string} baseTableName base table name
     * @property  {string} relatedTableName related table name
     * @property  {string} relationAuthor relationship author
     * @property  {string} mappingTableName mapping table name
     * @property  {module:extension/relatedTables~UserMappingTable} userMappingTable UserMappingTable
     * @property  {module:extension/relatedTables~UserRelatedTable} relatedTable UserRelatedTable
     */
    getRelationshipBuilder(): any;
    /**
     * Adds a relationship to the GeoPackage
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation | undefined>}
     */
    addRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Get the primary key column name from the specified table
     * @param  {string} tableName table name
     * @return {string}
     */
    getPrimaryKeyColumnName(tableName: string): string;
    /**
     * Adds a features relationship between the base feature and related feature
     * table. Creates a default user mapping table if needed.
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation>}
     */
    addFeaturesRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Adds a tiles relationship between the base table and related tile
     * table. Creates a default user mapping table if needed.
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation>}
     */
    addTilesRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Adds an attributes relationship between the base table and related attribute
     * table. Creates a default user mapping table if needed.
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation>}
     */
    addAttributesRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Adds a simple attributes relationship between the base table and user
     * simple attributes related table. Creates a default user mapping table and
     * the simple attributes table if needed.
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation>}
     */
    addSimpleAttributesRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Adds a media relationship between the base table and user media related
     * table. Creates a default user mapping table and the media table if
     * needed.
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to add
     * @return {Promise<ExtendedRelation>}
     */
    addMediaRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: UserMappingTable;
        relatedTable?: UserRelatedTable;
    }): Promise<ExtendedRelation>;
    /**
     * Remove a specific relationship from the GeoPackage
     * @param  {module:extension/relatedTables~Relationship|module:extension/relatedTables~ExtendedRelation} relationship relationship to remove
     * @return {Number} number of relationships removed
     */
    removeRelationship(relationship: ExtendedRelation | {
        relationType?: RelationType;
        relationName?: string;
        baseTableName?: string;
        relatedTableName?: string;
        relationAuthor?: string;
        mappingTableName?: string;
        userMappingTable?: string;
        relatedTable?: UserRelatedTable;
    }): number;
    /**
     * Create a default user mapping table and extension row if either does not
     * exist. When not created, there is no guarantee that an existing table has
     * the same schema as the provided tabled.
     * @param  {string | UserMappingTable} userMappingTableOrName user mapping table or name
     * @return {Promise<Boolean>}
     */
    createUserMappingTable(userMappingTableOrName: UserMappingTable | string): Promise<boolean>;
    /**
     * Create a user related table if it does not exist. When not created, there
     * is no guarantee that an existing table has the same schema as the
     * provided tabled.
     * @param  {module:extension/relatedTables~UserRelatedTable} relatedTable user related table
     * @return {Boolean} true if the table now exists
     */
    createRelatedTable(relatedTable: UserRelatedTable): boolean;
    /**
     * Validate that the relation name is valid between the base and related tables
     * @param  {string} baseTableName    base table name
     * @param  {string} relatedTableName related table name
     * @param  {string} relationName     relation name
     * @return {Boolean}
     */
    validateRelationship(baseTableName: string, relatedTableName: string, relationName: string): boolean;
    /**
     * Link related Ids
     * @param  {string} baseTableName    base table name
     * @param  {Number} baseId           base row id
     * @param  {string} relatedTableName related table name
     * @param  {Number} relatedId        related row id
     * @param  {module:extension/relatedTables~RelationType} relationType     relation type
     * @return {Promise}
     */
    linkRelatedIds(baseTableName: string, baseId: number, relatedTableName: string, relatedId: number, relationType: RelationType): Promise<number>;
    /**
     * Get the related id mappings for the base id
     * @param  {string} mappingTableName mapping table name
     * @param  {Number} baseId           base id
     * @return {Number[]} ids of related items
     */
    getMappingsForBase(mappingTableName: string | ExtendedRelation, baseId: number): number[];
    /**
     * Get the related id mapping rows for the base id
     * @param  {string} mappingTableName mapping table name
     * @param  {Number} baseId           base id
     * @return {module:extension/relatedTables~UserMappingRow[]} user mapping rows
     */
    getMappingRowsForBase(mappingTableName: string | ExtendedRelation, baseId: number): UserMappingRow[];
    /**
     * Get the base id mappings for the base id
     * @param  {string} mappingTableName mapping table name
     * @param  {Number} relatedId           related id
     * @return {Number[]} ids of base items
     */
    getMappingsForRelated(mappingTableName: string | ExtendedRelation, relatedId: number): number[];
    /**
     * Returns a {module:extension/relatedTables~MediaDao} from the table specified
     * @param  {string|MediaTable|ExtendedRelation} tableName either a table name or a MediaTable
     * @return {module:extension/relatedTables~MediaDao}
     */
    getMediaDao(tableName: string | MediaTable | ExtendedRelation): MediaDao<MediaRow>;
    /**
     * Returns a {module:extension/relatedTables~SimpleAttributesDao} from the table specified
     * @param  {string|SimpleAttributesTable|ExtendedRelation} tableName either a table name or a SimpleAttributesDao
     * @return {module:extension/relatedTables~SimpleAttributesDao}
     */
    getSimpleAttributesDao(tableName: string | ExtendedRelation | SimpleAttributesTable): SimpleAttributesDao<SimpleAttributesRow>;
    /**
     * Builds the custom relation name with the author
     * @param  {string} author author
     * @param  {string} name   name
     * @return {string}
     */
    buildRelationName(author: string, name: string): string;
    /**
     * Remove all traces of the extension
     */
    removeExtension(): void;
    /**
     * Determine if the GeoPackage has the extension
     * @param  {String} [mappingTableName] mapping table name to check, if not specified, this checks for any mapping table name
     * @return {Boolean}
     */
    has(mappingTableName?: string): boolean;
    static RelationshipBuilder(): any;
}
