/**
 * @memberOf module:extension/style
 * @class FeatureStyleExtension
 */
import { FeatureStyles } from './featureStyles';
import { FeatureStyle } from './featureStyle';
import { Styles } from './styles';
import { Icons } from './icons';
import { FeatureStyleExtension } from '.';
import { GeoPackage } from '../../geoPackage';
import { ExtendedRelation } from '../relatedTables/extendedRelation';
import { StyleMappingDao } from './styleMappingDao';
import { StyleDao } from './styleDao';
import { IconDao } from './iconDao';
import { StyleRow } from './styleRow';
import { IconRow } from './iconRow';
import { FeatureRow } from '../../features/user/featureRow';
import { FeatureTable } from '../../features/user/featureTable';
/**
 * Feature Table Styles, styles and icons for an individual feature table
 * @param  {module:geoPackage~GeoPackage} geoPackage GeoPackage object
 * @param {String} tableName
 * @constructor
 */
export declare class FeatureTableStyles {
    geoPackage: GeoPackage;
    featureStyleExtension: FeatureStyleExtension;
    cachedTableFeatureStyles: FeatureStyles;
    tableName: string;
    constructor(geoPackage: GeoPackage, tableNameOrTable: string | FeatureTable);
    /**
     * Get the feature style extension
     * @return {module:extension/style.FeatureStyleExtension} feature style extension
     */
    getFeatureStyleExtension(): FeatureStyleExtension;
    /**
     * Get the feature table name
     * @return {String} feature table name
     */
    getTableName(): string;
    /**
     * Determine if the GeoPackage has the extension for the table
     * @return {Boolean} true if has extension
     */
    has(): boolean;
    /**
     * Create style, icon, table style, and table icon relationships for the
     * feature table
     * @return {Promise}
     */
    createRelationships(): Promise<{
        styleRelationship: ExtendedRelation;
        tableStyleRelationship: ExtendedRelation;
        iconRelationship: ExtendedRelation;
        tableIconRelationship: ExtendedRelation;
    }>;
    /**
     * Check if feature table has a style, icon, table style, or table icon
     * relationships
     * @return {Boolean} true if has a relationship
     */
    hasRelationship(): boolean;
    /**
     * Create a style relationship for the feature table
     * @return {Promise}
     */
    createStyleRelationship(): Promise<ExtendedRelation>;
    /**
     * Determine if a style relationship exists for the feature table
     * @return {Boolean} true if relationship exists
     */
    hasStyleRelationship(): boolean;
    /**
     * Create a feature table style relationship
     * @return {Promise}
     */
    createTableStyleRelationship(): Promise<ExtendedRelation>;
    /**
     * Determine if feature table style relationship exists
     *
     * @return {Boolean} true if relationship exists
     */
    hasTableStyleRelationship(): boolean;
    /**
     * Create an icon relationship for the feature table
     * @return {Promise}
     */
    createIconRelationship(): Promise<ExtendedRelation>;
    /**
     * Determine if an icon relationship exists for the feature table
     * @return {Boolean} true if relationship exists
     */
    hasIconRelationship(): boolean;
    /**
     * Create a feature table icon relationship
     * @return {Promise}
     */
    createTableIconRelationship(): Promise<ExtendedRelation>;
    /**
     * Determine if feature table icon relationship exists
     * @return {Boolean} true if relationship exists
     */
    hasTableIconRelationship(): boolean;
    /**
     * Delete the style and icon table and row relationships for the feature
     * table
     */
    deleteRelationships(): {
        styleRelationships: number;
        tableStyleRelationships: number;
        iconRelationship: number;
        tableIconRelationship: number;
    };
    /**
     * Delete a style relationship for the feature table
     */
    deleteStyleRelationship(): number;
    /**
     * Delete a table style relationship for the feature table
     */
    deleteTableStyleRelationship(): number;
    /**
     * Delete a icon relationship for the feature table
     */
    deleteIconRelationship(): number;
    /**
     * Delete a table icon relationship for the feature table
     */
    deleteTableIconRelationship(): number;
    /**
     * Get a Style Mapping DAO
     * @return {module:extension/style.StyleMappingDao} style mapping DAO
     */
    getStyleMappingDao(): StyleMappingDao;
    /**
     * Get a Table Style Mapping DAO
     * @return {module:extension/style.StyleMappingDao} table style mapping DAO
     */
    getTableStyleMappingDao(): StyleMappingDao;
    /**
     * Get a Icon Mapping DAO
     * @return {module:extension/style.StyleMappingDao} icon mapping DAO
     */
    getIconMappingDao(): StyleMappingDao;
    /**
     * Get a Table Icon Mapping DAO
     * @return {module:extension/style.StyleMappingDao} table icon mapping DAO
     */
    getTableIconMappingDao(): StyleMappingDao;
    /**
     * Get a style DAO
     * @return {module:extension/style.StyleDao} style DAO
     */
    getStyleDao(): StyleDao;
    /**
     * Get a icon DAO
     * @return {module:extension/style.IconDao} icon DAO
     */
    getIconDao(): IconDao;
    /**
     * Get the table feature styles
     * @return {module:extension/style.FeatureStyles} table feature styles or null
     */
    getTableFeatureStyles(): FeatureStyles;
    /**
     * Get the table styles
     * @return {module:extension/style.Styles} table styles or null
     */
    getTableStyles(): Styles;
    /**
     * Get the cached table styles, querying and caching if needed
     * @return {module:extension/style.Styles} cached table styles
     */
    getCachedTableStyles(): Styles;
    /**
     * Get the table style of the geometry type
     * @param {String} geometryType geometry type
     * @return {module:extension/style.StyleRow} style row
     */
    getTableStyle(geometryType: string): StyleRow;
    /**
     * Get the table style default
     * @return {module:extension/style.StyleRow} style row
     */
    getTableStyleDefault(): StyleRow;
    /**
     * Get the table icons
     * @return {module:extension/style.Icons} table icons or null
     */
    getTableIcons(): Icons;
    /**
     * Get the cached table icons, querying and caching if needed
     * @return {module:extension/style.Icons} cached table icons
     */
    getCachedTableIcons(): Icons;
    /**
     * Get the table icon of the geometry type
     * @param {String} geometryType geometry type
     * @return {module:extension/style.IconRow} icon row
     */
    getTableIcon(geometryType: string): IconRow;
    /**
     * Get the table icon default
     * @return {module:extension/style.IconRow} icon row
     */
    getTableIconDefault(): IconRow;
    /**
     * Get the feature styles for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.FeatureStyles} feature styles or null
     */
    getFeatureStylesForFeatureRow(featureRow: FeatureRow): FeatureStyles;
    /**
     * Get the feature styles for the feature id
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.FeatureStyles} feature styles or null
     */
    getFeatureStyles(featureId: number): FeatureStyles;
    /**
     * Get the feature style (style and icon) of the feature row, searching in
     * order: feature geometry type style or icon, feature default style or
     * icon, table geometry type style or icon, table default style or icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.FeatureStyle} feature style
     */
    getFeatureStyleForFeatureRow(featureRow: FeatureRow): FeatureStyle;
    /**
     * Get the feature style (style and icon) of the feature row with the
     * provided geometry type, searching in order: feature geometry type style
     * or icon, feature default style or icon, table geometry type style or
     * icon, table default style or icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @return {module:extension/style.FeatureStyle} feature style
     */
    getFeatureStyleForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string): FeatureStyle;
    /**
     * Get the feature style default (style and icon) of the feature row,
     * searching in order: feature default style or icon, table default style or
     * icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.FeatureStyle} feature style
     */
    getFeatureStyleDefaultForFeatureRow(featureRow: FeatureRow): FeatureStyle;
    /**
     * Get the feature style (style and icon) of the feature, searching in
     * order: feature geometry type style or icon, feature default style or
     * icon, table geometry type style or icon, table default style or icon
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @return {module:extension/style.FeatureStyle} feature style
     */
    getFeatureStyle(featureId: number, geometryType: string): FeatureStyle;
    /**
     * Get the feature style (style and icon) of the feature, searching in
     * order: feature geometry type style or icon, feature default style or
     * icon, table geometry type style or icon, table default style or icon
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.FeatureStyle} feature style
     */
    getFeatureStyleDefault(featureId: number): FeatureStyle;
    /**
     * Get the styles for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.Styles} styles or null
     */
    getStylesForFeatureRow(featureRow: FeatureRow): Styles;
    /**
     * Get the styles for the feature id
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.Styles}  styles or null
     */
    getStylesForFeatureId(featureId: number): Styles;
    /**
     * Get the style of the feature row, searching in order: feature geometry
     * type style, feature default style, table geometry type style, table
     * default style
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.StyleRow} style row
     */
    getStyleForFeatureRow(featureRow: FeatureRow): StyleRow;
    /**
     * Get the style of the feature row with the provided geometry type,
     * searching in order: feature geometry type style, feature default style,
     * table geometry type style, table default style
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @return {module:extension/style.StyleRow} style row
     */
    getStyleForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string): StyleRow;
    /**
     * Get the default style of the feature row, searching in order: feature
     * default style, table default style
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.StyleRow} style row
     */
    getStyleDefaultForFeatureRow(featureRow: FeatureRow): StyleRow;
    /**
     * Get the style of the feature, searching in order: feature geometry type
     * style, feature default style, table geometry type style, table default
     * style
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @return {module:extension/style.StyleRow} style row
     */
    getStyle(featureId: number, geometryType: string): StyleRow;
    /**
     * Get the default style of the feature, searching in order: feature default
     * style, table default style
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.StyleRow} style row
     */
    getStyleDefault(featureId: number): StyleRow;
    /**
     * Get the icons for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.Icons} icons or null
     */
    getIconsForFeatureRow(featureRow: FeatureRow): Icons;
    /**
     * Get the icons for the feature id
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.Icons} icons or null
     */
    getIconsForFeatureId(featureId: number): Icons;
    /**
     * Get the icon of the feature row, searching in order: feature geometry
     * type icon, feature default icon, table geometry type icon, table default
     * icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.IconRow} icon row
     */
    getIconForFeatureRow(featureRow: FeatureRow): IconRow;
    /**
     * Get the icon of the feature row with the provided geometry type,
     * searching in order: feature geometry type icon, feature default icon,
     * table geometry type icon, table default icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @return {module:extension/style.IconRow} icon row
     */
    getIconForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string): IconRow;
    /**
     * Get the default icon of the feature row, searching in order: feature
     * default icon, table default icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @return {module:extension/style.IconRow} icon row
     */
    getIconDefaultForFeatureRow(featureRow: FeatureRow): IconRow;
    /**
     * Get the icon of the feature, searching in order: feature geometry type
     * icon, feature default icon, table geometry type icon, table default icon
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @return {module:extension/style.IconRow} icon row
     */
    getIcon(featureId: number, geometryType: string): IconRow;
    /**
     * Get the default icon of the feature, searching in order: feature default
     * icon, table default icon
     *
     * @param {Number} featureId feature id
     * @return {module:extension/style.IconRow} icon row
     */
    getIconDefault(featureId: number): IconRow;
    /**
     * Set the feature table default feature styles
     *
     * @param {module:extension/style.FeatureStyles} featureStyles default feature styles
     * @return {Promise}
     */
    setTableFeatureStyles(featureStyles: FeatureStyles): Promise<{
        tableStyles: {
            styleDefault: number;
            styles: number[];
        };
        tableIcons: {
            iconDefault: number;
            icons: number[];
        };
        deleted?: {
            styles: number;
            icons: number;
        };
    }>;
    /**
     * Set the feature table default styles
     *
     * @param {module:extension/style.Styles} styles default styles
     * @return {Promise}
     */
    setTableStyles(styles: Styles): Promise<{
        styleDefault: number;
        styles: number[];
        deleted: number;
    }>;
    /**
     * Set the feature table style default
     *
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setTableStyleDefault(style: StyleRow): Promise<number>;
    /**
     * Set the feature table style for the geometry type
     *
     * @param {String} geometryType geometry type
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setTableStyle(geometryType: string, style: StyleRow): Promise<number>;
    /**
     * Set the feature table default icons
     *
     * @param {module:extension/style.Icons} icons default icons
     * @return {Promise}
     */
    setTableIcons(icons: Icons): Promise<{
        iconDefault: number;
        icons: number[];
        deleted: number;
    }>;
    /**
     * Set the feature table icon default
     *
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setTableIconDefault(icon: IconRow): Promise<number>;
    /**
     * Set the feature table icon for the geometry type
     *
     * @param {String} geometryType geometry type
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setTableIcon(geometryType: string, icon: IconRow): Promise<number>;
    /**
     * Set the feature styles for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.FeatureStyles} featureStyles feature styles
     * @return {Promise}
     */
    setFeatureStylesForFeatureRow(featureRow: FeatureRow, featureStyles: FeatureStyles): Promise<{
        styles: {
            styleDefault: number;
            styles: number[];
        };
        icons: {
            iconDefault: number;
            icons: number[];
            deleted?: {
                style: number;
                icon: number;
            };
        };
    }>;
    /**
     * Set the feature styles for the feature table and feature id
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.FeatureStyles} featureStyles feature styles
     * @return {Promise}
     */
    setFeatureStyles(featureId: number, featureStyles: FeatureStyles): Promise<{
        styles: {
            styleDefault: number;
            styles: number[];
        };
        icons: {
            iconDefault: number;
            icons: number[];
        };
        deleted?: {
            deletedStyles: number;
            deletedIcons: number;
        };
    }>;
    /**
     * Set the feature style (style and icon) of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.FeatureStyle} featureStyle feature style
     * @return {Promise}
     */
    setFeatureStyleForFeatureRow(featureRow: FeatureRow, featureStyle: FeatureStyle): Promise<{
        style: number;
        icon: number;
        deleted?: {
            style: number;
            icon: number;
        };
    }>;
    /**
     * Set the feature style (style and icon) of the feature row for the
     * specified geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @param {module:extension/style.FeatureStyle} featureStyle feature style
     * @return {Promise}
     */
    setFeatureStyleForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string, featureStyle: FeatureStyle): Promise<{
        style: number;
        icon: number;
        deleted?: {
            style: number;
            icon: number;
        };
    }>;
    /**
     * Set the feature style default (style and icon) of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.FeatureStyle} featureStyle feature style
     * @return {Promise}
     */
    setFeatureStyleDefaultForFeatureRow(featureRow: FeatureRow, featureStyle: FeatureStyle): Promise<{
        style: number;
        icon: number;
        deleted?: {
            style: number;
            icon: number;
        };
    }>;
    /**
     * Set the feature style (style and icon) of the feature
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @param {module:extension/style.FeatureStyle} featureStyle feature style
     * @return {Promise}
     */
    setFeatureStyle(featureId: number, geometryType: string, featureStyle: FeatureStyle): Promise<{
        style: number;
        icon: number;
        deleted?: {
            style: number;
            icon: number;
        };
    }>;
    /**
     * Set the feature style (style and icon) of the feature
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.FeatureStyle} featureStyle feature style
     * @return {Promise}
     */
    setFeatureStyleDefault(featureId: number, featureStyle: FeatureStyle): Promise<{
        style: number;
        icon: number;
        deleted?: {
            style: number;
            icon: number;
        };
    }>;
    /**
     * Set the styles for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.Styles} styles styles
     * @return {Promise}
     */
    setStylesForFeatureRow(featureRow: FeatureRow, styles: Styles): Promise<{
        styleDefault: number;
        styles: number[];
        deleted: number;
    }>;
    /**
     * Set the styles for the feature table and feature id
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.Styles} styles styles
     * @return {Promise}
     */
    setStyles(featureId: number, styles: Styles): Promise<{
        styleDefault: number;
        styles: number[];
        deleted: number;
    }>;
    /**
     * Set the style of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setStyleForFeatureRow(featureRow: FeatureRow, style: StyleRow): Promise<number>;
    /**
     * Set the style of the feature row for the specified geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setStyleForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string, style: StyleRow): Promise<number>;
    /**
     * Set the default style of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setStyleDefaultForFeatureRow(featureRow: FeatureRow, style: StyleRow): Promise<number>;
    /**
     * Set the style of the feature
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setStyle(featureId: number, geometryType: string, style: StyleRow): Promise<number>;
    /**
     * Set the default style of the feature
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.StyleRow} style style row
     * @return {Promise}
     */
    setStyleDefault(featureId: number, style: StyleRow): Promise<number>;
    /**
     * Set the icons for the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.Icons} icons icons
     * @return {Promise}
     */
    setIconsForFeatureRow(featureRow: FeatureRow, icons: Icons): Promise<{
        iconDefault: number;
        icons: number[];
        deleted: number;
    }>;
    /**
     * Set the icons for the feature table and feature id
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.Icons} icons icons
     * @return {Promise}
     */
    setIcons(featureId: number, icons: Icons): Promise<{
        iconDefault: number;
        icons: number[];
        deleted: number;
    }>;
    /**
     * Set the icon of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setIconForFeatureRow(featureRow: FeatureRow, icon: IconRow): Promise<number>;
    /**
     * Set the icon of the feature row for the specified geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setIconForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string, icon: IconRow): Promise<number>;
    /**
     * Set the default icon of the feature row
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setIconDefaultForFeatureRow(featureRow: FeatureRow, icon: IconRow): Promise<number>;
    /**
     * Get the icon of the feature, searching in order: feature geometry type
     * icon, feature default icon, table geometry type icon, table default icon
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setIcon(featureId: number, geometryType: string, icon: IconRow): Promise<number>;
    /**
     * Set the default icon of the feature
     *
     * @param {Number} featureId feature id
     * @param {module:extension/style.IconRow} icon icon row
     * @return {Promise}
     */
    setIconDefault(featureId: number, icon: IconRow): Promise<number>;
    /**
     * Delete all feature styles including table styles, table icons, style, and
     * icons
     */
    deleteAllFeatureStyles(): {
        tableStyles: {
            styles: number;
            icons: number;
        };
        styles: {
            styles: number;
            icons: number;
        };
    };
    /**
     * Delete all styles including table styles and feature row styles
     */
    deleteAllStyles(): {
        tableStyles: number;
        styles: number;
    };
    /**
     * Delete all icons including table icons and feature row icons
     */
    deleteAllIcons(): {
        tableIcons: number;
        icons: number;
    };
    /**
     * Delete the feature table feature styles
     */
    deleteTableFeatureStyles(): {
        styles: number;
        icons: number;
    };
    /**
     * Delete the feature table styles
     */
    deleteTableStyles(): number;
    /**
     * Delete the feature table default style
     */
    deleteTableStyleDefault(): number;
    /**
     * Delete the feature table style for the geometry type
     *
     * @param {String} geometryType geometry type
     */
    deleteTableStyle(geometryType: string): number;
    /**
     * Delete the feature table icons
     */
    deleteTableIcons(): number;
    /**
     * Delete the feature table default icon
     */
    deleteTableIconDefault(): number;
    /**
     * Delete the feature table icon for the geometry type
     *
     * @param {String} geometryType geometry type
     */
    deleteTableIcon(geometryType: string): number;
    /**
     * Clear the cached table feature styles
     */
    clearCachedTableFeatureStyles(): void;
    /**
     * Clear the cached table styles
     */
    clearCachedTableStyles(): void;
    /**
     * Clear the cached table icons
     */
    clearCachedTableIcons(): void;
    /**
     * Delete all feature styles
     */
    deleteFeatureStyles(): {
        styles: number;
        icons: number;
    };
    /**
     * Delete all styles
     */
    deleteStyles(): number;
    /**
     * Delete feature row styles
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteStylesForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete feature row styles
     *
     * @param {Number} featureId feature id
     */
    deleteStylesForFeatureId(featureId: number): number;
    /**
     * Delete the feature row default style
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteStyleDefaultForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete the feature row default style
     *
     * @param {Number} featureId feature id
     */
    deleteStyleDefault(featureId: number): number;
    /**
     * Delete the feature row style for the feature row geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteStyleForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete the feature row style for the geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     */
    deleteStyleForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string): number;
    /**
     * Delete the feature row style for the geometry type
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     */
    deleteStyle(featureId: number, geometryType: string): number;
    /**
     * Delete the style and associated mappings using StyleRow
     *
     * @param {module:extension/style.StyleRow} styleRow style row
     */
    deleteStyleAndMappingsByStyleRow(styleRow: StyleRow): number;
    /**
     * Delete the style and associated mappings using StyleRow's Id
     *
     * @param {Number} styleRowId style row id
     */
    deleteStyleAndMappingsByStyleRowId(styleRowId: number): number;
    /**
     * Delete all icons
     */
    deleteIcons(): number;
    /**
     * Delete feature row icons
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteIconsForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete feature row icons
     *
     * @param {Number} featureId feature id
     */
    deleteIconsForFeatureId(featureId: number): number;
    /**
     * Delete the feature row default icon
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteIconDefaultForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete the feature row default icon
     *
     * @param {Number} featureId feature id
     */
    deleteIconDefault(featureId: number): number;
    /**
     * Delete the feature row icon for the feature row geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     */
    deleteIconForFeatureRow(featureRow: FeatureRow): number;
    /**
     * Delete the feature row icon for the geometry type
     *
     * @param {module:features/user/featureRow} featureRow feature row
     * @param {String} geometryType geometry type
     */
    deleteIconForFeatureRowAndGeometryType(featureRow: FeatureRow, geometryType: string): number;
    /**
     * Delete the feature row icon for the geometry type
     *
     * @param {Number} featureId feature id
     * @param {String} geometryType geometry type
     */
    deleteIcon(featureId: number, geometryType: string): number;
    /**
     * Delete the icon and associated mappings using IconRow
     *
     * @param {module:extension/style.IconRow} iconRow icon row
     */
    deleteIconAndMappingsByIconRow(iconRow: IconRow): number;
    /**
     * Delete the icon and associated mappings using IconRow
     *
     * @param {Number} iconRowId icon row id
     */
    deleteIconAndMappingsByIconRowId(iconRowId: number): number;
    /**
     * Get all the unique style row ids the table maps to
     *
     * @return {module:extension/style.StyleRow} style row ids
     */
    getAllTableStyleIds(): number[];
    /**
     * Get all the unique icon row ids the table maps to
     *
     * @return {module:extension/style.IconRow} icon row ids
     */
    getAllTableIconIds(): number[];
    /**
     * Get all the unique style row ids the features map to
     *
     * @return {module:extension/style.StyleRow} style row ids
     */
    getAllStyleIds(): number[];
    /**
     * Get all the unique icon row ids the features map to
     *
     * @return {module:extension/style.IconRow} icon row ids
     */
    getAllIconIds(): number[];
}
