import proj4 from 'proj4';
import { Geometry } from 'geojson';
import { FeatureDao } from '../../features/user/featureDao';
import { BoundingBox } from '../../boundingBox';
import { IconCache } from '../../extension/style/iconCache';
import { GeometryCache } from './geometryCache';
import { FeatureDrawType } from './featureDrawType';
import { FeaturePaintCache } from './featurePaintCache';
import { Paint } from './paint';
import { FeatureTableStyles } from '../../extension/style/featureTableStyles';
import { GeoPackage } from '../../geoPackage';
import { FeatureRow } from '../../features/user/featureRow';
import { StyleRow } from '../../extension/style/styleRow';
import { FeatureTilePointIcon } from './featureTilePointIcon';
import { CustomFeaturesTile } from './custom/customFeaturesTile';
import { FeatureStyle } from '../../extension/style/featureStyle';
/**
 * FeatureTiles module.
 * @module tiles/features
 */
/**
 *  Tiles drawn from or linked to features. Used to query features and optionally draw tiles
 *  from those features.
 */
export declare class FeatureTiles {
    featureDao: FeatureDao<FeatureRow>;
    tileWidth: number;
    tileHeight: number;
    private static readonly isElectron;
    private static readonly isNode;
    private static readonly useNodeCanvas;
    projection: proj4.Converter;
    simplifyGeometries: boolean;
    compressFormat: string;
    pointRadius: number;
    pointPaint: Paint;
    pointIcon: FeatureTilePointIcon;
    linePaint: Paint;
    lineStrokeWidth: number;
    polygonPaint: Paint;
    polygonStrokeWidth: number;
    fillPolygon: boolean;
    polygonFillPaint: Paint;
    featurePaintCache: FeaturePaintCache;
    geometryCache: GeometryCache;
    cacheGeometries: boolean;
    iconCache: IconCache;
    scale: number;
    geoPackage: GeoPackage;
    featureTableStyles: FeatureTableStyles;
    maxFeaturesPerTile: number;
    maxFeaturesTileDraw: CustomFeaturesTile;
    widthOverlap: number;
    heightOverlap: number;
    constructor(featureDao: FeatureDao<FeatureRow>, tileWidth?: number, tileHeight?: number);
    /**
     * Manually set the width and height draw overlap
     * @param {Number} pixels pixels
     */
    setDrawOverlap(pixels: number): void;
    /**
     * Get the width draw overlap
     * @return {Number} width draw overlap
     */
    getWidthDrawOverlap(): number;
    /**
     * Manually set the width draw overlap
     * @param {Number} pixels pixels
     */
    setWidthDrawOverlap(pixels: number): void;
    /**
     * Get the height draw overlap
     * @return {Number} height draw overlap
     */
    getHeightDrawOverlap(): number;
    /**
     * Manually set the height draw overlap
     * @param {Number} pixels pixels
     */
    setHeightDrawOverlap(pixels: number): void;
    /**
     * Get the feature DAO
     * @return {module:features/user/featureDao} feature dao
     */
    getFeatureDao(): FeatureDao<FeatureRow>;
    /**
     * Get the feature table styles
     * @return {module:extension/style~FeatureTableStyles} feature table styles
     */
    getFeatureTableStyles(): FeatureTableStyles;
    /**
     * Set the feature table styles
     * @param {module:extension/style~FeatureTableStyles} featureTableStyles feature table styles
     */
    setFeatureTableStyles(featureTableStyles: FeatureTableStyles): void;
    /**
     * Ignore the feature table styles within the GeoPackage
     */
    ignoreFeatureTableStyles(): void;
    /**
     * Clear all caches
     */
    clearCache(): void;
    /**
     * Clear the style paint cache
     */
    clearStylePaintCache(): void;
    /**
     * Set / resize the style paint cache size
     *
     * @param {Number} size
     * @since 3.3.0
     */
    setStylePaintCacheSize(size: number): void;
    /**
     * Clear the icon cache
     */
    clearIconCache(): void;
    /**
     * Set / resize the icon cache size
     * @param {Number} size new size
     */
    setIconCacheSize(size: number): void;
    /**
     * Get the tile width
     * @return {Number} tile width
     */
    getTileWidth(): number;
    /**
     * Set the tile width
     * @param {Number} tileWidth tile width
     */
    setTileWidth(tileWidth: number): void;
    /**
     * Get the tile height
     * @return {Number} tile height
     */
    getTileHeight(): number;
    /**
     * Set the tile height
     * @param {Number} tileHeight tile height
     */
    setTileHeight(tileHeight: number): void;
    /**
     * Get the compress format
     * @return {String} compress format
     */
    getCompressFormat(): string;
    /**
     * Set the compress format
     * @param {String} compressFormat compress format
     */
    setCompressFormat(compressFormat: string): void;
    /**
     * Set the scale
     *
     * @param {Number} scale scale factor
     */
    setScale(scale: number): void;
    /**
     * Set CacheGeometries flag. When set to true, geometries will be cached.
     * @param {Boolean} cacheGeometries
     */
    setCacheGeometries(cacheGeometries: boolean): void;
    /**
     * Set geometry cache's max size
     * @param {Number} maxSize
     */
    setGeometryCacheMaxSize(maxSize: number): void;
    /**
     * Set SimplifyGeometries flag. When set to true, geometries will be simplified when possible.
     * @param {Boolean} simplifyGeometries
     */
    setSimplifyGeometries(simplifyGeometries: boolean): void;
    /**
     * Get the scale
     * @return {Number} scale factor
     */
    getScale(): number;
    calculateDrawOverlap(): void;
    setDrawOverlapsWithPixels(pixels: number): void;
    getFeatureStyle(featureRow: FeatureRow): FeatureStyle;
    /**
     * Get the point paint for the feature style, or return the default paint
     * @param featureStyle feature style
     * @return paint
     */
    getPointPaint(featureStyle: FeatureStyle): Paint;
    /**
     * Get the line paint for the feature style, or return the default paint
     * @param featureStyle feature style
     * @return paint
     */
    getLinePaint(featureStyle: FeatureStyle): Paint;
    /**
     * Get the polygon paint for the feature style, or return the default paint
     * @param featureStyle feature style
     * @return paint
     */
    getPolygonPaint(featureStyle: FeatureStyle): Paint;
    /**
     * Get the polygon fill paint for the feature style, or return the default
     * paint
     * @param featureStyle feature style
     * @return paint
     */
    getPolygonFillPaint(featureStyle: FeatureStyle): Paint;
    /**
     * Get the feature style paint from cache, or create and cache it
     * @param featureStyle feature style
     * @param drawType draw type
     * @return feature style paint
     */
    getFeatureStylePaint(featureStyle: FeatureStyle, drawType: FeatureDrawType): Paint;
    /**
     * Get the style paint from cache, or create and cache it
     * @param style style row
     * @param drawType draw type
     * @return {Paint} paint
     */
    getStylePaint(style: StyleRow, drawType: FeatureDrawType): Paint;
    /**
     * Get the point radius
     * @return {Number} radius
     */
    getPointRadius(): number;
    /**
     * Set the point radius
     * @param {Number} pointRadius point radius
     */
    setPointRadius(pointRadius: number): void;
    /**
     * Get point color
     * @return {String} color
     */
    getPointColor(): string;
    /**
     * Set point color
     * @param {String} pointColor point color
     */
    setPointColor(pointColor: string): void;
    /**
     * Get the point icon
     * @return {module:tiles/features.FeatureTilePointIcon} icon
     */
    getPointIcon(): FeatureTilePointIcon;
    /**
     * Set the point icon
     * @param {module:tiles/features.FeatureTilePointIcon} pointIcon point icon
     */
    setPointIcon(pointIcon: FeatureTilePointIcon): void;
    /**
     * Get line stroke width
     * @return {Number} width
     */
    getLineStrokeWidth(): number;
    /**
     * Set line stroke width
     * @param {Number} lineStrokeWidth line stroke width
     */
    setLineStrokeWidth(lineStrokeWidth: number): void;
    /**
     * Get line color
     * @return {String} color
     */
    getLineColor(): string;
    /**
     * Set line color
     * @param {String} lineColor line color
     */
    setLineColor(lineColor: string): void;
    /**
     * Get polygon stroke width
     * @return {Number} width
     */
    getPolygonStrokeWidth(): number;
    /**
     * Set polygon stroke width
     * @param {Number} polygonStrokeWidth polygon stroke width
     */
    setPolygonStrokeWidth(polygonStrokeWidth: number): void;
    /**
     * Get polygon color
     * @return {String} color
     */
    getPolygonColor(): string;
    /**
     * Set polygon color
     * @param {String} polygonColor polygon color
     */
    setPolygonColor(polygonColor: string): void;
    /**
     * Is fill polygon
     * @return {Boolean} true if fill polygon
     */
    isFillPolygon(): boolean;
    /**
     * Set the fill polygon
     * @param {Boolean} fillPolygon fill polygon
     */
    setFillPolygon(fillPolygon: boolean): void;
    /**
     * Get polygon fill color
     * @return {String} color
     */
    getPolygonFillColor(): string;
    /**
     * Set polygon fill color
     * @param {String} polygonFillColor polygon fill color
     */
    setPolygonFillColor(polygonFillColor: string): void;
    /**
     * Get the max features per tile
     * @return {Number} max features per tile or null
     */
    getMaxFeaturesPerTile(): number;
    /**
     * Set the max features per tile. When more features are returned in a query
     * to create a single tile, the tile is not created.
     * @param {Number} maxFeaturesPerTile  max features per tile
     */
    setMaxFeaturesPerTile(maxFeaturesPerTile: number): void;
    /**
     * Get the max features tile draw, the custom tile drawing implementation
     * for tiles with more features than the max at #getMaxFeaturesPerTile
     * @return {module:tiles/features/custom~CustomFeatureTile} max features tile draw or null
     */
    getMaxFeaturesTileDraw(): CustomFeaturesTile;
    /**
     * Set the max features tile draw, used to draw tiles when more features for
     * a single tile than the max at #getMaxFeaturesPerTile exist
     * @param {module:tiles/features/custom~CustomFeatureTile} maxFeaturesTileDraw max features tile draw
     */
    setMaxFeaturesTileDraw(maxFeaturesTileDraw: CustomFeaturesTile): void;
    getFeatureCountXYZ(x: number, y: number, z: number): number;
    drawTile(x: number, y: number, z: number, canvas?: any): Promise<any>;
    drawTileQueryAll(x: number, y: number, zoom: number, canvas?: any): Promise<any>;
    webMercatorTransform(geoJson: any): any;
    drawTileQueryIndex(x: number, y: number, z: number, tileCanvas?: any): Promise<any>;
    drawTileWithBoundingBox(boundingBox: BoundingBox, zoom: number, tileCanvas?: any): Promise<any>;
    /**
     * Draw a point in the context
     * @param geoJson
     * @param context
     * @param boundingBox
     * @param featureStyle
     * @param transform
     */
    drawPoint(geoJson: any, context: any, boundingBox: BoundingBox, featureStyle: FeatureStyle, transform: Function): Promise<void>;
    /**
     * When the simplify tolerance is set, simplify the points to a similar
     * curve with fewer points.
     * @param simplifyTolerance simplify tolerance in meters
     * @param lineString GeoJSON
     * @return simplified GeoJSON
     * @since 2.0.0
     */
    simplifyPoints(simplifyTolerance: number, lineString: any): any;
    /**
     * Get the path of the line string
     * @param simplifyTolerance simplify tolerance in meters
     * @param transform
     * @param lineString
     * @param context
     * @param boundingBox
     */
    getPath(simplifyTolerance: number, lineString: any, transform: Function, context: any, boundingBox: BoundingBox): void;
    /**
     * Draw a line in the context
     * @param simplifyTolerance
     * @param geoJson
     * @param context
     * @param featureStyle
     * @param transform
     * @param boundingBox
     */
    drawLine(simplifyTolerance: number, geoJson: any, context: any, featureStyle: FeatureStyle, transform: Function, boundingBox: BoundingBox): void;
    /**
     * Draw a polygon in the context
     * @param simplifyTolerance
     * @param geoJson
     * @param context
     * @param featureStyle
     * @param transform
     * @param boundingBox
     */
    drawPolygon(simplifyTolerance: any, geoJson: any, context: any, featureStyle: FeatureStyle, transform: Function, boundingBox: BoundingBox): void;
    /**
     * Add a feature to the batch
     * @param simplifyTolerance
     * @param geoJson
     * @param context
     * @param transform
     * @param boundingBox
     * @param featureStyle
     */
    drawGeometry(simplifyTolerance: number, geoJson: Geometry, context: any, transform: Function, boundingBox: BoundingBox, featureStyle: FeatureStyle): Promise<void>;
    /**
     * Create an expanded bounding box to handle features outside the tile that overlap
     * @param webMercatorBoundingBox  web mercator bounding box
     * @return {BoundingBox} bounding box
     */
    expandBoundingBox(webMercatorBoundingBox: BoundingBox): BoundingBox;
    /**
     * Create an expanded bounding box to handle features outside the tile that overlap
     * @param webMercatorBoundingBox web mercator bounding box
     * @param tileWebMercatorBoundingBox  tile web mercator bounding box
     * @return {BoundingBox} bounding box
     */
    expandWebMercatorBoundingBox(webMercatorBoundingBox: BoundingBox, tileWebMercatorBoundingBox: BoundingBox): BoundingBox;
}
